% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/semivariance.R
\name{semivariance}
\alias{semivariance}
\alias{semivariance.krige}
\alias{semivariance.lm}
\alias{semivariance.default}
\title{Semivariance for Geospatial Data}
\usage{
semivariance(object, ...)

\method{semivariance}{krige}(object, bins = 13, terms = "all", plot = FALSE, ...)

\method{semivariance}{lm}(
  object,
  bins = 13,
  coords,
  terms = c("raw", "residual"),
  east,
  north,
  plot = FALSE,
  ...
)

\method{semivariance}{default}(object, bins = 13, coords, data, east, north, plot = FALSE, ...)
}
\arguments{
\item{object}{An object for which the semivariance is desired. The object can 
be a \code{krige} object, a \code{lm} object, or a vector of variables (or 
variable names in the \code{data}).}

\item{\dots}{Additional arguments passed to \code{semivariance} methods.}

\item{bins}{Number of bins into which distances should be divided. The observed 
distances will be split at equal intervals, and the semivariance will be computed 
within each interval. Defaults to 13 intervals.}

\item{terms}{A vector of strings specifies for which the semivariogram is created. 
Options are "raw" (the semivariogram for raw data), "residual" (the semivariogram
for residuals from linear regression).}

\item{plot}{Logical values indicates whether a graph of the empirical semivariogram 
should be presented with a run of the function. Default omits the plot and only 
returns semivariance values. See \code{\link{semivariogram}} for additional 
plotting functions.}

\item{coords}{A matrix of coordinates for all observations or a vector of variable 
names indicating the coordinates variables in the data. Alternatively, the 
coordinates can also be specified separately using \code{east} and \code{north}.}

\item{east}{Alternative specification for the vector of eastings for all observations.}

\item{north}{Alternative specification for the vector of northing for all observations.}

\item{data}{If object is a variable name, a data frame must be provided.}
}
\value{
A semivariance object. It will be a numeric vector with each bin's value 
  of the semivariance if only one kind of semivariance is computed; a list including 
  different kinds of semivariance if both raw and residual semivariance is computed.
}
\description{
This function computes the empirical semivariance for a spatially-distributed 
variable. Based on the user's chosen level of coarsening, the semivariance is 
presented for various distances.
}
\details{
Semivariance is equal to half of the variance of the difference in a 
  variable's values at a given distance. That is, the semivariance is defined 
  as: \eqn{\gamma(h)=0.5*E[X(s+h)-X(s)]^2}, where \eqn{X} is the variable of 
  interest, s is a location, and h is the distance from s to another location.
  
  The function can be applied to a variable, a fitted linear model (\code{lm} 
  object) before fitting a spatial model or to a \code{krige} object or \code{semivariance}
  object to assess the model fit. When applying to a variable, it will describes 
  the raw data; for a \code{lm} object, the default will present empirical 
  semivariogram for both the raw data and linear residuals. Users can also specify 
  which semivariance is needed in the \code{terms} argument if there are multiple 
  kinds of semivariogram can be plotted. A \code{semivariance} object can also 
  be used to create semivariogram afterwards using generic \code{plot} function 
  with more options.
}
\examples{
\dontrun{
# Summarize example data
summary(ContrivedData)

# Empirical semivariance for variable y
semivariance(ContrivedData$y,coords = cbind(ContrivedData$s.1, ContrivedData$s.2))

# Initial OLS Model
contrived.ols<-lm(y~x.1+x.2,data=ContrivedData); summary(contrived.ols)

# Empirical semivariance for ols fit
(sv.ols <- semivariance(contrived.ols, coords = c("s.1","s.2"), bins=13))
plot(sv.ols)

# Estimation using metropolis.krige()
# Set seed
set.seed(1241060320)

M <- 100

contrived.run <- metropolis.krige(y ~ x.1 + x.2, coords = c("s.1","s.2"), 
  data = ContrivedData, n.iter = M, range.tol = 0.05)
  
# Parametric semivariance
(sv.krige <- semivariance(contrived.run, plot = TRUE))

# Convert to other format for further use
as.matrix(sv.krige)
as.data.frame(sv.krige)
}

}
\references{
Sudipto Banerjee, Bradley P. Carlin, and Alan E. Gelfand. 2015. \emph{Hierarchical 
  Modeling and Analysis for Spatial Data}. 2nd ed. Boca Raton, FL: CRC Press.
}
\seealso{
\code{\link{semivariogram}}, \code{\link{plot.semivariance}}, \code{\link{exponential.semivariance}}
}
