% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kde1d.R
\name{kde1d}
\alias{kde1d}
\title{Univariate local-polynomial likelihood kernel density estimation}
\usage{
kde1d(
  x,
  xmin = NaN,
  xmax = NaN,
  type = "continuous",
  mult = 1,
  bw = NA,
  deg = 2,
  weights = numeric(0)
)
}
\arguments{
\item{x}{vector (or one-column matrix/data frame) of observations; can be
\code{numeric} or \code{ordered}.}

\item{xmin}{lower bound for the support of the density (only for continuous
data); \code{NaN} means no boundary.}

\item{xmax}{upper bound for the support of the density (only for continuous
data); \code{NaN} means no boundary.}

\item{type}{variable type; must be one of \verb{\{c, cont, continuous\}} for
continuous variables, one of \verb{\{d, disc, discrete\}} for discrete integer
variables, or one of \verb{\{zi, zinfl, zero-inflated\}} for zero-inflated
variables.}

\item{mult}{positive bandwidth multiplier; the actual bandwidth used is
\eqn{bw*mult}.}

\item{bw}{bandwidth parameter; has to be a positive number or \code{NA}; the
latter uses the plug-in methodology of Sheather and Jones (1991) with
appropriate modifications for \code{deg > 0}.}

\item{deg}{degree of the polynomial; either \code{0}, \code{1}, or \code{2} for
log-constant, log-linear, and log-quadratic fitting, respectively.}

\item{weights}{optional vector of weights for individual observations.}
}
\value{
An object of class \code{kde1d}.
}
\description{
The estimators can handle data with bounded, unbounded, and discrete support,
see \emph{Details}.
}
\details{
A Gaussian kernel is used in all cases. If \code{xmin} or \code{xmax} are
finite, the density estimate will be 0 outside of \eqn{[xmin, xmax]}. A
log-transform is used if there is only one boundary (see, Geenens and Wang,
2018); a probit transform is used if there are two (see, Geenens, 2014).

Discrete variables are handled via jittering (see, Nagler, 2018a, 2018b).
A specific form of deterministic jittering is used, see \code{\link[=equi_jitter]{equi_jitter()}}.

Zero-inflated densities are estimated by a hurdle-model with discrete
mass at 0 and the remainder estimated as for \code{type = "continuous"}.
}
\examples{

## unbounded data
x <- rnorm(500) # simulate data
fit <- kde1d(x) # estimate density
dkde1d(0, fit) # evaluate density estimate
summary(fit) # information about the estimate
plot(fit) # plot the density estimate
curve(dnorm(x),
  add = TRUE, # add true density
  col = "red"
)

## bounded data, log-linear
x <- rgamma(500, shape = 1) # simulate data
fit <- kde1d(x, xmin = 0, deg = 1) # estimate density
dkde1d(seq(0, 5, by = 1), fit) # evaluate density estimate
summary(fit) # information about the estimate
plot(fit) # plot the density estimate
curve(dgamma(x, shape = 1), # add true density
  add = TRUE, col = "red",
  from = 1e-3
)

## discrete data
x <- rbinom(500, size = 5, prob = 0.5) # simulate data
fit <- kde1d(x, xmin = 0, xmax = 5, type = "discrete") # estimate density
fit <- kde1d(ordered(x, levels = 0:5)) # alternative API
dkde1d(sort(unique(x)), fit) # evaluate density estimate
summary(fit) # information about the estimate
plot(fit) # plot the density estimate
points(ordered(0:5, 0:5), # add true density
  dbinom(0:5, 5, 0.5),
  col = "red"
)

## zero-inflated data
x <- rexp(500, 0.5)  # simulate data
x[sample(1:500, 200)] <- 0 # add zero-inflation
fit <- kde1d(x, xmin = 0, type = "zi") # estimate density
dkde1d(sort(unique(x)), fit) # evaluate density estimate
summary(fit) # information about the estimate
plot(fit) # plot the density estimate
lines(  # add true density
  seq(0, 20, l = 100),
  0.6 * dexp(seq(0, 20, l = 100), 0.5),
  col = "red"
)
points(0, 0.4, col = "red")

## weighted estimate
x <- rnorm(100) # simulate data
weights <- rexp(100) # weights as in Bayesian bootstrap
fit <- kde1d(x, weights = weights) # weighted fit
plot(fit) # compare with unweighted fit
lines(kde1d(x), col = 2)
}
\references{
Geenens, G. (2014). \emph{Probit transformation for kernel density
estimation on the unit interval}. Journal of the American Statistical
Association, 109:505, 346-358,
\href{https://arxiv.org/abs/1303.4121}{arXiv:1303.4121}

Geenens, G., Wang, C. (2018). \emph{Local-likelihood transformation kernel density
estimation for positive random variables.} Journal of Computational and
Graphical Statistics, to appear,
\href{https://arxiv.org/abs/1602.04862}{arXiv:1602.04862}

Nagler, T. (2018a). \emph{A generic approach to nonparametric function estimation
with mixed data.} Statistics & Probability Letters, 137:326–330,
\href{https://arxiv.org/abs/1704.07457}{arXiv:1704.07457}

Nagler, T. (2018b). \emph{Asymptotic analysis of the jittering kernel density
estimator.} Mathematical Methods of Statistics, in press,
\href{https://arxiv.org/abs/1705.05431}{arXiv:1705.05431}

Sheather, S. J. and Jones, M. C. (1991). A reliable data-based bandwidth
selection method for kernel density estimation. Journal of the Royal
Statistical Society, Series B, 53, 683–690.
}
\seealso{
\code{\link[=dkde1d]{dkde1d()}}, \code{\link[=pkde1d]{pkde1d()}}, \code{\link[=qkde1d]{qkde1d()}}, \code{\link[=rkde1d]{rkde1d()}},
\code{\link[=plot.kde1d]{plot.kde1d()}}, \code{\link[=lines.kde1d]{lines.kde1d()}}
}
