% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/jamba-drawlabels.R
\name{coordPresets}
\alias{coordPresets}
\title{Process coordinate adjustment presets}
\usage{
coordPresets(
  preset = "default",
  x = 0,
  y = 0,
  adjPreset = "default",
  adjX = 0.5,
  adjY = 0.5,
  adjOffsetX = 0,
  adjOffsetY = 0,
  preset_type = c("plot"),
  verbose = FALSE,
  ...
)
}
\arguments{
\item{preset}{\code{character} vector of coordinate positions, or the default
"default" to use the \verb{x,y} coordinates.
\itemize{
\item Recognized terms: center, bottom, top, left, right, topleft,
topright, bottomleft, bottomright.
}}

\item{x, y}{\code{numeric} vectors indicating the default coordinates \verb{x,y}.}

\item{adjPreset}{\code{character} vector of text label positions, or
the default "default" to use \code{preset}, or when \code{preset="default"} the
\verb{adjX,adjY} values are used.
\itemize{
\item Recognized terms: center, bottom, top, left, right, topleft,
topright, bottomleft, bottomright.
}}

\item{adjX, adjY}{numeric vectors indicating default text adjustment
values, as described for \code{adj} in \code{graphics::text()}.}

\item{adjOffsetX, adjOffsetY}{\code{numeric} vector used to apply an offset
value to the \verb{adjX,adjY} values, where positive values would
place a label farther away from center. Note these units are
relative to the text label size, when used with \code{graphics::text()},
larger labels will be adjusted more than smaller labels.}

\item{preset_type}{\code{character} string indicating the reference point
for the preset boundaries:
\itemize{
\item \code{"plot"} uses the plot border.
\item \code{"margin"} uses the margin border. Note that the margin used
is the inner margin around the plot figure, not the outer margin
which may be applied around multi-panel plot figures.
}}

\item{verbose}{\code{logical} indicating whether to print verbose output.}

\item{...}{additional arguments are ignored.}
}
\value{
\code{data.frame} after adjustment, where the number of rows
is determined by the longest input argument, with colnames:
\itemize{
\item x
\item y
\item adjX
\item adjY
\item preset
\item adjPreset
}
}
\description{
Process coordinate adjustment presets
}
\details{
This function is intended to be a convenient way to define
coordinates using preset terms like "topleft", "bottom", "center".

Similarly, it is intended to help define corresponding text
adjustments, using \code{adj} compatible with \code{graphics::text()},
using preset terms like "bottomright", "center".

When \code{preset} is \code{"default"}, the original \verb{x,y} coordinates
are used. Otherwise the \verb{x,y} coordinates are defined using the
plot region coordinates, where \code{"left"} uses \code{graphics::par("usr")[1]},
and \code{"top"} uses \code{graphics::par("usr")[4]}.

When \code{adjPreset} is \code{"default"} it will use the \code{preset} to
define a reciprocal text placement. For example when \code{preset="topright"}
the text placement will be equivalent to \code{adjPreset="bottomleft"}.
The \code{adjPreset} terms \code{"top"}, \code{"bottom"}, \code{"right"}, \code{"left"},
and \code{"center"} refer to the text label placement relative to
\verb{x,y} coordinate.

If both \code{preset="default"} and \code{adjPreset="default"} the original
\verb{adjX,adjY} values are returned.

The function is vectorized, and uses the longest input argument,
so one can supply a vector of \code{preset} and it will return coordinates
and adjustments of length equal to the input \code{preset} vector.
The \code{preset} value takes priority over the supplied \verb{x,y} coordinates.
}
\examples{
# determine coordinates
presetV <- c("top",
   "bottom",
   "left",
   "right",
   "topleft");
cp1 <- coordPresets(preset=presetV);
cp1;

# make sure to prepare the plot region first
jamba::nullPlot(plotAreaTitle="");
graphics::points(cp1$x, cp1$y, pch=20, cex=2, col="red");

# unfortunately graphics::text() does not have vectorized adj
# so it must iterate each row
graphics::title(main="graphics::text() is not vectorized, text is adjacent to edges")
for (i in seq_along(presetV)) {
   graphics::text(cp1$x[i], cp1$y[i],
      labels=presetV[i],
      adj=c(cp1$adjX[i], cp1$adjY[i]));
}

# drawLabels() will be vectorized for unique adj subsets
# and adds a small buffer around text
jamba::nullPlot(plotAreaTitle="");
graphics::title(main="drawLabels() is vectorized, includes small buffer")
drawLabels(txt=presetV,
   preset=presetV)

jamba::nullPlot(plotAreaTitle="");
graphics::title(main="drawLabels() can place labels outside plot edges")
drawLabels(txt=presetV,
   preset=presetV,
   adjPreset=presetV)

# drawLabels() is vectorized for example
jamba::nullPlot(plotAreaTitle="");
graphics::title(main="Use adjPreset to position labels at a center point")
presetV2 <- c("topleft",
   "topright",
   "bottomleft",
   "bottomright");
cp2 <- coordPresets(preset="center",
   adjPreset=presetV2,
   adjOffsetX=0.1,
   adjOffsetY=0.4);
graphics::points(cp2$x,
   cp2$y,
   pch=20,
   cex=2,
   col="red");
drawLabels(x=cp2$x,
   y=cp2$y,
   adjX=cp2$adjX,
   adjY=cp2$adjY,
   txt=presetV2,
   boxCexAdjust=c(1.15,1.6),
   labelCex=1.3,
   lx=rep(1.5, 4),
   ly=rep(1.5, 4))

# demonstrate margin coordinates
withr::with_par(list("oma"=c(1, 1, 1, 1)), {
nullPlot(xlim=c(0, 1), ylim=c(1, 5));
cpxy <- coordPresets(rep(c("top", "bottom", "left", "right"), each=2),
   preset_type=rep(c("plot", "figure"), 4));
drawLabels(preset=c("top", "top"),
   txt=c("top label relative to figure",
      "top label relative to plot"),
   preset_type=c("figure", "plot"))
graphics::points(cpxy$x, cpxy$y, cex=2,
   col="red4", bg="red1", xpd=NA,
   pch=rep(c(21, 23), 4))
})

}
\seealso{
Other jam plot functions: 
\code{\link{adjustAxisLabelMargins}()},
\code{\link{decideMfrow}()},
\code{\link{drawLabels}()},
\code{\link{getPlotAspect}()},
\code{\link{groupedAxis}()},
\code{\link{imageByColors}()},
\code{\link{imageDefault}()},
\code{\link{minorLogTicksAxis}()},
\code{\link{nullPlot}()},
\code{\link{plotPolygonDensity}()},
\code{\link{plotRidges}()},
\code{\link{plotSmoothScatter}()},
\code{\link{shadowText}()},
\code{\link{shadowText_options}()},
\code{\link{showColors}()},
\code{\link{sqrtAxis}()},
\code{\link{usrBox}()}
}
\concept{jam plot functions}
