#' @title Simulate example JAGS data.
#' @export
#' @description An example dataset compatible with the model file
#'   from [tar_jags_example_file()]. The output has a `.join_data`
#'   element so the true value of `beta` from the simulation
#'   is automatically appended to the `beta` rows of the
#'   summary output.
#' @details The `tar_jags_example_data()` function draws a JAGS
#'   dataset from the prior predictive distribution of the
#'   model from [tar_jags_example_file()]. First, the
#'   regression coefficient `beta` is drawn from its standard
#'   normal prior, and the covariate `x` is computed.
#'   Then, conditional on the `beta` draws and the covariate,
#'   the response vector `y` is drawn from its
#'   Normal(`x * beta`, 1) likelihood.
#' @return List, dataset compatible with the model file from
#'   [tar_jags_example_file()]. The output has a `.join_data`
#'   element so the true value of `beta` from the simulation
#'   is automatically appended to the `beta` rows of the
#'   summary output.
#' @format A list with the following elements:
#'   * `n`: integer, number of data points.
#'   * `x`: numeric, covariate vector.
#'   * `y`: numeric, response variable.
#'   * `true_beta`: numeric of length 1, value of the regression
#'     coefficient `beta` used in simulation.
#'   * `.join_data`: a list of simulated values to be appended
#'     to as a `.join_data` column in the output of
#'     targets generated by functions such as
#'     [tar_jags_rep_summary()]. Contains the
#'     regression coefficient `beta` (numeric of length 1)
#'     and prior predictive data `y` (numeric vector).
#' @param n Integer of length 1, number of data points.
#' @param n Integer of length 1, number of data points.
#' @examples
#' tar_jags_example_data()
tar_jags_example_data <- function(n = 10L) {
  beta <- stats::rnorm(n = 1, mean = 0, sd = 1)
  x <- seq(from = -1, to = 1, length.out = n)
  y <- stats::rnorm(n, x * beta, 1)
  out <- list(
    n = n,
    x = x,
    y = y,
    .join_data = list(beta = beta)
  )
  out
}
