\name{ArgMinDist}
\alias{ArgMinDist}
\title{A Minimum Distance Estimation}
\description{The function calculates
  a minimum distance estimator for an imprecise probability model.
  The imprecise probability model consists of upper coherent previsions whose credal sets are given
  by finite numbers of constraints on the expectations. The parameter set is finite. The estimator chooses
  that parameter such that the empirical measure lies next to the corresponding credal set with respect to
  the total variation norm.
}
\usage{
ArgMinDist(x, lbomega, ubomega, epsilon, ImpreciseModel)
}
\arguments{
  \item{x}{a matrix where each row corresponds to one observation}
  \item{lbomega}{a vector containing the lower bounds of the sample space}
  \item{ubomega}{a vector containing the upper bounds of the sample space}
  \item{epsilon}{a positive real number; step size of the discretization}
  \item{ImpreciseModel}{a list of upper coherent previsions; see 'Details'}
}
\details{The matrix \code{x} containes independent identically distributed data.
         Each row corresponds to one observation. The sample space is assumed to be a hyperrectangle in
         \eqn{$R^k$}{R^k}. The lower bounds of this hyperrectangle are given by \code{lbomega};
         the upper bounds of this hyperrectangle are given by \code{ubomega}.
         Accordingly, \code{length(lbomega)}, \code{length(ubomega)} and
         \code{length(x[,1])} are equal to k.
         
         Smaller values of \code{epsilon} may lead to more accurate results but increase the calculation time. 
         Too small values of \code{epsilon} may cause an error due to RAM limitations.
         
         \code{ImpreciseModel} containes an imprecise model consisting of upper coherent previsions.
         \code{ImpreciseModel} is a list; each component of \code{ImpreciseModel} is again a list
         which corresponds to an upper coherent prevision. Each upper coherent prevision is given by
         a list containing a list of functions and a corresponding vector of upper previsions.

         For example, the imprecise model \code{ImpreciseModel} may consist of three coherent upper previsions
         \code{ImpreciseModel <- list(CohUpPrev1,CohUpPrev2,CohUpPrev3)}. \code{CohUpPrev1} may be defined by
         \code{CohUpPrev1 <- list(ListOfFunctions,UpperPrevisions)}. Here, \code{ListOfFunctions} is a list
         of functions, e.g., \code{ListOfFunctions <- list(f1,f2,f3,f4)}. Every function has to accept a single numeric
         argument and to return a numeric vector of the same length;
         the infimum of every function has to be 0, the supremum has to be 1. \code{UpperPrevisions} is
         a vector which contains the values of the upper coherent prevision at the functions in \code{ListOfFunctions}.
         That is, e.g., \code{UpperPrevisions[2]} is the value of the upper prevision at the function \code{f2}.
         Accordingly, the number of elements of the list \code{ListOfFunctions} is
         equal to \code{length(UpperPrevisions)}.
         
         The estimation is that coherent upper prevision whose credal set has minimal total variation distance
         to the empirical measure generated by the observations \code{x}.
         Confer Hable (2008) for the definition of this minimum distance estimator; confer Walley (1991)
         and Hable (2008) for the theory of imprecise probabilities based on coherent upper previsions
         or coherent lower previsions.
}
\value{
  \code{ArgMinDist} returns a list, e.g. \code{results}, containing three components
  \item{results[[1]]}{the estimation; that is, the number of the minimizing coherent upper prevision in
                      \code{ImpreciseModel}}
  \item{results[[2]]}{the total variation distance of the minimizing coherent upper prevision}
  \item{results[[3]]}{the number of linear programms which had to be solved}
}
\references{Hable, R. (2008) \emph{Data-Based Decisions under Complex Uncertainty}. Ph.D. thesis,
            LMU Munich, \emph{in preparation}.

            Walley, P. (1991) \emph{Statistical reasoning with imprecise probabilities}. Chapman & Hall, London.
}
\author{Robert Hable}
\note{R programming support was given by Matthias Kohl}
%\seealso{ ~~objects to See Also as \code{\link{help}}, ~~~ }
\examples{
f1 <- function(v){ ifelse( abs(v-1)<1e-10 ,1,0) }
f2 <- function(v){ ifelse( abs(v-2)<1e-10 ,1,0) }
f3 <- function(v){ ifelse( abs(v-3)<1e-10 ,1,0) }
f4 <- function(v){ 1-ifelse( abs(v-3)<1e-10 ,1,0) }

x <- matrix(c(1,2,3,4),nrow=1)

UpperPrevisions1 <- c(1/4-0.03,1/4-0.03,1/4+0.01,1)
ListOfFunctions1 <- list(f1,f2,f3,f4)
CohUpPrev1 <- list(ListOfFunctions1,UpperPrevisions1)

UpperPrevisions2 <- c(1/4-0.04,1/4+0.04,1/4-0.01)
ListOfFunctions2 <- list(f1,f2,f3)
CohUpPrev2 <- list(ListOfFunctions2,UpperPrevisions2)

ImpreciseModel <- list(CohUpPrev1,CohUpPrev2)

lbomega <- 1
ubomega <- 4
epsilon <- 0.01

ArgMinDist(x,lbomega,ubomega,epsilon,ImpreciseModel)
}
\keyword{nonparametric}


