% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/weighted_CN.R
\name{weighted_CN}
\alias{weighted_CN}
\title{Calculate the Weighted CN (Curve Number)}
\usage{
weighted_CN(
  CN = NULL,
  CN_area_table = NULL,
  CN_area_pct_table = NULL,
  area = NULL,
  area_pct = NULL,
  area_units = c("acre", "square feet", "square mile", "hectare", "square kilometers")
)
}
\arguments{
\item{CN}{numeric vector containing dimensionless Curve Number(s)}

\item{CN_area_table}{data.frame/data.table/tibble, list, or matrix
containing the CN in column 1 and the area in column 2}

\item{CN_area_pct_table}{data.frame/data.table/tibble, list, or matrix
containing the CN in column 1 and the area_pct in column 2}

\item{area}{numeric vector containing the surface land area}

\item{area_pct}{numeric vector containing the surface land area, as a
percent (decimal or whole number)}

\item{area_units}{character vector containing the units for area
(default = "acre"). The units should be consistent and not mixed.}
}
\value{
the Weighted Curve Number as a single numeric vector, in the range
   [0, 100]
}
\description{
This function computes the weighted CN (Curve Number) using the
user-supplied unit or the default unit of an acre.
}
\note{
This function was originally part of Claudia Vitolo's curvenumber package
that Irucka now maintains.
}
\examples{

# Note: the default area unit is an acre

# Example 1

library("iemisc")

area1 <- c(220, 150, 30)
CN1 <- c(75, 89, 80)
weighted_CN(CN = CN1, area = area1)


# Example 2

library("iemisc")

area2 <- c(220, 150, 30)
area_pct2 <- area2 / sum(area2)
CN2 <- c(80, 95, 80)
CN_area_pct_table2 <- data.frame(CN2, area_pct2)
weighted_CN(CN_area_pct_table = CN_area_pct_table2)


# Example 3

install.load::load_package("iemisc", "data.table")

CN_area_table3 <- data.table(CN = c(98, 100, 45), area = c(2.53, 453.00, 0.21))
weighted_CN(CN_area_table = CN_area_table3)


# Example 4

library("iemisc")

CN4 <- c(98, 100, 45)
area_pct4 <- c(0.15, 0.23, 0.62)
weighted_CN(CN = CN4, area_pct = area_pct4)


# Example 5

library("iemisc")

import::from(ramify, mat)


data_matrix5a <- matrix(c(98, 30, 40, 43, 57, 3.24, 1, 30, 50, 123),
nrow = 5, ncol = 2, dimnames = list(rep("", 5), c("C", "Area")))
weighted_CN(CN_area_table = data_matrix5a)


# using ramify to create the matrix
data_matrix5b <- mat("98 30 40 43 57;3.24 1 30 50 123", rows = FALSE,
sep = " ", dimnames = list(rep("", 5), c("CN", "Area")))
weighted_CN(CN_area_table = data_matrix5b)


# Example 6 - using area in square feet

library("iemisc")

data_list6 <- list(CN = c(77, 29, 68), Area = c(43560, 56893, 345329.32))
weighted_CN(CN_area_table = data_list6, area_units = "square feet")


# Example 7 - using area in whole percents

library("iemisc")

CN7 <- c(61, 74)
area_pct7 <- c(30, 70)
weighted_CN(CN = CN7, area_pct = area_pct7)


}
\references{
\enumerate{
   \item United States Department of Agriculture Soil Conservation Service National Employee Development Staff, "Engineering Hydrology Training Series Module 104 - Runoff Curve Number Computations Study Guide", September 1989, page 21, \url{https://web.archive.org/web/20210414043852/https://www.wcc.nrcs.usda.gov/ftpref/wntsc/H&H/training/runoff-curve-numbers1.pdf}. Retrieved thanks to the Internet Archive: Wayback Machine
   \item Dr. Clyde Munster, P.E., Texas A&M University Department of Biological and Agricultural Engineering, "Rational Method: Calculating Peak Flow Rates", \url{https://web.archive.org/web/20180218221234/http://munster.tamu.edu/Study_Abroad/BAEN460_AGSM335/PowerPoint/RationalMethod_5.pdf}. Retrieved thanks to the Internet Archive: Wayback Machine
   \item United States Department of Agriculture Natural Resources Conservation Service Conservation Engineering Division, "Urban Hydrology for Small Watersheds Technical Release 55 (TR-55)", June 1986, \url{https://web.archive.org/web/20220114201558/https://www.nrcs.usda.gov/Internet/FSE_DOCUMENTS/stelprdb1044171.pdf}. Retrieved thanks to the Internet Archive: Wayback Machine
}
}
