context("comorbidities, generally slow tests")


set.seed(1441)

n <- 500
np <- round(n/20) # icd9 codes per patients

randomShortIcd9 <- as.character(floor(runif(min = 10000, max = 99999, n = n)))
randomSampleAhrq <- sample(unname(c(ahrqComorbid, recursive = TRUE)), replace = TRUE, size = n)
fewIcd9 <- c("27801", "7208", "25001", "34400", "4011", "4011")

patientData <- data.frame(
  visitId = c(1000, 1000, 1000, 1001, 1001, 1002),
  icd9 = fewIcd9,
  poa = factor(c("Y", "N", "Y", "N", "Y", "N"))
)

randomPatients <- data.frame(
  visitId = sample(seq(1, np), replace = TRUE, size=n),
  icd9 = randomShortIcd9,
  poa = as.factor(sample(x=c("Y", "N", "n", "n", "y", "X", "E", "", NA), replace = TRUE, size=n))
)

# random patients with icd9 codes selected from ahrq data
randomPatientsAhrqIcd9 <- randomPatients
randomPatientsAhrqIcd9[["icd9"]] <- randomSampleAhrq

test_that("no NA values in the co-morbidity lists", {
  expect_false(any(is.na(unlist(unname(ahrqComorbid)))))
  expect_false(any(is.na(unlist(unname(ahrqComorbidAll)))))
  expect_false(any(is.na(unlist(unname(quanDeyoComorbid)))))
  expect_false(any(is.na(unlist(unname(quanElixhauserComorbid)))))
  expect_false(any(is.na(unlist(unname(elixhauserComorbid)))))
})

test_that("built-in icd9 to comorbidity mappings are all valid", {
  expect_true(icd9ValidMappingShort(ahrqComorbid))
  expect_true(icd9ValidMappingShort(quanDeyoComorbid))
  expect_true(icd9ValidMappingShort(quanElixhauserComorbid))
  expect_true(icd9ValidMappingShort(elixhauserComorbid))
})

test_that("ahrq icd9 mappings are all generated from the current generation code", {
  expect_identical(ahrqComorbid, parseAhrqSas(condense = FALSE, save = FALSE, returnAll = FALSE)) # same but from source data. Should be absolutely identical.
  expect_identical(ahrqComorbidAll, parseAhrqSas(condense = FALSE, save = FALSE, returnAll = TRUE)) # same but from source data. Should be absolutely identical.
})
test_that("quan charlson icd9 mappings are all generated from the current generation code", {
  expect_identical(quanDeyoComorbid, parseQuanDeyoSas(condense = FALSE, save = FALSE))
})
test_that("quan elixhauser icd9 mappings are all generated from the current generation code", {
  expect_identical(quanElixhauserComorbid, parseQuanElixhauser(condense = FALSE, save = FALSE))
})
test_that("elixhauser icd9 mappings are all generated from the current generation code", {
  expect_identical(elixhauserComorbid, parseElixhauser(condense = FALSE, save = FALSE))
})

test_that("can condense the big lists of comorbidities without errors", {
  # this is a useful test because the data weren't generated by just expanding
  # base ranges (which is how the condense works in reverse)
  ahrq <- lapply(ahrqComorbid, icd9CondenseShort)
  quanDeyo <- lapply(quanDeyoComorbid, icd9CondenseShort)
  quanElixhauser <- lapply(quanElixhauserComorbid, icd9CondenseShort)
  elixhauser <- lapply(elixhauserComorbid, icd9CondenseShort)
  expect_is(ahrq, class = "list")
  expect_is(elixhauser, class = "list")
  expect_is(quanDeyo, class = "list")
  expect_is(quanElixhauser, class = "list")
  # the comorbidity mappings save in \code{data} should not be condensed.
  expect_that(ahrq, not(equals(ahrqComorbid)))
  expect_that(elixhauser, not(equals(elixhauserComorbid)))
  expect_that(quanDeyo, not(equals(quanDeyoComorbid)))
  expect_that(quanElixhauser, not(equals(quanElixhauserComorbid)))
})

test_that("ahrq make sure all the children are listed in the saved data.", {
  ahrq <- lapply(ahrqComorbid, icd9ChildrenShort)
  expect_equal(ahrq, ahrqComorbid)
})
test_that("elixhauser make sure all the children are listed in the saved data.", {
  elixhauser <- lapply(elixhauserComorbid, icd9ChildrenShort)
  expect_equal(elixhauser, elixhauserComorbid)
})
test_that("quan charlson make sure all the children are listed in the saved data.", {
  quanDeyo <- lapply(quanDeyoComorbid, icd9ChildrenShort)
  expect_equal(quanDeyo, quanDeyoComorbid)
})
test_that("quan elixhauser make sure all the children are listed in the saved data.", {
  quanElixhauser <- lapply(quanElixhauserComorbid, icd9ChildrenShort)
  expect_equal(quanElixhauser, quanElixhauserComorbid)
})

test_that("condense an ICD-9 code set to minimal group", {
  expect_equal(sort(icd9CondenseShort("98799" %i9s% "98901")), sort(c("98799", "988", "98900", "98901")))
  # TODO: more tests
})


# test_that("AHRQ interpretation at least returns something reasonable", {
#   result <- parseAhrqSas(sasPath = system.file("extdata", "comformat2012-2013.txt", package="icd9"), save = FALSE)
#   expect_that(result, is_a("list"))
#   expect_true(length(result) > 10)
# })

test_that("HTN subgroups all worked", {
  # pick one subcategory
  expect_true(all(ahrqComorbidAll$HTNPREG %in% ahrqComorbid$HTNCX))

  # and we didn't drop any:
  expect_true(all(ahrqComorbidAll$HTNCX %in% ahrqComorbid$HTNCX))
  expect_true(all(ahrqComorbidAll$CHF %in% ahrqComorbid$CHF))
  expect_true(all(ahrqComorbidAll$RENLFAIL %in% ahrqComorbid$RENLFAIL))

})


test_that("icd9 codes to comorbities", {

  testdat <- structure(
    list(visitId = c(207210584L, 207210584L, 207210584L,
                     207210584L, 207210584L, 207210600L, 207210600L,
                     207210600L, 207210600L, 207210600L, 207210600L,
                     207210600L, 207210600L, 207210600L, 207210600L,
                     207210600L, 207210600L, 207210600L, 207210618L, 207210618L),
         icd9Code = structure(
           c(17L, 1L, 14L, 10L, 13L, 11L, 8L, 6L,
             18L, 2L, 7L, 19L, 3L, 5L, 20L, 16L, 12L, 4L, 15L, 9L),
           .Label = c("04104", "1912", "2449", "2949", "29680", "4254", "4371", "4530",
                      "5070", "59370", "5990", "71595", "74689", "7757", "85226",
                      "V153", "77182", "45341", "78097", "V1529"), class = "factor"),
         poa = c("N", "N", "N", "Y", "Y", "Y", "Y", "Y", "Y", "Y",
                 "Y", "Y", "Y", "Y", "E", "E", "Y", "Y", "Y", "N")),
    .Names = c("visitId", "icd9Code", "poa"),
    row.names = 5000000:5000019,
    class = "data.frame")

  # TODO write the test!

})

test_that("icd9 comorbidities are created correctly, and logical to binary conversion ok", {

  ptdf <- icd9Comorbidities(icd9df = patientData, icd9Mapping = ahrqComorbid, visitId = "visitId")

  expect_equal(names(ptdf), c("visitId", names(ahrqComorbid)))

  expect_true(all(sapply(names(ahrqComorbid), function(x) class(ptdf[, x])) == "logical"))
  ptdflogical <- logicalToBinary(ptdf)
  expect_true(all(sapply(names(ahrqComorbid), function(x) class(ptdflogical[, x])) == "integer"))
  # do not expect all the rest of patient data to be returned - we aren't
  # responsible for aggregating other fields by visitId!
  expect_equal(dim(ptdf), c(length(unique(patientData[["visitId"]])), 1 + length(ahrqComorbid)))
  expect_true(all(names(ptdf) %in% c("visitId", names(ahrqComorbid))))
  expect_true(all(names(ptdflogical) %in% c("visitId", names(ahrqComorbid))))

  expect_equal(
    logicalToBinary(data.frame(a = c("jack", "hayley"), b = c(TRUE, FALSE), f = c(TRUE, TRUE))),
    data.frame(a = c("jack", "hayley"), b = c(1, 0), f = c(1, 1))
  )

})
