#' Estimates in Linear M-model and Linear Y-model with Delta Method
#' 
#' Parameter estimates in system of correlated linear M-model and linear
#' Y-model with treatment-mediator interaction using delta method.
#' 
#' %% ~~ If necessary, more details than the description above ~~ 
#' P-values are computed from normal distribution.
#' 
#' @param fit.M a fitted model object for mediator. It must be an object
#' generated by function ``\code{lm}''
#' @param fit.Y a fitted model object for outcome. It must be an object
#' generated by function ``\code{lm}''. It can contain treatment-mediator
#' interaction
#' @param rho a numerical variable specifying the correlation coefficient
#' between the residual of the M-model and the residual of the Y-model. Its
#' range is between \code{-1} and \code{1}
#' @return A list containing the following components: \item{M.model}{a data
#' frame containing the results for the M-model } \item{Y.model}{a data frame
#' containing the results for the Y-model }
#' @author Kai Wang \code{<kai-wang@@uiowa.edu>}
#' @keywords estimates
#' @examples
#' 
#' data("jobs", package = "mediation")
#' 
#' fit.M <- lm(job_seek ~ treat + econ_hard + sex + age, data=jobs)
#' fit.Y <- lm(depress2 ~ treat + job_seek + econ_hard + sex + age, data=jobs)
#' delta.lnl(fit.M, fit.Y, rho=0.2)
#' 
#' 
#' fit.M <- lm(job_seek ~ treat + econ_hard + sex + age , data=jobs)
#' fit.Y <- lm(depress2 ~ treat*job_seek+ econ_hard + sex + age , data=jobs)
#' delta.lnl(fit.M, fit.Y, rho=0.5)
#' 
#' @export delta.lnl
#' @import stats
delta.lnl = function(fit.M, fit.Y, rho=0) {
    M = all.vars(formula(fit.M))[1]
    Y.factors = attr(terms(formula(fit.Y)), "term.labels")
    n = length(residuals(fit.M))

    s1 = sigma(fit.M)
    s2.t = sigma(fit.Y)              # s.tilde
    tau = rho/sqrt(1-rho^2)*s2.t/s1

    M.c = coef(summary(fit.M))
    adjd = coef(summary(fit.Y))
    for (var in row.names(M.c)){
       	adjd[var,"Estimate"] = adjd[var,"Estimate"] + M.c[var,"Estimate"]*tau
     	adjd[var,"Std. Error"] = sqrt(adjd[var,"Std. Error"]^2+tau^2*(M.c[var,"Std. Error"]^2+M.c[var,"Estimate"]^2/n))
    }
    adjd[M,"Estimate"] = adjd[M,"Estimate"] - tau
    adjd[M,"Std. Error"] = sqrt(adjd[M,"Std. Error"]^2 + tau^2/n)
    adjd[, "t value"] = adjd[,"Estimate"]/adjd[,"Std. Error"]
    adjd[, "Pr(>|t|)"] = signif(2*pnorm(abs(adjd[,"t value"]), lower.tail=FALSE), 4)

    list(M.model = as.data.frame(M.c), Y.model = as.data.frame(adjd))
}
