## ---- include = FALSE---------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

## ----setup--------------------------------------------------------------------
library(hypervolume)
library(palmerpenguins)
library(ggplot2)
library(gridExtra)
set.seed(123)

## ---- results = "hide"--------------------------------------------------------
data(penguins)
data(quercus)

## ---- results = "hide", eval = FALSE------------------------------------------
#  # Run time with cores = 2 is around 25 minutes
#  hv = hypervolume(na.omit(penguins)[,3:4], verbose = FALSE)
#  resample_seq_path = hypervolume_resample("penguins_hvs", hv, method = "bootstrap seq", n = 30, seq = c(100, 125, 150, 175, 200, 225, 250, 275, 300), cores = 20)
#  
#  hypervolume_funnel(resample_seq_path, title = "Volume of Hypervolumes at Different Resample Sizes") + ylab("Volume")
#  
#  plot1 = hypervolume_funnel(resample_seq_path, title = "Mean of Bill Length at Different Resample Sizes",
#                     func = function(x) {get_centroid(x)["bill_length_mm"]}) +
#    ylab("Bill Length (mm)")
#  
#  plot2 = hypervolume_funnel(resample_seq_path, title = "Mean of Bill Depth at Different Resample Sizes",
#                     func = function(x) {get_centroid(x)["bill_depth_mm"]}) +
#    ylab("Bill Depth (mm)")
#  
#  grid.arrange(plot1, plot2, nrow = 2)

## ---- fig.width=8, echo=FALSE-------------------------------------------------
upperq = c(205.1020, 198.3495, 197.0977, 185.7079, 179.1139, 184.4861, 180.7195, 176.4766, 175.8285)
sample_mean = c(179.3345, 176.0790, 170.4279, 170.0950, 167.8839, 166.2428, 167.3649, 162.8489, 162.5742)
lowerq = c(158.3234, 151.5433, 153.7521, 153.8541, 157.1399, 147.0054, 152.7173, 151.0459, 148.4239)
n = c(100, 125, 150, 175, 200, 225, 250, 275, 300)
volume_dat = data.frame(upperq, sample_mean, lowerq, n)
ggplot(volume_dat, aes(x = n)) + geom_line(aes(y = upperq)) + geom_line(aes(y = sample_mean), col = "blue") + geom_line(aes(y = lowerq)) + ggtitle("Volume of Hypervolumes at Different Resample Sizes", "Confidence interval: 0.95") + ylab("Volume") + xlab("Resample size")

upperq = c(45.20545, 44.78558, 44.99452, 45.02968, 45.02962, 44.76026, 45.04892, 45.11330, 44.91576)
sample_mean = c(44.21241, 44.24773, 44.40511, 44.21123, 44.28321, 44.32482, 44.44618, 44.35699, 44.38820)
lowerq = c(43.39146, 43.62949, 43.73162, 43.77126, 43.74826, 43.72950, 43.89345, 43.65643, 43.85460)
length_dat = data.frame(upperq, sample_mean, lowerq, n)
ggplot(length_dat, aes(x = n)) + geom_line(aes(y = upperq)) + geom_line(aes(y = sample_mean), col = "blue") + geom_line(aes(y = lowerq)) + ggtitle("Average Bill Length at Different Resample Sizes", "Confidence interval: 0.95") + ylab("bill length (mm)") + xlab("Resample size")

upperq = c(17.59443, 17.58865, 17.50011, 17.54498, 17.52506, 17.59228, 17.53206, 17.47092, 17.54269)
sample_mean = c(17.33624, 17.33152, 17.30638, 17.34717, 17.30928, 17.33852, 17.31785, 17.32374, 17.34064)
lowerq = c(16.96755, 17.09749, 17.12346, 17.15404, 17.10467, 17.12946, 17.13240, 17.16913, 17.17636)
depth_dat = data.frame(upperq, sample_mean, lowerq, n)
ggplot(depth_dat, aes(x = n)) + geom_line(aes(y = upperq)) + geom_line(aes(y = sample_mean), col = "blue") + geom_line(aes(y = lowerq)) + ggtitle("Average Bill Depth at Different Resample Sizes", "Confidence interval: 0.95") + ylab("bill depth (mm)") + xlab("Resample size")

## ---- results = "hide"--------------------------------------------------------
hv = hypervolume(na.omit(penguins)[,3:6], verbose = FALSE)
cols_to_bias = c("bill_length_mm", "bill_depth_mm")
mu = apply(hv@Data, 2, max)[cols_to_bias]
sigma = apply(hv@Data, 2, var)[cols_to_bias]*2
biased_path = hypervolume_resample("Bill bias", hv, method = "biased bootstrap", n = 1, mu = mu, sigma = sigma, cols_to_bias = cols_to_bias)

# Read in hypervolume object from file
biased_hv = readRDS(file.path(biased_path, "resample 1.rds"))

combined_dat = data.frame(rbind(hv@Data, biased_hv@Data))
combined_dat['Type'] = rep(c('original', 'biased'), each = nrow(hv@Data))

## ---- fig.width=8, fig.height=8-----------------------------------------------
plot1 = ggplot(combined_dat, aes(y = ..density..)) + geom_histogram(aes(x = bill_depth_mm, fill = Type), bins = 20) +
  facet_wrap(~Type) +
  ggtitle("Distribution of Bill Depth", "Biased resample vs Original sample") +
  xlab("bill depth (mm)")
plot2 = ggplot(combined_dat, aes(y = ..density..)) + geom_histogram(aes(x = bill_length_mm, fill = Type), bins = 20) +
  facet_wrap(~Type) +
  ggtitle("Distribution of Bill Length", "Biased resample vs Original sample") +
  xlab("bill length(mm)")
grid.arrange(plot1, plot2, nrow = 2)

## ---- fig.width=8, fig.height=8-----------------------------------------------
plot1 = ggplot(combined_dat, aes(y = ..density..)) + geom_histogram(aes(x = flipper_length_mm, fill = Type), bins = 20) +
  facet_wrap(~Type) +
  ggtitle("Distribution of Flipper Length", "Biased resample vs Original sample") + 
  xlab("flipper length (mm)")
plot2 = ggplot(combined_dat, aes(y = ..density..)) + geom_histogram(aes(x = body_mass_g, fill = Type), bins = 20) +
  facet_wrap(~Type) +
  ggtitle("Distribution of Body Mass", "Biased resample vs Original sample") +
  xlab("body mass (g)")
grid.arrange(plot1, plot2, nrow = 2)

## ---- eval=FALSE--------------------------------------------------------------
#  data("quercus")
#  data_alba = subset(quercus, Species=="Quercus alba")[,c("Longitude","Latitude")]
#  data_rubra = subset(quercus, Species=="Quercus rubra")[,c("Longitude","Latitude")]
#  climatelayers <- getData('worldclim', var='bio', res=10, path=tempdir())
#  
#  # z-transform climate layers to make axes comparable
#  climatelayers_ss = climatelayers[[c(1,4,12,15)]]
#  for (i in 1:nlayers(climatelayers_ss))
#  {
#    climatelayers_ss[[i]] <- (climatelayers_ss[[i]] - cellStats(climatelayers_ss[[i]], 'mean')) / cellStats(climatelayers_ss[[i]], 'sd')
#  }
#  climatelayers_ss_cropped = crop(climatelayers_ss, extent(-150,-50,15,60))
#  
#  # extract transformed climate values
#  climate_alba = extract(climatelayers_ss_cropped, data_alba)
#  climate_rubra = extract(climatelayers_ss_cropped, data_rubra)
#  
#  # Generate Hypervolumes
#  hv_alba = hypervolume(climate_alba,name='alba',samples.per.point=10)
#  hv_rubra = hypervolume(climate_rubra,name='rubra',samples.per.point=10)
#  
#  # Method 1: 2hr runtime with 12 threads
#  combined_sample = rbind(climate_alba, climate_rubra)
#  population_hat = hypervolume(combined_sample)
#  
#  # Create bootstrapped hypervolumes of both sample sizes
#  method1_path_size_1669 = hypervolume_resample("quercus_1669_boot", population_hat, "bootstrap", n = 100, cores = 12)
#  method1_path_size_2110 = hypervolume_resample("quercus_2110_boot", population_hat, "bootstrap", n = 100, cores = 12)
#  
#  
#  result1 = hypervolume_overlap_test(hv_alba, hv_rubra, c(method1_path_size_1669, method1_path_size_2110), cores = 12)
#  
#  #Method 2: 9hr runtime with 12 threads
#  method2_path = hypervolume_permute("rubra_alba_permutation", hv1, hv2, n = 1357, cores = 12)
#  
#  result2 = hypervolume_overlap_test(hv1, hv2, method2_path, cores = 2)
#  
#  # Graphical Results of null sorensen statistic
#  plot1 = result1$plots$sorensen + ggtitle("Method 1", as.character(result1$p_values$sorensen)) + xlab("Sorensen Index")
#  plot2 = result2$plots$sorensen + ggtitle("Method 2", as.character(result2$p_values$sorensen)) + xlab("Sorensen Index")
#  grid.arrange(plot1, plot2, ncol=2)

## ---- fig.width=8, echo=FALSE-------------------------------------------------
dist1 = c(0.9460681,0.9070749,0.9012417,0.9305221,0.9299168,0.9309477,0.8835270,0.9452199,0.9274258,0.9371782,0.9214103,0.9066668,0.9481383,0.9307223,0.9306863,0.9054644,0.9378928,0.9126366,0.9320379,0.9033453,0.9035829,0.9266842,0.9087388,0.9375628,0.9250577,0.9384370,0.9155008,0.9255651,0.8785377,0.9411214,0.9417392,0.9303459,0.9406045,0.9394282,0.9409825,0.9099912,0.9188723,0.9455169,0.9441615,0.9381291,0.9201767,0.9308442,0.9037008,0.9069284,0.9418246,0.9458473,0.8986521,0.9278720,0.9213327,0.9438552,0.9434039,0.8957342,0.9237137,0.9498590,0.9417622,0.9259587,0.9487872,0.9234788,0.9372707,0.9414040,0.9255091,0.9303649,0.9207590,0.9350798,0.9003145,0.9320198,0.9344136,0.9371855,0.9444113,0.9286985,0.9334406,0.9364226,0.9434455,0.9281701,0.9168504,0.9236851,0.9213450,0.9403249,0.9402832,0.9408108,0.9390874,0.9433672,0.9195069,0.9445766,0.9242449,0.9158771,0.9208028,0.9279579,0.9296719,0.9382046,0.9375214,0.8596468,0.9202397,0.9431921,0.9220212,0.9283620,0.9263612,0.9378907,0.9446075,0.9469510,0.9416750,0.9174898,0.9142472,0.9419483,0.9268888,0.9275230,0.8740735,0.9402285,0.9258918,0.9378495,0.9313972,0.9190398,0.9426908,0.9172693,0.9297691,0.8946302,0.9307448,0.9025249,0.9411694,0.8992851,0.9162579,0.9321675,0.9129656,0.9429609,0.9287345,0.9458498,0.9078201,0.9211948,0.8685230,0.9370924,0.9350370,0.9334251,0.9281050,0.9448739,0.9457466,0.9002925,0.9163378,0.9336595,0.9441684,0.9467563,0.9294146,0.9320860,0.8942672,0.9127085,0.9422962,0.9398696,0.9011797,0.9387426,0.9244588,0.9353988,0.9408580,0.8917510,0.9362818,0.9372461,0.9375901,0.9406204,0.9388617,0.9296240,0.9362498,0.9408446,0.9144343,0.9203367,0.9308163,0.9378775,0.8945142,0.9391516,0.9144410,0.9449339,0.9428640,0.9323588,0.9286753,0.9280600,0.9405896,0.9267552,0.9249778,0.9403535,0.9170223,0.9367964,0.9319856,0.9384584,0.9378330,0.9379354,0.9314727,0.9445634,0.9111022,0.9210154,0.9292732,0.9383870,0.9445829,0.9409028,0.9415540,0.8742098,0.9241977,0.9342755,0.9128078,0.9472493,0.9177193,0.9350640,0.9347127,0.9383755,0.9179255,0.9345976,0.9361317,0.9154486,0.8965217,0.8930005,0.8470711,0.9107274,0.8931210,0.9126133,0.9399040,0.9347790,0.9138847,0.8905834,0.9021116,0.8713966,0.9011049,0.8743153,0.9232492,0.8677898,0.9356719,0.9368138,0.9379876,0.9247039,0.9373844,0.9192620,0.8757240,0.8912669,0.8420318,0.9091607,0.9086170,0.9263992,0.9065180,0.9145806,0.9198836,0.8724096,0.9417133,0.9121319,0.9142872,0.9247273,0.9410916,0.9312479,0.8669129,0.9336792,0.9144051,0.9094759,0.9309020,0.9343480,0.9432917,0.9033208,0.9119306,0.8593097,0.9324312,0.9215192,0.9200495,0.9298712,0.9132026,0.9295808,0.9166263,0.9186002,0.8938688,0.8972378,0.9428333,0.9306597,0.8675478,0.9340588,0.8950648,0.9306419,0.9077695,0.9322730,0.9021959,0.9016679,0.9122411,0.8940686,0.9391135,0.9360313,0.8933143,0.9231561,0.9068472,0.9078741,0.9257860,0.9171196,0.9456894,0.9114957,0.8817139,0.9308609,0.9300170,0.9273263,0.9309959,0.9265962,0.9198476,0.8928112,0.9363096,0.9020887,0.8795862,0.9279021,0.8938274,0.9058196,0.9100463,0.9178352,0.9278224,0.9248570,0.9270174,0.9326933,0.9099459,0.9068120,0.8639008,0.9267430,0.9041175,0.9211399,0.9424802,0.9266019,0.9253601,0.9052850,0.9153269,0.8812055,0.9144973,0.8836188,0.9331039,0.8852366,0.9315810,0.9396253,0.9313514,0.9294229,0.9435959,0.9304538,0.8868119,0.9066770,0.8560026,0.9264286,0.9161029,0.9384359,0.9144393,0.9305870,0.9269926,0.8917923,0.9359512,0.9160938,0.9275823,0.9321011,0.9345662,0.9372209,0.8801747,0.9303369,0.9225228,0.9238123,0.9146706,0.9424385,0.9382203,0.9200049,0.9239515,0.8780610,0.9431734,0.9204056,0.9286092,0.9347176,0.9211361,0.9412891,0.9252267,0.9267707,0.9025012,0.9033373,0.9392135,0.9354383,0.8822929,0.9352114,0.9042768,0.9429496,0.9227074,0.9451251,0.9133370,0.9150151,0.9284442,0.9127500,0.9420626,0.9411241,0.9009964,0.9299288,0.9113331,0.9224550,0.9333778,0.9248915,0.9413686,0.9248776,0.8980088,0.9366750,0.9442741,0.9285248,0.9419840,0.9322387,0.9295496,0.8836573,0.9431791,0.9183833,0.8972568,0.9382744,0.8993305,0.9146686,0.9234997,0.9250063,0.9402216,0.9077112,0.9007680,0.9344980,0.9378118,0.9408165,0.8908411,0.9438828,0.9348365,0.9336770,0.9171933,0.9050635,0.9495185,0.9320209,0.9398588,0.9122524,0.9444939,0.9108583,0.9307064,0.9134880,0.9046334,0.9212272,0.9017800,0.9241972,0.9206854,0.9355844,0.9141821,0.9300069,0.8830238,0.9472438,0.9407605,0.9313007,0.9488465,0.9324061,0.9331007,0.9133504,0.9149815,0.9392784,0.9427297,0.9335574,0.9147923,0.9239161,0.9095491,0.9041123,0.9468129,0.9456558,0.8972753,0.9253755,0.9108320,0.9419728,0.9454913,0.9021142,0.9253175,0.9396025,0.9362496,0.9235382,0.9431750,0.9116553,0.9362689,0.9321532,0.9330271,0.9346255,0.9187037,0.9267565,0.9065845,0.9137875,0.9331889,0.9285605,0.9452919,0.9203895,0.9415122,0.9400935,0.9380065,0.9374390,0.9140501,0.9234496,0.9319154,0.9311163,0.9441387,0.9454932,0.9319499,0.9401536,0.9113772,0.9381910,0.9292266,0.9146140,0.9202136,0.9203775,0.9300198,0.9369247,0.9310035,0.8553784,0.9166319,0.9448831,0.9285749,0.9304772,0.9309784,0.9407031,0.9423020,0.9374500,0.9382663,0.8980675,0.8997643,0.9325679,0.9373064,0.9299949,0.8920204,0.9357541,0.9314716,0.9363540,0.9148224,0.8971656,0.9433103,0.9284003,0.9373651,0.9057449,0.9408541,0.9116994,0.9256260,0.9101419,0.9021257,0.9200964,0.8974124,0.9188385,0.9170826,0.9304429,0.9155218,0.9296723,0.8843138,0.9488385,0.9474999,0.9337139,0.9430422,0.9311437,0.9360338,0.9154709,0.9034401,0.9461818,0.9406338,0.9320295,0.9135622,0.9250473,0.9090055,0.9032867,0.9448814,0.9470167,0.8851598,0.9280575,0.9112556,0.9412271,0.9395439,0.9015434,0.9211490,0.9325940,0.9362253,0.9289488,0.9385967,0.9166581,0.9409664,0.9284452,0.9271538,0.9345293,0.9137338,0.9309811,0.9108597,0.9222332,0.9297226,0.9327764,0.9432228,0.9171964,0.9355315,0.9417046,0.9395167,0.9373896,0.9070436,0.9212544,0.9327092,0.9291415,0.9406555,0.9459693,0.9267723,0.9328416,0.9172815,0.9414591,0.9210002,0.9133082,0.9201231,0.9202846,0.9298130,0.9306113,0.9250361,0.8573313,0.9117243,0.9426228,0.9171587,0.9342976,0.9316790,0.9411775,0.9422041,0.9452930,0.9072013,0.9346068,0.9366534,0.9146966,0.8869410,0.8832858,0.8379595,0.9041089,0.8895279,0.9053368,0.9304160,0.9407627,0.9053537,0.8842637,0.8941976,0.8574639,0.8941813,0.8649377,0.9109924,0.8634489,0.9414229,0.9288607,0.9391783,0.9105915,0.9261897,0.9110110,0.8659701,0.8794141,0.8260228,0.8994811,0.8974895,0.9152710,0.8952726,0.9074777,0.9143776,0.8625510,0.9274666,0.8987411,0.9076780,0.9124580,0.9359695,0.9203904,0.8530325,0.9417693,0.9061917,0.8998047,0.9359047,0.9271035,0.9332254,0.8988596,0.9051294,0.8536035,0.9251108,0.9093529,0.9023233,0.9255479,0.9006250,0.9225945,0.9024716,0.9049690,0.8795616,0.8885646,0.9324182,0.9151187,0.8577483,0.9154193,0.8846041,0.9215378,0.9019854,0.9221465,0.8851584,0.8940611,0.9029761,0.8866203,0.9348974,0.9264544,0.8813430,0.9092661,0.8958501,0.8996468,0.9146116,0.9002626,0.9285263,0.9018395,0.8762654,0.9219489,0.9185449,0.9150011,0.9216332,0.9128407,0.9062563,0.9046283,0.9316818,0.8956863,0.8755920,0.9159281,0.8794955,0.8973867,0.9006331,0.9061253,0.9330021,0.8885736,0.8901781,0.9217071,0.9540697,0.9429510,0.9067673,0.9311329,0.9422221,0.9274377,0.9017122,0.8893676,0.9387795,0.9414618,0.9396040,0.9264471,0.9373291,0.9255632,0.9264870,0.9302220,0.8898828,0.9101796,0.8958762,0.9183721,0.9087812,0.9213416,0.9242072,0.9422854,0.8934191,0.9392360,0.9377275,0.9176658,0.9469909,0.9224790,0.9203703,0.9284140,0.9048992,0.9315716,0.9337147,0.9343881,0.8986500,0.9135637,0.9188926,0.8927145,0.9324314,0.9376769,0.8768078,0.9186669,0.8967375,0.9444261,0.9346210,0.9169297,0.9173919,0.9282850,0.9251364,0.9115899,0.9379553,0.9072896,0.9250009,0.9272880,0.9388664,0.9460279,0.9087615,0.9151501,0.9269850,0.9131509,0.9394712,0.9199199,0.9337083,0.9060896,0.9558964,0.9425689,0.9309771,0.9481693,0.9015353,0.9078652,0.9396301,0.9296944,0.9408098,0.9337212,0.9210895,0.9285168,0.8974208,0.9324495,0.9405079,0.9054479,0.9116171,0.9133785,0.9208156,0.9257079,0.9215361,0.8384565,0.9095794,0.9380126,0.9392325,0.9178892,0.9413310,0.9381765,0.9371941,0.9302486,0.8945525,0.9364540,0.9290325,0.9092512,0.8804945,0.8777547,0.8313547,0.9002893,0.8795806,0.8947145,0.9187954,0.9380909,0.9002645,0.8751724,0.8929665,0.8555929,0.8819962,0.8553944,0.9058163,0.8558004,0.9370199,0.9227768,0.9355025,0.9068772,0.9210069,0.9032561,0.8609005,0.8761161,0.8235306,0.8967054,0.8927580,0.9113283,0.8868651,0.9035975,0.9015499,0.8555584,0.9289645,0.8938153,0.9027173,0.9064809,0.9328029,0.9137372,0.8467164,0.9350553,0.8919964,0.8960392,0.9409072,0.9212703,0.9275076,0.8921687,0.8954131,0.8450793,0.9204234,0.8996971,0.9004230,0.9220464,0.8970643,0.9182468,0.8975441,0.9021703,0.8754260,0.8812724,0.9258912,0.9099555,0.8469328,0.9035944,0.8786133,0.9166273,0.8951646,0.9171547,0.8810940,0.8854963,0.9036496,0.8826335,0.9288790,0.9211384,0.8801925,0.8991941,0.8833482,0.8944685,0.9100025,0.9001074,0.9299715,0.8915836,0.8734357,0.9173796,0.9116051,0.9140182,0.9206527,0.9076611,0.9079657,0.9097968,0.9225441,0.8871469,0.8680650,0.9133851,0.8746801,0.8841828,0.8946363,0.8991347,0.9187859,0.9279711,0.9259385,0.9324314,0.9077120,0.9074063,0.8561481,0.9205226,0.9079352,0.9241547,0.9423992,0.9266027,0.9243914,0.9001167,0.9100194,0.8750500,0.9148546,0.8823913,0.9286920,0.8837317,0.9308605,0.9398685,0.9242018,0.9245263,0.9409368,0.9285416,0.8838691,0.9002751,0.8466682,0.9207851,0.9174366,0.9320392,0.9127454,0.9282472,0.9293501,0.8839350,0.9332151,0.9117502,0.9235061,0.9293586,0.9449293,0.9370926,0.8738459,0.9266718,0.9241555,0.9194945,0.9174272,0.9425429,0.9322156,0.9094865,0.9230568,0.8748102,0.9435912,0.9236552,0.9269200,0.9329313,0.9207074,0.9367621,0.9214351,0.9211175,0.9037586,0.9051636,0.9382056,0.9364915,0.8735447,0.9286070,0.9094061,0.9353250,0.9184566,0.9398323,0.9061101,0.9119244,0.9247103,0.9095636,0.9398350,0.9435480,0.9037801,0.9201127,0.9061091,0.9174934,0.9349024,0.9219617,0.9357906,0.9274903,0.8942746,0.9346490,0.9365808,0.9328804,0.9403156,0.9303643,0.9204942,0.8799838,0.9365005,0.9124400,0.8931091,0.9331945,0.8990600,0.9170631,0.9210046,0.9193598,0.9406625,0.9006348,0.8988115,0.9328791,0.9358336,0.9298143,0.8896429,0.9399179,0.9289781,0.9335227,0.9123026,0.8964796,0.9447681,0.9307676,0.9433164,0.9121685,0.9368687,0.9118913,0.9286632,0.9157161,0.8987677,0.9153418,0.9016522,0.9237265,0.9174738,0.9244593,0.9147303,0.9262330,0.8799145,0.9462942,0.9431512,0.9257306,0.9394712,0.9328732,0.9356561,0.9140854,0.9122216,0.9407393,0.9382207,0.9343042,0.9168717,0.9224620,0.9040625,0.9057582,0.9433267,0.9438824,0.8901826,0.9268929,0.9059877,0.9482208,0.9431442,0.9046278,0.9206769,0.9421945,0.9381212,0.9160926,0.9410654,0.9138013,0.9363683,0.9294092,0.9234823,0.9316537,0.9168664,0.9214181,0.9083280,0.9249011,0.9316858,0.9316203,0.9377137,0.9228512,0.9310505,0.9369539,0.9386183,0.9238743,0.9144796,0.9160503,0.9291796,0.9363020,0.9313031,0.9418297,0.9267701,0.9354202,0.9071825,0.9385578,0.9270710,0.9166757,0.9173597,0.9190928,0.9218132,0.9281496,0.9337022,0.8491554,0.9101117,0.9416301,0.9175965,0.9228160,0.9281252,0.9411835,0.9419653,0.9393229,0.8964690,0.9309779,0.9373939,0.9113630,0.8816001,0.8836233,0.8262107,0.8935647,0.8788695,0.8963781,0.9237522,0.9340166,0.8972957,0.8719491,0.8879887,0.8491115,0.8867752,0.8528881,0.9036233,0.8518086,0.9394043,0.9189690,0.9321391,0.9034320,0.9174359,0.9072083,0.8553889,0.8723948,0.8183007,0.8956078,0.8902874,0.9058795,0.8875033,0.8992486,0.9011040,0.8525298,0.9222125,0.8912969,0.8989682,0.9012968,0.9338803,0.9127437,0.8484132,0.9327210,0.8956091,0.8921274,0.9472214,0.9149831,0.9287645,0.8860120,0.8943020,0.8440448,0.9169365,0.8996688,0.8974015,0.9125635,0.8910495,0.9082418,0.8951544,0.8977528,0.8742898,0.8766770,0.9234556,0.9111996,0.8466831,0.9144829,0.8739493,0.9123677,0.8932665,0.9165842,0.8798006,0.8811456,0.8937256,0.8832686,0.9204290,0.9186078,0.8755429,0.8976209,0.8887422,0.8942730,0.9095370,0.8956462,0.9312503,0.8975958,0.8694061,0.9077666,0.9164944,0.9117576,0.9114936,0.9043285,0.8965536,0.9120577,0.9173581,0.8886341,0.8672160,0.9109028,0.8747259,0.8865988,0.8890584,0.8961687,0.9490503,0.9117648,0.9107522,0.9364678,0.9316270,0.9277792,0.8834220,0.9443607,0.9231499,0.9429668,0.9291412,0.9092075,0.9467511,0.9298297,0.9353198,0.9037671,0.9370494,0.9049892,0.9344552,0.9064141,0.9113306,0.9312391,0.9116003,0.9360931,0.9321481,0.9407788,0.9114121,0.9240867,0.8739597,0.9411318,0.9390881,0.9358475,0.9376200,0.9425229,0.9445213,0.9055106,0.9207372,0.9369550,0.9484392,0.9450277,0.9216210,0.9390420,0.8966931,0.9091952,0.9452897,0.9460941,0.8980188,0.9450695,0.9255182,0.9447182,0.9466395,0.8959815,0.9305345,0.9485067,0.9500107,0.9309512,0.9445511,0.9276706,0.9457588,0.9450393,0.9253452,0.9286086,0.9250753,0.9391880,0.9006546,0.9355843,0.9270621,0.9427939,0.9400781,0.9288400,0.9373366,0.9342116,0.9462092,0.9288946,0.9233696,0.9317281,0.9197340,0.9449681,0.9331091,0.9474480,0.9383188,0.9456272,0.9278099,0.9419636,0.9211465,0.9215216,0.9278775,0.9267735,0.9414144,0.9374783,0.9404476,0.8680312,0.9246270,0.9410620,0.9200814,0.9382380,0.9175294,0.9372226,0.9418853,0.9472136,0.9245299,0.8852407,0.8850394,0.9122270,0.9354183,0.9339173,0.9083398,0.9187328,0.9405512,0.9246567,0.8987280,0.8826457,0.9295994,0.9376702,0.9264399,0.9276483,0.9333084,0.9279677,0.9165171,0.9293726,0.8839849,0.8992667,0.8821250,0.9115478,0.8975245,0.9176536,0.9282809,0.9480019,0.8950617,0.9277041,0.9183701,0.9160855,0.9371498,0.9133376,0.9205888,0.9273113,0.8929783,0.9307026,0.9305661,0.9193284,0.8987328,0.9042317,0.9252923,0.8870697,0.9275985,0.9268474,0.8723203,0.9092484,0.8890595,0.9331461,0.9326259,0.9224089,0.9075062,0.9198931,0.9217604,0.9055648,0.9180622,0.8957692,0.9191077,0.9133097,0.9472659,0.9465836,0.9008468,0.9166500,0.9242730,0.9055146,0.9323057,0.9128261,0.9207088,0.9033197,0.9404461,0.9398926,0.9243784,0.9351392,0.8951451,0.9037068,0.9420262,0.9192530,0.9273343,0.9315601,0.9116296,0.9231382,0.8955101,0.9269554,0.9447330,0.8916112,0.8999344,0.8989844,0.9106751,0.9090306,0.9068568,0.8370515,0.8937961,0.9301087,0.9419663,0.9173698,0.9411141,0.9324210,0.9270195,0.9244401,0.9263334,0.8849528,0.8844606,0.9211803,0.9442939,0.9420387,0.9037818,0.9272606,0.9431169,0.9214926,0.8997236,0.8881269,0.9312791,0.9400116,0.9310891,0.9237016,0.9317163,0.9278828,0.9153406,0.9229423,0.8858741,0.9038674,0.8858815,0.9123704,0.9061542,0.9182555,0.9238211,0.9442362,0.8925482,0.9323392,0.9268215,0.9179858,0.9417027,0.9181687,0.9173140,0.9238976,0.8990771,0.9328196,0.9274454,0.9197266,0.8976200,0.9148346,0.9215436,0.8914249,0.9270538,0.9305961,0.8706356,0.9142723,0.8977649,0.9292172,0.9326353,0.9198831,0.9103214,0.9270823,0.9255161,0.9069097,0.9254116,0.9021855,0.9189076,0.9161094,0.9429731,0.9453958,0.9031408,0.9186470,0.9177641,0.9067551,0.9389941,0.9137760,0.9304516,0.9065788,0.9455439,0.9405187,0.9221656,0.9405081,0.8983441,0.9049948,0.9455522,0.9245455,0.9267196,0.9345846,0.9162047,0.9231471,0.8933497,0.9254959,0.9468390,0.8987520,0.9010081,0.9086951,0.9094162,0.9166286,0.9149779,0.8313253,0.8978508,0.9299222,0.9359309,0.9139160,0.9401187,0.9330227,0.9292533,0.9241890,0.9425630,0.9075907,0.9031294,0.9341044,0.9360735,0.9289122,0.8934488,0.9401838,0.9298613,0.9373715,0.9174475,0.9047770,0.9506436,0.9306124,0.9416294,0.9052138,0.9431855,0.9092941,0.9284130,0.9123606,0.9060797,0.9184861,0.9068534,0.9183465,0.9201243,0.9314810,0.9167498,0.9260574,0.8857788,0.9501501,0.9398555,0.9332046,0.9436065,0.9312912,0.9385228,0.9158181,0.9108475,0.9410567,0.9418067,0.9374306,0.9130150,0.9243617,0.9037549,0.9045266,0.9414255,0.9535265,0.8904480,0.9273279,0.9127540,0.9445659,0.9439584,0.9058859,0.9214025,0.9366802,0.9414297,0.9227871,0.9458014,0.9152633,0.9351015,0.9369582,0.9275098,0.9345105,0.9169498,0.9289601,0.9095694,0.9208559,0.9295335,0.9348501,0.9466525,0.9222103,0.9369245,0.9405816,0.9424609,0.9402760,0.9146657,0.9226968,0.9282824,0.9345298,0.9385920,0.9460935,0.9387122,0.9354671,0.9175410,0.9407685,0.9210571,0.9129708,0.9145489,0.9187175,0.9324332,0.9345074,0.9296580,0.8539708,0.9189552,0.9423547,0.9223767,0.9248912,0.9302156,0.9401967,0.9417779,0.9443719,0.9064392,0.9366839,0.9432619,0.9103277,0.8871758,0.8831082,0.8347828,0.9031359,0.8869354,0.9009416,0.9303412,0.9409495,0.9083973,0.8769105,0.8878583,0.8561017,0.8986172,0.8615858,0.9094987,0.8553992,0.9424472,0.9341361,0.9384659,0.9114651,0.9331099,0.9061101,0.8636346,0.8837706,0.8292067,0.9030722,0.8974756,0.9126540,0.8970333,0.9072892,0.9090261,0.8617578,0.9316213,0.8941267,0.9074765,0.9072880,0.9343375,0.9260052,0.8575151,0.9390864,0.9035179,0.9026589,0.9377352,0.9184677,0.9377011,0.8943225,0.8981417,0.8486165,0.9229383,0.9073125,0.9093596,0.9176744,0.9000295,0.9181024,0.9051042,0.9060488,0.8821374,0.8839033,0.9348536,0.9138419,0.8541112,0.9160173,0.8858641,0.9185727,0.9008879,0.9200911,0.8920980,0.8861744,0.9033585,0.8843055,0.9303918,0.9232248,0.8757006,0.9071223,0.8900297,0.8932930,0.9103230,0.9110516,0.9349328,0.9041061,0.8736250,0.9188634,0.9256500,0.9125453,0.9235640,0.9122334,0.9052046,0.9036295,0.9281931,0.8929784,0.8781636,0.9200321,0.8804571,0.9000293,0.9003987,0.9019034,0.9289814,0.9198599,0.9180219,0.9324582,0.9085141,0.8983622,0.8641492,0.9252851,0.9045214,0.9162492,0.9377975,0.9238498,0.9265666,0.9054059,0.9209684,0.8832992,0.9089915,0.8813695,0.9264739,0.8849851,0.9231163,0.9406137,0.9218609,0.9354377,0.9402920,0.9327512,0.8904699,0.8993072,0.8610800,0.9229514,0.9180080,0.9328067,0.9137937,0.9290562,0.9267610,0.8888794,0.9291862,0.9185375,0.9252968,0.9329819,0.9348090,0.9405802,0.8786947,0.9244913,0.9271386,0.9189574,0.9108513,0.9407069,0.9351136,0.9172602,0.9234720,0.8744896,0.9402544,0.9282061,0.9239574,0.9399921,0.9274235,0.9416521,0.9227391,0.9231779,0.9022996,0.9060625,0.9348962,0.9334094,0.8778965,0.9299256,0.9153490,0.9361600,0.9245738,0.9312368,0.9112820,0.9107886,0.9211035,0.9059980,0.9290066,0.9350481,0.9044039,0.9233014,0.9175780,0.9129003,0.9274545,0.9229172,0.9329502,0.9228473,0.9005322,0.9330010,0.9351468,0.9406047,0.9322064,0.9305619,0.9310000,0.8770160,0.9308232,0.9147611,0.9000126,0.9381464,0.9010442,0.9103106,0.9247255,0.9262430,0.9260495,0.8901796,0.8919868,0.9231826,0.9440291,0.9409534,0.9048380,0.9313649,0.9403120,0.9275629,0.9043906,0.8907459,0.9320911,0.9423946,0.9347129,0.9189825,0.9457646,0.9196232,0.9241184,0.9173576,0.8904455,0.9102290,0.8921645,0.9121462,0.9060320,0.9272349,0.9307013,0.9425899,0.8979098,0.9366885,0.9348130,0.9164838,0.9447099,0.9242806,0.9274249,0.9236933,0.8977097,0.9364393,0.9337070,0.9253131,0.9006241,0.9116828,0.9168879,0.8909550,0.9277172,0.9402470,0.8777765,0.9217148,0.8991692,0.9429981,0.9408473,0.9183873,0.9187345,0.9195195,0.9338974,0.9134527,0.9274134,0.9031932,0.9291354,0.9226412,0.9362575,0.9480869,0.9069359,0.9189702,0.9209366,0.9079998,0.9349002,0.9237199,0.9378040,0.9076825,0.9473177,0.9490181,0.9309453,0.9452046,0.9042473,0.9120791,0.9408205,0.9294098,0.9315200,0.9394543,0.9223829,0.9282488,0.9033347,0.9275869,0.9368551,0.9085745,0.9092596,0.9126693,0.9228864,0.9206581,0.9188516,0.8437034,0.9074636,0.9390787,0.9350308,0.9193408,0.9378172,0.9344528,0.9347729,0.9312257,0.9286930,0.9358516,0.9361463,0.9325640,0.9082030,0.9067637,0.8571687,0.9264666,0.9025714,0.9203658,0.9462565,0.9335175,0.9242716,0.9016079,0.9070284,0.8734333,0.9123053,0.8785298,0.9247555,0.8772127,0.9283515,0.9475300,0.9283335,0.9321792,0.9425334,0.9275039,0.8874430,0.8981338,0.8514569,0.9187114,0.9166896,0.9370845,0.9117600,0.9260660,0.9305786,0.8833573,0.9367397,0.9212113,0.9295713,0.9283282,0.9483200,0.9436850,0.8733891,0.9277222,0.9236231,0.9249063,0.9211459,0.9467901,0.9398277,0.9180006,0.9257341,0.8687974,0.9403300,0.9255870,0.9245808,0.9404498,0.9222963,0.9416395,0.9234855,0.9278426,0.8990574,0.9036967,0.9440815,0.9431309,0.8751643,0.9347490,0.9033685,0.9406872,0.9186270,0.9408721,0.9080297,0.9110012,0.9216037,0.9092422,0.9382180,0.9494085,0.9020684,0.9248119,0.9126738,0.9178967,0.9339085,0.9224571,0.9469023,0.9267858,0.8925781,0.9388336,0.9416470,0.9376854,0.9398171,0.9343451,0.9265177,0.8912667,0.9409336,0.9168138,0.8933507,0.9391025,0.8993107,0.9163082,0.9182807,0.9293339,0.9370046,0.8972958,0.8933327,0.9311257,0.9402113,0.9392304,0.8958123,0.9369077,0.9371096,0.9359614,0.9118213,0.8954406,0.9385476,0.9340196,0.9310447,0.9134450,0.9451802,0.9182909,0.9270401,0.9196701,0.9008603,0.9153959,0.8989889,0.9240620,0.9132772,0.9353322,0.9200470,0.9315436,0.8908788,0.9392681,0.9322932,0.9255852,0.9468870,0.9365346,0.9363610,0.9242788,0.9021348,0.9411161,0.9295589,0.9276160,0.9085370,0.9154534,0.9125816,0.8993764,0.9389538,0.9394076,0.8898594,0.9188250,0.9054019,0.9446036,0.9379103,0.9131804,0.9186856,0.9375031,0.9356312,0.9200600,0.9356047,0.9097809,0.9312256,0.9268035,0.9372763,0.9414671,0.9126953,0.9263948,0.9116850,0.9225811,0.9397004,0.9220446,0.9429492,0.9162965,0.9366010,0.9443577,0.9363377,0.9394663,0.9132080,0.9160134,0.9338490,0.9325561,0.9446786,0.9413694,0.9319597,0.9386558,0.9085076,0.9389801,0.9300898,0.9062194,0.9193808,0.9138536,0.9235346,0.9298771,0.9249561,0.8486337,0.9116082,0.9410419,0.9259418,0.9258614,0.9374401,0.9429907,0.9414452,0.9376888,0.9256499,0.8866431,0.8866511,0.9112573,0.9322345,0.9334997,0.9016103,0.9234792,0.9392173,0.9226071,0.9008078,0.8820328,0.9307618,0.9401830,0.9289295,0.9234751,0.9328494,0.9189041,0.9166200,0.9197833,0.8901480,0.9097370,0.8913684,0.9124768,0.8996875,0.9184777,0.9281840,0.9367809,0.8988590,0.9295396,0.9245400,0.9166852,0.9355910,0.9171063,0.9202775,0.9221859,0.8958340,0.9351951,0.9248228,0.9227488,0.9008235,0.9155443,0.9158874,0.8902358,0.9239984,0.9344299,0.8702559,0.9105861,0.8990526,0.9370379,0.9330447,0.9117237,0.9032661,0.9225244,0.9276271,0.9111735,0.9277697,0.9051784,0.9266216,0.9245689,0.9332589,0.9442726,0.9039513,0.9193561,0.9152292,0.9040379,0.9398659,0.9171624,0.9291041,0.9035602,0.9486941,0.9464012,0.9300894,0.9413005,0.8971113,0.9034772,0.9369739,0.9251991,0.9320987,0.9354608,0.9175771,0.9222880,0.9022965,0.9241223,0.9320478,0.8978448,0.8984576,0.9087315,0.9150557,0.9144295,0.9252164,0.8405455,0.8984509,0.9345249,0.9323832,0.9163496,0.9320681,0.9318338,0.9311610,0.9291779,0.9469995,0.9052564,0.9018258,0.9284883,0.9380350,0.9340786,0.8919201,0.9446423,0.9332653,0.9372997,0.9191930,0.9042017,0.9506794,0.9332718,0.9369492,0.9071804,0.9457322,0.9092454,0.9278249,0.9126778,0.9077145,0.9249331,0.9104219,0.9254531,0.9168543,0.9326411,0.9136523,0.9320980,0.8799974,0.9441992,0.9440598,0.9300349,0.9454064,0.9360455,0.9332168,0.9172514,0.9121159,0.9431707,0.9377394,0.9313192,0.9088641,0.9277321,0.9039687,0.9042882,0.9437989,0.9484329,0.8932664,0.9254870,0.9176589,0.9464071,0.9461277,0.9034292,0.9198043,0.9373963,0.9420602,0.9187403,0.9364840,0.9140781,0.9323006,0.9361626,0.9294930,0.9319229,0.9182473,0.9270483,0.9057733,0.9201105,0.9373122,0.9291123,0.9509959,0.9186418,0.9382704,0.9421862,0.9345370,0.9416061,0.9134192,0.9217066,0.9280024,0.9417317,0.9391645,0.9432719,0.9324573,0.9379301,0.9150991,0.9376579,0.9253487,0.9071597,0.9222022,0.9198652,0.9276302,0.9321008,0.9260157,0.8565680,0.9169127,0.9446652,0.9287385,0.9295408,0.9273136,0.9413820,0.9415586,0.9387585,0.9406758,0.8988925,0.8991187,0.9355189,0.9496101,0.9406021,0.8971054,0.9362897,0.9412628,0.9396690,0.9146316,0.8983627,0.9440609,0.9370937,0.9421454,0.9139431,0.9450284,0.9217919,0.9262142,0.9196923,0.9000749,0.9159586,0.9023085,0.9288765,0.9194927,0.9311514,0.9230901,0.9388065,0.8905759,0.9471864,0.9494841,0.9283786,0.9542362,0.9343830,0.9305412,0.9216864,0.9086309,0.9450934,0.9362668,0.9396030,0.9127637,0.9195488,0.9088540,0.9026048,0.9366530,0.9445026,0.8854404,0.9262918,0.9100300,0.9476437,0.9463772,0.9169531,0.9207419,0.9384334,0.9386905,0.9215782,0.9460922,0.9179736,0.9324433,0.9339276,0.9425683,0.9409869,0.9177686,0.9261842,0.9142572,0.9198935,0.9447241,0.9284800,0.9440122,0.9170223,0.9413458,0.9473412,0.9392204,0.9454363,0.9121022,0.9172746,0.9412693,0.9322724,0.9442519,0.9442314,0.9265039,0.9336840,0.9097470,0.9382915,0.9404149,0.9179335,0.9127381,0.9178810,0.9261280,0.9277955,0.9314817,0.8503803,0.9163053,0.9423575,0.9291858,0.9245823,0.9371258,0.9487723,0.9423463,0.9404188,0.9257336,0.9153445,0.9170228,0.9337669,0.9122226,0.9080569,0.8715948,0.9300550,0.9097844,0.9261913,0.9333060,0.9251110,0.9275277,0.9036864,0.9206705,0.8886411,0.9172993,0.8908997,0.9389487,0.8864773,0.9181738,0.9391656,0.9243409,0.9357790,0.9351284,0.9389819,0.8966957,0.9051303,0.8600248,0.9276175,0.9205832,0.9334694,0.9205830,0.9350709,0.9340193,0.8912661,0.9215600,0.9270516,0.9323716,0.9365915,0.9287587,0.9400033,0.8843323,0.9222534,0.9262917,0.9247533,0.9103205,0.9446201,0.9311620,0.9161426,0.9235011,0.8825767,0.9402749,0.9331711,0.9279302,0.9453323,0.9276033,0.9366327,0.9258371,0.9365712,0.9062933,0.9111357,0.9381346,0.9406857,0.8805613,0.9403072,0.9153956,0.9423779,0.9282425,0.9319386,0.9134695,0.9172383,0.9261136,0.9107604,0.9277371,0.9383937,0.9094708,0.9333973,0.9265044,0.9271323,0.9419283,0.9319213,0.9294895,0.9222339,0.9060812,0.9297820,0.9387208,0.9419771,0.9415282,0.9409050,0.9315112,0.8780099,0.9352787,0.9196898,0.9018082,0.9410683,0.9078867,0.9175622,0.9250200,0.9309102,0.9164725,0.9221844,0.9265995,0.9246060,0.9064976,0.8958529,0.8575580,0.9178941,0.8949800,0.9116292,0.9343090,0.9224646,0.9186972,0.9081666,0.9086154,0.8765802,0.9074048,0.8799932,0.9205126,0.8780620,0.9244439,0.9448515,0.9282371,0.9232843,0.9494932,0.9255068,0.8820696,0.9022216,0.8518002,0.9122573,0.9208273,0.9283533,0.9065739,0.9254601,0.9227866,0.8811454,0.9348501,0.9121565,0.9213690,0.9246502,0.9383525,0.9382666,0.8753230,0.9261571,0.9197807,0.9200348,0.9174449,0.9343093,0.9378364,0.9106571,0.9147105,0.8722027,0.9350674,0.9253482,0.9238291,0.9321363,0.9235855,0.9308647,0.9186514,0.9198529,0.8954409,0.8992164,0.9386578,0.9334359,0.8768597,0.9327527,0.9044680,0.9269888,0.9123668,0.9339605,0.9062746,0.9074720,0.9238560,0.8999862,0.9389371,0.9352603,0.9009087,0.9250155,0.9061627,0.9133150,0.9289918,0.9217496,0.9373961,0.9142899,0.8980073,0.9414079,0.9395200,0.9298662,0.9310802,0.9241497,0.9206831,0.8782148,0.9327581,0.9108735,0.8917666,0.9324499,0.8969422,0.9110415,0.9200465,0.9234361,0.9422954,0.9137205,0.9181811,0.9427479,0.9257140,0.9251069,0.8719853,0.9373800,0.9202188,0.9395530,0.9304480,0.9099837,0.9382369,0.9161383,0.9293015,0.8929643,0.9247588,0.8957800,0.9420169,0.8985383,0.9131422,0.9322500,0.9111658,0.9449195,0.9276287,0.9443971,0.9052003,0.9092360,0.8687232,0.9320723,0.9323760,0.9408281,0.9260711,0.9491575,0.9425342,0.8992298,0.9202666,0.9286401,0.9370169,0.9427503,0.9279293,0.9351425,0.8907237,0.9107118,0.9343792,0.9336630,0.9043975,0.9389478,0.9233588,0.9316899,0.9408324,0.8890681,0.9359337,0.9358816,0.9323545,0.9366391,0.9373185,0.9281011,0.9336614,0.9409030,0.9120501,0.9203660,0.9246119,0.9432655,0.8940876,0.9384107,0.9167976,0.9468759,0.9359940,0.9339890,0.9190962,0.9243823,0.9400322,0.9186402,0.9260134,0.9338660,0.9166315,0.9349257,0.9326959,0.9321748,0.9359495,0.9354319,0.9283705,0.9385225,0.9129104,0.9207670,0.9370412,0.9338474,0.9351244,0.9386802,0.9390187,0.8723265,0.9252733,0.9351331,0.9098250,0.9414334,0.9146566,0.9272349,0.9332654,0.9409805,0.9459354,0.9091468,0.9072008,0.9398218,0.9312090,0.9294152,0.8808358,0.9424087,0.9226929,0.9443880,0.9188889,0.9095976,0.9426499,0.9246820,0.9422396,0.9007756,0.9360970,0.9041744,0.9478584,0.9034469,0.9116500,0.9270007,0.9133364,0.9366736,0.9262409,0.9463014,0.9056786,0.9271301,0.8739311,0.9404935,0.9435939,0.9386587,0.9361855,0.9491557,0.9470759,0.8980433,0.9216194,0.9361442,0.9454451,0.9464234,0.9197628,0.9301545,0.8992784,0.9061615,0.9401481,0.9420487,0.8973299,0.9389367,0.9196725,0.9358348,0.9399091,0.8962498,0.9370460,0.9410434,0.9525227,0.9316879,0.9410222,0.9210765,0.9387181,0.9450978,0.9261133,0.9237765,0.9259800,0.9397740,0.9020899,0.9398753,0.9182257,0.9348752,0.9372509,0.9293320,0.9290183,0.9349412,0.9464924,0.9230535,0.9276099,0.9296228,0.9207168,0.9480022,0.9319471,0.9393123,0.9414392,0.9458188,0.9193311,0.9407386,0.9251507,0.9209012,0.9254705,0.9349912,0.9400853,0.9407683,0.9421662,0.8606078,0.9262697,0.9369150,0.9157463,0.9423511,0.9184321,0.9347267,0.9437126,0.9451571,0.9218133,0.8802896,0.8861355,0.9139612,0.9341028,0.9318430,0.8980960,0.9174428,0.9395046,0.9216755,0.9023857,0.8773729,0.9251288,0.9409604,0.9234364,0.9261800,0.9281217,0.9279422,0.9119982,0.9237981,0.8819394,0.8968082,0.8759707,0.9110720,0.8980367,0.9192100,0.9323663,0.9370677,0.8946280,0.9277415,0.9295020,0.9087919,0.9328743,0.9195440,0.9194929,0.9245858,0.8884546,0.9291320,0.9221936,0.9170254,0.8966128,0.9023370,0.9205694,0.8800294,0.9203925,0.9221448,0.8671251,0.9106153,0.8900697,0.9343524,0.9314979,0.9208713,0.9049165,0.9152465,0.9198250,0.9080422,0.9272554,0.8967383,0.9220709,0.9093528,0.9385924,0.9446763,0.8929767,0.9141163,0.9179307,0.9070019,0.9290302,0.9108628,0.9280704,0.9000651,0.9401710,0.9405186,0.9217237,0.9351798,0.8945398,0.9004338,0.9445893,0.9171890,0.9273294,0.9295232,0.9118264,0.9187232,0.8942357,0.9245913,0.9341781,0.8928744,0.9024806,0.9002530,0.9090126,0.9088259,0.9163371,0.8338273,0.8917656,0.9323722,0.9304359,0.9118465,0.9495223,0.9338359,0.9210423,0.9240461,0.9396136,0.8996174,0.8962514,0.9235313,0.9418782,0.9338578,0.8973136,0.9327882,0.9288082,0.9268581,0.9115508,0.8968061,0.9411400,0.9321197,0.9350837,0.9118971,0.9410840,0.9183893,0.9230694,0.9130851,0.9006029,0.9154076,0.9018515,0.9201812,0.9169994,0.9302340,0.9168931,0.9355468,0.8850258,0.9429605,0.9422144,0.9240195,0.9489265,0.9348474,0.9352947,0.9157304,0.9117124,0.9402049,0.9368273,0.9313006,0.9085346,0.9246409,0.9122135,0.8979190,0.9383665,0.9439698,0.8860227,0.9261605,0.9124334,0.9477488,0.9420728,0.9044149,0.9177864,0.9348235,0.9400330,0.9178674,0.9369091,0.9109977,0.9372868,0.9284960,0.9357287,0.9408359,0.9101725,0.9246164,0.9151706,0.9202934,0.9347369,0.9275907,0.9353386,0.9165370,0.9438939,0.9514221,0.9371434,0.9402611,0.9094133,0.9149157,0.9350007,0.9318964,0.9366164,0.9411015,0.9276304,0.9345333,0.9143659,0.9390197,0.9297172,0.9038998,0.9128532,0.9176216,0.9262048,0.9283585,0.9318580,0.8504507,0.9147941,0.9393964,0.9283540,0.9244296,0.9306226,0.9369198,0.9359983,0.9333758,0.9434845,0.9101097,0.9042795,0.9363129,0.9331644,0.9278314,0.8838073,0.9486789,0.9277779,0.9415668,0.9218659,0.9090480,0.9484652,0.9225290,0.9383832,0.9057968,0.9397087,0.9088376,0.9384185,0.9043127,0.9066432,0.9234493,0.9114984,0.9337959,0.9283644,0.9414011,0.9121085,0.9252934,0.8784755,0.9429074,0.9405871,0.9357136,0.9367360,0.9432711,0.9430201,0.9065788,0.9141401,0.9392618,0.9384314,0.9409098,0.9186345,0.9278458,0.9008648,0.9080342,0.9421537,0.9476997,0.8947775,0.9337178,0.9166208,0.9327535,0.9436402,0.9029338,0.9311579,0.9461444,0.9493231,0.9278566,0.9455643,0.9248743,0.9399021,0.9401419,0.9289834,0.9289719,0.9210292,0.9392765,0.9013662,0.9296651,0.9291445,0.9342571,0.9380084,0.9242539,0.9342668,0.9338877,0.9462003,0.9242756,0.9221828,0.9274428,0.9252812,0.9481677,0.9343191,0.9417698,0.9353563,0.9451102,0.9154515,0.9430799,0.9265577,0.9204322,0.9247816,0.9306514,0.9352484,0.9418322,0.9344879,0.8558786,0.9214063,0.9380357,0.9220570,0.9322458,0.9249384,0.9371592,0.9443852,0.9472793,0.8741205,0.8336979,0.8328072,0.8683808,0.9013168,0.8942672,0.9378963,0.8755384,0.8924222,0.8736763,0.8505060,0.8303356,0.8819633,0.8969752,0.8823430,0.9215064,0.8827795,0.9201077,0.8731627,0.9166554,0.8324890,0.8534909,0.8321651,0.8667741,0.8543276,0.8700619,0.9172386,0.9027870,0.9371767,0.8833664,0.8829048,0.8677088,0.8973923,0.8710517,0.8741128,0.9142869,0.8439134,0.8875401,0.8793266,0.8754905,0.8482347,0.8593071,0.9213640,0.8346598,0.8748794,0.8808538,0.8196354,0.8652231,0.8477161,0.8887367,0.8834684,0.9258061,0.8593522,0.8781351,0.8724327,0.8565449,0.8851105,0.8577885,0.8727358,0.8715301,0.8982483,0.9067382,0.8538822,0.8635818,0.9279371,0.8640426,0.8905564,0.8660755,0.8751353,0.8570938,0.8986939,0.8969312,0.8797642,0.8962510,0.8527394,0.8533318,0.8984422,0.8797901,0.8832008,0.8815775,0.8681522,0.8754348,0.8459093,0.8836896,0.9064521,0.8523828,0.8526655,0.8593705,0.8622445,0.8688480,0.8701963,0.7801782,0.8485134,0.8876925,0.9092058,0.8646663,0.9032609,0.8880502,0.8846885,0.8799469,0.8912355,0.8500828,0.8476193,0.8863285,0.9142301,0.9091007,0.9355925,0.8898723,0.9132054,0.8868437,0.8677730,0.8513085,0.8973414,0.9177476,0.9031200,0.9383447,0.9003069,0.9318099,0.8866511,0.9356681,0.8527769,0.8719368,0.8523077,0.8792979,0.8718654,0.8862245,0.9307309,0.9281917,0.9253732,0.9029432,0.8987749,0.8838678,0.9080422,0.8856205,0.8927086,0.9267180,0.8654995,0.8986571,0.8967758,0.8966171,0.8640674,0.8792612,0.9428409,0.8525015,0.8952506,0.9034888,0.8366641,0.8800289,0.8624394,0.9025804,0.9035601,0.9376971,0.8774104,0.8930137,0.8968812,0.8758797,0.8996260,0.8674359,0.8963362,0.8918767,0.9216799,0.9166452,0.8683460,0.8826148,0.9425148,0.8774257,0.9065304,0.8792390,0.8956898,0.8718545,0.9206300,0.9107606,0.8972676,0.9143719,0.8659032,0.8726376,0.9184906,0.8927522,0.9028588,0.9027895,0.8847358,0.8938125,0.8641593,0.8970584,0.9226364,0.8718447,0.8699555,0.8767475,0.8816434,0.8835839,0.8915385,0.7996584,0.8654609,0.9037191,0.9282443,0.8796985,0.9173524,0.8981671,0.9034766,0.8940545,0.9176247,0.8832071,0.8859579,0.9211453,0.9404449,0.9348376,0.8986676,0.9234361,0.9438681,0.9256780,0.9031365,0.8886481,0.9292337,0.9415042,0.9242271,0.9209010,0.9337692,0.9272130,0.9108269,0.9258946,0.8880742,0.9033111,0.8805273,0.9115757,0.9048698,0.9250380,0.9316627,0.9395414,0.8949600,0.9278640,0.9350942,0.9129131,0.9368736,0.9163986,0.9183140,0.9276430,0.8876489,0.9328555,0.9289984,0.9146931,0.9023525,0.9091342,0.9199356,0.8827456,0.9241480,0.9264294,0.8693966,0.9120648,0.8906077,0.9305798,0.9343107,0.9181358,0.9077194,0.9254844,0.9210633,0.9089185,0.9192761,0.9054035,0.9148867,0.9159530,0.9390857,0.9483558,0.9042140,0.9181534,0.9177201,0.9049685,0.9379444,0.9107337,0.9317974,0.9031954,0.9368573,0.9363560,0.9229194,0.9436620,0.8974379,0.9043260,0.9378739,0.9192505,0.9301321,0.9290585,0.9226116,0.9223034,0.8964796,0.9237330,0.9376256,0.8993275,0.9060780,0.9037940,0.9105418,0.9093824,0.9159853,0.8391711,0.8927846,0.9277563,0.9371567,0.9191078,0.9427943,0.9369474,0.9304569,0.9240855,0.9174795,0.8766662,0.8755419,0.9093659,0.9319325,0.9285323,0.9203963,0.9199120,0.9350490,0.9171588,0.8940282,0.8779159,0.9212788,0.9395397,0.9211822,0.9340430,0.9244149,0.9394822,0.9146013,0.9306250,0.8785840,0.8969236,0.8767318,0.9087764,0.8955285,0.9145416,0.9387643,0.9446156,0.9136542,0.9265957,0.9241943,0.9090948,0.9352836,0.9196101,0.9127248,0.9381722,0.8842110,0.9288541,0.9153757,0.9210732,0.8944601,0.8999473,0.9289557,0.8815690,0.9149788,0.9260657,0.8634940,0.9079004,0.8913717,0.9318970,0.9283934,0.9305543,0.9012590,0.9159240,0.9183710,0.9013357,0.9264021,0.8982403,0.9171540,0.9136773,0.9417611,0.9467991,0.8952512,0.9077247,0.9288142,0.9038186,0.9313843,0.9111517,0.9249624,0.8972185,0.9365265,0.9429787,0.9161844,0.9336403,0.8897795,0.8979838,0.9467141,0.9178251,0.9270677,0.9285919,0.9096580,0.9150926,0.8918104,0.9183347,0.9461053,0.8912511,0.8958318,0.9042650,0.9016163,0.9079578,0.9136351,0.8263837,0.8934692,0.9246015,0.9444147,0.9079486,0.9435421,0.9251370,0.9245579,0.9238182,0.8854424,0.9191733,0.9188860,0.8890018,0.8618781,0.8622099,0.8133446,0.8841656,0.8569225,0.8772550,0.9054954,0.9252067,0.8781194,0.8529262,0.8693989,0.8346127,0.8681670,0.8402533,0.8862822,0.8354196,0.9233443,0.9036557,0.9184395,0.8903463,0.9044258,0.8870964,0.8435106,0.8508354,0.8085596,0.8790417,0.8726557,0.8897078,0.8713947,0.8925306,0.8879721,0.8403783,0.9083545,0.8692191,0.8857406,0.8860397,0.9140745,0.9001033,0.8345430,0.9161059,0.8835585,0.8781444,0.9365207,0.9000032,0.9106975,0.8771939,0.8756405,0.8232860,0.9011023,0.8799143,0.8774444,0.8993877,0.8768422,0.8994982,0.8763386,0.8838858,0.8565748,0.8596707,0.9103147,0.8971918,0.8318629,0.8935804,0.8664455,0.8942055,0.8778239,0.8974560,0.8595736,0.8683321,0.8793257,0.8601395,0.9101585,0.9028615,0.8558089,0.8819637,0.8724072,0.8728354,0.8939956,0.8847448,0.9131407,0.8770449,0.8509242,0.8952846,0.9005014,0.8948526,0.8970602,0.8921663,0.8865580,0.9262138,0.9082252,0.8766668,0.8494634,0.8929479,0.8580928,0.8702495,0.8786943,0.8820910,0.9320515,0.8901553,0.8863989,0.9194050,0.9468203,0.9391237,0.9020151,0.9335575,0.9447766,0.9316989,0.9015537,0.8881459,0.9387491,0.9367905,0.9332178,0.9201182,0.9403516,0.9316184,0.9217953,0.9227861,0.8893697,0.9094337,0.8954831,0.9127165,0.9069590,0.9222707,0.9323569,0.9368362,0.8933512,0.9388123,0.9331847,0.9207314,0.9464539,0.9253907,0.9292842,0.9255639,0.8987616,0.9329739,0.9321617,0.9255549,0.8998092,0.9137424,0.9192586,0.8923392,0.9311090,0.9376806,0.8786578,0.9171377,0.8995512,0.9434058,0.9351169,0.9171159,0.9086931,0.9265854,0.9299031,0.9157144,0.9357279,0.9087126,0.9283205,0.9221237,0.9409476,0.9421867,0.9049043,0.9147545,0.9249040,0.9154472,0.9342246,0.9254859,0.9417793,0.9080509,0.9485456,0.9483590,0.9305650,0.9463185,0.9013546,0.9128249,0.9409034,0.9273007,0.9365890,0.9364714,0.9245681,0.9283722,0.9031463,0.9334734,0.9344128,0.8984863,0.9104099,0.9124597,0.9214972,0.9253503,0.9163388,0.8418634,0.9086126,0.9356123,0.9339115,0.9158848,0.9439748,0.9404896,0.9340857,0.9334833,0.9346064,0.9121345,0.9086539,0.9394718,0.9195123,0.9129978,0.8743182,0.9275442,0.9145527,0.9348232,0.9240476,0.9113727,0.9352992,0.9147331,0.9273107,0.8897723,0.9269179,0.8954880,0.9483455,0.8967740,0.9155200,0.9323117,0.9118423,0.9347536,0.9256132,0.9353123,0.8990190,0.9119403,0.8601914,0.9280741,0.9248721,0.9361874,0.9205574,0.9359345,0.9356398,0.8909942,0.9159317,0.9209235,0.9308094,0.9381575,0.9237904,0.9275260,0.8904860,0.9135500,0.9376005,0.9314486,0.9036105,0.9397560,0.9234232,0.9250442,0.9277251,0.8816471,0.9378188,0.9372083,0.9346530,0.9318939,0.9359945,0.9291786,0.9287740,0.9340122,0.9105932,0.9136854,0.9290690,0.9370411,0.8914345,0.9395543,0.9045453,0.9441165,0.9251198,0.9289171,0.9220507,0.9185141,0.9261566,0.9074160,0.9234093,0.9267052,0.9102316,0.9371404,0.9219243,0.9283549,0.9390422,0.9269300,0.9261835,0.9323952,0.9056900,0.9209019,0.9322213,0.9375808,0.9398771,0.9410891,0.9348200,0.8689672,0.9214360,0.9271407,0.9049889,0.9372066,0.9076924,0.9229098,0.9280305,0.9318049,0.9269728,0.9261793,0.9270825,0.9369510,0.9074292,0.9040496,0.8641615,0.9263593,0.9018056,0.9193424,0.9394374,0.9276085,0.9316987,0.9060297,0.9114875,0.8773060,0.9158914,0.8871692,0.9323006,0.8857441,0.9241524,0.9405742,0.9285473,0.9251894,0.9467617,0.9322427,0.8895210,0.9020718,0.8531655,0.9225035,0.9216793,0.9341055,0.9170464,0.9222236,0.9341229,0.8937407,0.9309255,0.9159926,0.9257719,0.9285326,0.9381517,0.9402897,0.8776218,0.9250791,0.9236914,0.9230560,0.9152219,0.9435251,0.9342489,0.9133139,0.9255806,0.8803691,0.9408119,0.9320701,0.9254165,0.9286269,0.9251639,0.9398190,0.9229586,0.9263377,0.9046821,0.9092819,0.9432533,0.9385540,0.8818404,0.9372100,0.9081165,0.9431797,0.9165873,0.9463185,0.9089889,0.9146683,0.9221323,0.9095112,0.9394937,0.9376158,0.9026378,0.9258729,0.9064371,0.9164469,0.9367504,0.9285952,0.9376993,0.9210987,0.8967720,0.9457148,0.9398171,0.9302532,0.9423512,0.9313559,0.9242923,0.8795057,0.9407352,0.9145696,0.8983551,0.9325046,0.8999311,0.9192859,0.9247309,0.9249736,0.9308514,0.8847887,0.8895204,0.9212386,0.9411727,0.9384327,0.9045855,0.9328628,0.9402768,0.9254650,0.9045762,0.8882758,0.9359660,0.9420175,0.9308937,0.9196249,0.9413675,0.9265358,0.9243892,0.9257695,0.8878381,0.9101529,0.8968257,0.9135460,0.9070653,0.9211886,0.9227184,0.9415837,0.8905792,0.9422024,0.9273616,0.9206364,0.9404942,0.9249159,0.9307487,0.9253977,0.9038338,0.9299742,0.9373128,0.9315821,0.8998634,0.9113385,0.9185279,0.8945547,0.9306706,0.9400295,0.8765142,0.9151690,0.9000357,0.9382994,0.9371047,0.9198241,0.9102492,0.9269303,0.9284875,0.9136501,0.9351480,0.9121757,0.9291981,0.9267699,0.9428953,0.9403032,0.9082884,0.9171358,0.9233384,0.9135259,0.9364547,0.9235567,0.9291778,0.9115466,0.9463316,0.9487481,0.9275061,0.9413851,0.8934521,0.9083998,0.9362936,0.9335700,0.9365391,0.9416230,0.9248145,0.9275335,0.9003812,0.9316669,0.9317277,0.8997571,0.9113569,0.9126677,0.9211419,0.9247162,0.9169933,0.8402195,0.9097989,0.9352819,0.9343122,0.9121791,0.9348943,0.9362241,0.9310200,0.9312374,0.9453864,0.9034279,0.9042822,0.9310513,0.9318467,0.9240834,0.8817822,0.9410674,0.9326924,0.9427909,0.9242050,0.9039851,0.9454487,0.9260340,0.9344527,0.8989251,0.9339280,0.9049067,0.9337649,0.9029747,0.9001225,0.9271501,0.9089651,0.9346809,0.9260069,0.9413483,0.9119982,0.9293353,0.8718031,0.9443110,0.9406413,0.9307037,0.9314412,0.9422360,0.9408533,0.9039326,0.9118492,0.9408624,0.9447933,0.9401925,0.9161573,0.9325916,0.8943460,0.9080729,0.9382787,0.9418281,0.8915312,0.9343764,0.9222360,0.9333671,0.9432291,0.9011924,0.9237882,0.9397104,0.9446269,0.9330478,0.9415439,0.9252904,0.9389516,0.9351766,0.9243542,0.9253354,0.9207426,0.9340253,0.8993402,0.9276960,0.9248966,0.9369386,0.9463692,0.9208036,0.9284444,0.9329201,0.9453028,0.9323265,0.9148862,0.9274508,0.9238799,0.9471253,0.9416883,0.9407253,0.9327371,0.9394859,0.9176568,0.9414992,0.9235192,0.9150087,0.9203960,0.9321845,0.9349531,0.9360768,0.9326312,0.8618046,0.9153448,0.9329136,0.9177806,0.9371411,0.9201963,0.9350442,0.9402273,0.9468540,0.9094499,0.9476427,0.9485295,0.9183147,0.8905761,0.8871237,0.8434539,0.9086565,0.8885885,0.9065372,0.9324220,0.9508319,0.9118329,0.8910423,0.8965741,0.8605270,0.9005988,0.8635861,0.9132759,0.8654613,0.9459222,0.9344367,0.9432795,0.9120419,0.9327435,0.9162984,0.8704605,0.8787031,0.8333919,0.9042727,0.9049544,0.9164154,0.8972069,0.9137852,0.9146731,0.8666403,0.9323679,0.8970252,0.9116070,0.9150630,0.9429826,0.9296658,0.8553153,0.9517432,0.9128519,0.9102281,0.9319160,0.9306909,0.9361938,0.8975581,0.9038861,0.8564325,0.9266779,0.9121289,0.9072498,0.9275268,0.9075639,0.9272436,0.9014799,0.9094916,0.8845556,0.8904979,0.9372730,0.9238751,0.8588837,0.9189545,0.8892584,0.9225139,0.9044946,0.9240088,0.8923425,0.8941475,0.9126129,0.8867178,0.9316034,0.9326198,0.8822779,0.9075169,0.9008785,0.9088153,0.9131477,0.9091966,0.9387623,0.9013643,0.8815985,0.9245058,0.9234425,0.9220786,0.9273135,0.9161492,0.9146901,0.9014523,0.9345098,0.9027381,0.8771122,0.9239110,0.8827284,0.8964911,0.9045673,0.9103104,0.9135596,0.8717515,0.8715946,0.9052831,0.9370939,0.9295535,0.9248228,0.9118955,0.9339493,0.9131163,0.8830840,0.8689488,0.9212707,0.9349987,0.9253050,0.9372131,0.9242399,0.9479213,0.9081133,0.9451351,0.8748241,0.8872192,0.8772089,0.8944001,0.8919924,0.9109522,0.9389063,0.9455518,0.9069257,0.9224006,0.9216830,0.9029880,0.9270250,0.9098357,0.9103604,0.9415043,0.8893703,0.9215932,0.9153658,0.9119630,0.8853416,0.8998224,0.9358783,0.8776656,0.9128160,0.9217703,0.8583765,0.8977981,0.8846657,0.9262822,0.9214856,0.9379889,0.8973698,0.9092437,0.9145256,0.8939189,0.9187268,0.8910685,0.9126786,0.9105583,0.9424716,0.9401076,0.8916589,0.9018435,0.9394959,0.8914043,0.9302434,0.9010719,0.9225887,0.8886569,0.9324286,0.9303962,0.9146889,0.9312635,0.8844640,0.8914523,0.9412976,0.9134663,0.9187503,0.9235298,0.9029948,0.9144026,0.8808863,0.9154349,0.9451179,0.8882044,0.8877787,0.8953852,0.9012943,0.9019900,0.9070952,0.8177226,0.8860353,0.9149604,0.9487251,0.9035157,0.9386631,0.9241952,0.9180285,0.9134017,0.9097012,0.8640152,0.8625391,0.9005944,0.9305285,0.9211892,0.9184607,0.9093316,0.9328623,0.9074423,0.8810238,0.8675581,0.9176000,0.9277851,0.9202459,0.9421564,0.9209384,0.9440680,0.9044289,0.9398772,0.8703112,0.8859023,0.8735346,0.8977539,0.8904429,0.9063154,0.9363384,0.9389804,0.9104203,0.9193110,0.9147201,0.8981347,0.9284082,0.9042005,0.9061024,0.9352190,0.8859226,0.9129372,0.9080364,0.9139562,0.8781004,0.8930101,0.9370250,0.8727744,0.9114267,0.9186706,0.8574259,0.8955622,0.8807652,0.9214536,0.9169432,0.9329699,0.8904850,0.9082059,0.9098503,0.8905523,0.9199783,0.8901802,0.9106919,0.9053346,0.9382676,0.9347859,0.8859509,0.9003933,0.9340449,0.8912161,0.9266469,0.8938601,0.9128384,0.8860083,0.9304250,0.9274563,0.9087778,0.9240321,0.8825112,0.8889688,0.9334193,0.9071346,0.9212348,0.9150374,0.9032411,0.9116542,0.8775492,0.9165240,0.9364492,0.8873229,0.8862670,0.8983499,0.8963027,0.9036982,0.9036112,0.8167653,0.8851194,0.9173806,0.9448495,0.8964723,0.9323762,0.9155482,0.9183542,0.9108880,0.9058994,0.9352322,0.9406311,0.9096361,0.8859685,0.8779096,0.8366413,0.9036765,0.8813278,0.8985390,0.9302401,0.9356518,0.9066631,0.8804291,0.8924787,0.8536566,0.8904017,0.8585847,0.9099540,0.8576614,0.9431917,0.9248941,0.9366550,0.9105956,0.9244313,0.9081411,0.8607914,0.8775630,0.8275556,0.8942802,0.8902504,0.9162702,0.8886022,0.9041861,0.9035149,0.8629695,0.9315688,0.8977392,0.9017207,0.9088523,0.9327584,0.9189051,0.8498201,0.9328775,0.8996846,0.8978005,0.9424282,0.9214120,0.9300271,0.8955888,0.9017771,0.8497448,0.9208976,0.9033149,0.9059682,0.9129402,0.9004998,0.9200621,0.9045060,0.9029841,0.8789631,0.8789583,0.9296036,0.9197534,0.8530611,0.9037977,0.8835348,0.9153206,0.8932419,0.9132998,0.8838455,0.8892554,0.9037046,0.8840057,0.9301576,0.9214145,0.8773546,0.9059608,0.8863201,0.8944062,0.9077119,0.8979492,0.9313235,0.8985027,0.8800406,0.9157226,0.9137570,0.9098830,0.9192796,0.9082348,0.9027407,0.9053051,0.9245102,0.8912604,0.8718108,0.9161198,0.8751275,0.8921680,0.8995470,0.9040899,0.9234818,0.9328041,0.9313217,0.9344290,0.9069412,0.9070045,0.8563511,0.9210145,0.9031354,0.9190640,0.9504892,0.9320138,0.9211778,0.9030569,0.9104123,0.8771933,0.9116321,0.8775078,0.9248199,0.8759961,0.9317195,0.9442470,0.9329433,0.9282339,0.9488625,0.9307866,0.8829776,0.8964386,0.8492236,0.9196909,0.9192294,0.9381797,0.9143875,0.9215517,0.9289737,0.8835018,0.9381906,0.9181854,0.9260903,0.9262242,0.9467178,0.9451620,0.8732600,0.9289435,0.9194066,0.9221723,0.9242014,0.9441922,0.9481680,0.9167399,0.9226961,0.8734332,0.9378155,0.9259404,0.9247764,0.9344241,0.9204398,0.9372499,0.9228385,0.9275347,0.8987211,0.9034541,0.9476365,0.9383059,0.8706655,0.9347893,0.9091361,0.9363960,0.9191100,0.9414434,0.9060569,0.9065311,0.9214241,0.9047789,0.9445161,0.9450424,0.8994828,0.9218180,0.9099910,0.9216929,0.9308187,0.9202501,0.9423006,0.9194508,0.8981573,0.9392713,0.9398158,0.9284160,0.9387065,0.9254060,0.9230744,0.8888749,0.9361249,0.9113954,0.8966360,0.9349487,0.8992089,0.9094207,0.9163719,0.9235485,0.9414990,0.9016455,0.9019325,0.9328119,0.9393036,0.9283991,0.8907250,0.9364974,0.9309663,0.9395471,0.9191824,0.9024350,0.9453874,0.9248994,0.9397204,0.9091850,0.9402232,0.9158111,0.9355403,0.9136759,0.9040415,0.9211038,0.9005464,0.9292581,0.9190543,0.9344607,0.9192253,0.9271267,0.8785759,0.9431782,0.9453961,0.9264879,0.9408696,0.9388542,0.9424752,0.9110495,0.9077912,0.9407622,0.9417074,0.9398871,0.9157915,0.9274754,0.9060352,0.9038996,0.9441126,0.9449997,0.8887738,0.9343326,0.9109899,0.9434569,0.9470029,0.8984377,0.9237753,0.9399247,0.9361017,0.9247841,0.9459858,0.9214631,0.9364255,0.9339985,0.9209299,0.9378990,0.9179578,0.9267100,0.9108522,0.9244339,0.9271847,0.9338181,0.9459388,0.9188608,0.9279150,0.9373322,0.9373291,0.9354932,0.9142988,0.9209055,0.9282680,0.9401200,0.9369023,0.9389916,0.9379824,0.9357462,0.9142355,0.9451885,0.9237473,0.9141936,0.9165278,0.9281130,0.9265116,0.9363352,0.9359492,0.8542160,0.9177620,0.9427197,0.9191663,0.9279966,0.9299061,0.9461369,0.9422813,0.9451571,0.9334528,0.9012022,0.8968287,0.9239737,0.9417888,0.9392534,0.8959783,0.9317449,0.9335800,0.9361573,0.9128811,0.8933769,0.9374257,0.9286263,0.9323633,0.9094811,0.9468137,0.9177992,0.9289042,0.9127859,0.8967459,0.9129286,0.8946282,0.9268871,0.9122907,0.9313258,0.9187338,0.9332427,0.8851560,0.9396354,0.9372098,0.9227434,0.9480244,0.9368900,0.9310534,0.9129462,0.9030816,0.9382230,0.9354326,0.9317657,0.9060161,0.9147193,0.9079257,0.8949833,0.9312173,0.9425797,0.8784546,0.9234156,0.9040701,0.9444096,0.9440962,0.9066037,0.9145008,0.9301877,0.9293112,0.9179622,0.9358962,0.9059566,0.9279402,0.9309259,0.9318771,0.9442631,0.9159292,0.9225692,0.9154574,0.9179173,0.9329643,0.9279416,0.9393928,0.9130016,0.9353314,0.9484609,0.9318351,0.9366892,0.9072382,0.9176445,0.9357356,0.9301718,0.9444369,0.9423093,0.9253623,0.9311599,0.9095790,0.9380189,0.9274447,0.9023829,0.9090757,0.9139485,0.9216753,0.9276540,0.9289604,0.8470337,0.9141611,0.9451682,0.9249960,0.9245247,0.9320471,0.9439780,0.9349887,0.9357722,0.9359389,0.8922455,0.8921254,0.9292273,0.9483799,0.9353802,0.8999778,0.9343754,0.9389923,0.9321205,0.9092034,0.8924747,0.9407571,0.9390373,0.9380045,0.9153813,0.9393694,0.9241508,0.9296132,0.9226026,0.8982009,0.9160618,0.8931470,0.9231765,0.9140833,0.9303792,0.9237587,0.9399043,0.8856880,0.9377284,0.9450830,0.9229673,0.9452177,0.9292714,0.9302450,0.9220769,0.9047191,0.9426003,0.9393780,0.9331090,0.9076106,0.9213450,0.9171928,0.8923841,0.9406409,0.9412373,0.8790797,0.9240018,0.9021313,0.9467584,0.9434304,0.9079084,0.9254673,0.9328504,0.9333121,0.9193874,0.9379867,0.9129338,0.9301314,0.9264115,0.9372214,0.9385026,0.9124844,0.9232859,0.9192671,0.9168917,0.9394455,0.9272214,0.9392446,0.9140203,0.9472320,0.9485616,0.9358579,0.9397523,0.9104922,0.9135580,0.9407464,0.9340511,0.9438224,0.9367587,0.9259531,0.9320239,0.9104830,0.9378394,0.9336153,0.9171005,0.9123573,0.9241740,0.9227055,0.9237499,0.9308981,0.8490874,0.9053753,0.9413529,0.9315145,0.9285651,0.9341412,0.9437521,0.9436346,0.9365531,0.9298527,0.9294389,0.9297473,0.9322972,0.9080751,0.9060066,0.8595452,0.9156107,0.9087006,0.9276120,0.9396697,0.9317140,0.9254100,0.9083262,0.9088868,0.8782933,0.9179111,0.8820000,0.9287495,0.8830394,0.9289872,0.9480155,0.9288808,0.9339943,0.9488180,0.9310837,0.8877961,0.8973063,0.8541568,0.9194120,0.9205543,0.9370820,0.9154058,0.9281353,0.9298833,0.8839432,0.9342368,0.9160234,0.9233600,0.9245830,0.9379711,0.9476291,0.8753414,0.9283907,0.9296886,0.9230997,0.9171138,0.9416035,0.9357196,0.9133648,0.9213821,0.8671014,0.9379569,0.9288310,0.9244708,0.9403101,0.9225441,0.9375055,0.9217921,0.9358367,0.9003591,0.9051266,0.9433981,0.9389286,0.8762874,0.9366594,0.9102835,0.9357347,0.9197564,0.9417498,0.9064398,0.9124362,0.9267215,0.9053333,0.9386024,0.9473765,0.8976483,0.9259969,0.9162628,0.9217231,0.9317810,0.9259616,0.9402968,0.9203978,0.8989244,0.9388106,0.9363076,0.9334329,0.9379598,0.9237820,0.9290080,0.8836072,0.9427098,0.9191959,0.8966953,0.9399568,0.9026450,0.9163951,0.9244698,0.9281466,0.9311961,0.8924302,0.8990287,0.9259961,0.9405366,0.9334138,0.9029986,0.9404430,0.9367351,0.9337278,0.9092803,0.8964890,0.9407730,0.9367828,0.9329992,0.9191805,0.9404116,0.9208896,0.9238520,0.9182832,0.8920810,0.9134281,0.9024628,0.9214079,0.9102285,0.9266343,0.9258397,0.9407206,0.8962546,0.9422794,0.9377900,0.9279733,0.9520047,0.9291224,0.9240771,0.9244582,0.9005616,0.9449333,0.9366853,0.9267153,0.9078934,0.9166928,0.9131134,0.9026129,0.9341998,0.9447035,0.8839374,0.9208203,0.9050589,0.9429981,0.9374779,0.9086383,0.9160387,0.9357810,0.9378639,0.9195583,0.9392035,0.9105324,0.9290232,0.9283443,0.9367357,0.9410310,0.9125513,0.9218991,0.9126762,0.9152448,0.9412258,0.9239093,0.9366050,0.9142974,0.9464231,0.9471746,0.9366412,0.9437351,0.9055687,0.9122020,0.9401408,0.9299355,0.9382808,0.9376005,0.9230031,0.9338226,0.9098583,0.9287016,0.9299475,0.9093162,0.9115167,0.9186133,0.9224863,0.9306852,0.9278794,0.8485279,0.9128351,0.9393390,0.9312187,0.9246304,0.9390993,0.9421373,0.9384363,0.9369336,0.9414420,0.9086774,0.9037114,0.9416715,0.9353087,0.9330541,0.8870429,0.9451621,0.9281576,0.9457999,0.9221537,0.9084202,0.9487236,0.9285785,0.9372554,0.9084589,0.9376423,0.9131951,0.9335904,0.9040492,0.9091923,0.9259695,0.9083699,0.9351763,0.9306531,0.9460725,0.9144089,0.9281931,0.8818255,0.9436276,0.9508324,0.9372619,0.9416916,0.9432222,0.9416429,0.9067556,0.9129208,0.9439504,0.9516572,0.9436638,0.9186805,0.9353799,0.9070676,0.9074319,0.9458397,0.9520296,0.8889727,0.9400533,0.9182332,0.9397150,0.9478493,0.8984624,0.9295023,0.9478071,0.9496495,0.9353598,0.9481057,0.9228144,0.9389598,0.9446996,0.9281091,0.9347424,0.9210778,0.9369620,0.9056490,0.9314483,0.9361781,0.9369047,0.9440172,0.9274441,0.9376912,0.9400148,0.9476441,0.9361521,0.9235088,0.9279209,0.9288069,0.9456476,0.9394512,0.9410914,0.9365306,0.9484983,0.9198796,0.9488176,0.9263821,0.9232828,0.9196245,0.9316792,0.9341599,0.9380997,0.9429061,0.8587485,0.9272209,0.9474871,0.9232315,0.9369149,0.9312777,0.9419931,0.9478616,0.9472752,0.9262850,0.9297030,0.9310887,0.9305142,0.9123370,0.9048066,0.8590421,0.9208863,0.9091375,0.9192252,0.9416992,0.9317495,0.9230389,0.9040631,0.9115198,0.8761973,0.9161211,0.8833197,0.9286182,0.8809413,0.9248387,0.9437764,0.9248247,0.9367977,0.9496779,0.9335739,0.8896449,0.8975399,0.8516166,0.9221594,0.9199779,0.9349007,0.9108717,0.9319344,0.9290750,0.8872368,0.9312245,0.9157193,0.9243179,0.9358432,0.9399462,0.9453554,0.8790301,0.9318084,0.9251232,0.9254026,0.9162988,0.9419594,0.9379690,0.9161383,0.9228083,0.8711867,0.9453087,0.9233050,0.9251699,0.9367309,0.9256358,0.9425078,0.9225380,0.9271767,0.9011442,0.9049404,0.9418788,0.9411379,0.8747806,0.9346803,0.9141988,0.9390698,0.9223418,0.9359640,0.9045739,0.9119322,0.9186884,0.9059493,0.9440649,0.9503649,0.8990760,0.9203263,0.9154001,0.9199612,0.9324365,0.9248064,0.9403453,0.9269667,0.8983539,0.9371847,0.9365392,0.9358383,0.9349823,0.9256338,0.9305356,0.8829424,0.9366013,0.9220901,0.8971830,0.9442372,0.8986316,0.9128228,0.9262181,0.9267896,0.9198071,0.8863097,0.8797610,0.9168963,0.9389208,0.9357037,0.9074887,0.9228044,0.9418428,0.9187318,0.8976157,0.8845596,0.9266941,0.9363854,0.9266102,0.9265536,0.9278425,0.9269260,0.9161383,0.9201944,0.8839322,0.8977269,0.8853046,0.9068981,0.8983675,0.9210525,0.9267841,0.9365744,0.8938024,0.9294679,0.9217898,0.9201939,0.9398932,0.9149194,0.9239724,0.9266060,0.8928048,0.9342544,0.9224694,0.9216689,0.8979600,0.9035621,0.9219757,0.8921191,0.9198001,0.9309840,0.8732518,0.9083557,0.8923952,0.9314436,0.9291644,0.9252810,0.9075415,0.9213373,0.9228305,0.9027882,0.9238936,0.8999236,0.9233332,0.9150379,0.9414955,0.9517117,0.9014114,0.9133064,0.9201683,0.9098693,0.9341772,0.9165498,0.9270843,0.9046783,0.9416146,0.9400281,0.9259096,0.9339546,0.8988983,0.9038891,0.9426999,0.9189867,0.9289592,0.9322660,0.9166152,0.9206658,0.8915358,0.9222495,0.9438931,0.8942701,0.8975970,0.9007238,0.9065432,0.9143947,0.9107723,0.8301932,0.8943107,0.9309165,0.9349708,0.9120474,0.9385409,0.9264743,0.9269457,0.9268550,0.9405421,0.9175643,0.9203977,0.9465609,0.9228812,0.9162027,0.8718556,0.9309983,0.9173597,0.9283923,0.9371013,0.9183451,0.9391148,0.9090915,0.9210898,0.8902026,0.9226086,0.8955450,0.9396578,0.8918884,0.9204095,0.9359636,0.9157568,0.9445656,0.9380423,0.9378385,0.8947842,0.9136829,0.8587345,0.9373405,0.9254555,0.9472031,0.9263893,0.9401312,0.9396554,0.8952279,0.9304420,0.9268004,0.9404765,0.9436362,0.9305405,0.9369101,0.8898765,0.9209218,0.9373105,0.9375529,0.9078618,0.9417146,0.9273201,0.9239732,0.9309884,0.8820780,0.9404850,0.9319614,0.9388654,0.9376657,0.9322382,0.9352977,0.9318638,0.9371956,0.9130342,0.9168190,0.9324413,0.9433270,0.8910120,0.9380865,0.9182161,0.9492254,0.9261861,0.9395972,0.9237910,0.9215492,0.9304785,0.9167369,0.9303083,0.9356227,0.9101174,0.9378021,0.9194841,0.9277898,0.9436260,0.9386491,0.9302770,0.9390503,0.9087166,0.9270855,0.9366403,0.9387385,0.9414514,0.9397258,0.9327829,0.8760036,0.9335995,0.9308581,0.9075856,0.9487789,0.9092544,0.9276532,0.9336417,0.9337651,0.9319447,0.8907791,0.8952968,0.9259772,0.9474005,0.9403880,0.8998093,0.9325909,0.9384267,0.9360740,0.9144782,0.8950744,0.9411677,0.9403296,0.9331966,0.9202572,0.9389401,0.9188530,0.9248374,0.9155390,0.8911351,0.9112905,0.8924759,0.9227921,0.9109338,0.9291688,0.9305732,0.9421815,0.8927528,0.9409464,0.9397357,0.9216153,0.9483382,0.9305560,0.9268713,0.9213926,0.8985253,0.9395045,0.9362846,0.9330002,0.9064556,0.9202049,0.9104112,0.8951707,0.9317584,0.9455428,0.8801001,0.9275841,0.9082991,0.9471731,0.9450443,0.9129661,0.9164175,0.9326203,0.9342728,0.9193590,0.9363165,0.9095270,0.9314222,0.9315632,0.9349324,0.9419277,0.9123319,0.9228605,0.9175364,0.9160418,0.9379481,0.9313666,0.9384179,0.9088239,0.9477766,0.9482472,0.9398779,0.9422662,0.9053363,0.9165676,0.9444148,0.9323509,0.9457140,0.9463846,0.9207409,0.9317861,0.9088460,0.9312839,0.9347632,0.9091470,0.9136799,0.9205980,0.9235616,0.9269717,0.9293259,0.8481895,0.9119558,0.9437543,0.9334298,0.9265292,0.9371347,0.9385115,0.9358772,0.9332331,0.9389823,0.9005464,0.8969395,0.9295573,0.9310913,0.9260053,0.8827762,0.9444592,0.9205034,0.9382110,0.9171946,0.9001130,0.9446882,0.9308353,0.9372201,0.9037891,0.9390515,0.9090904,0.9326472,0.9101853,0.9021635,0.9181783,0.9068595,0.9330078,0.9196293,0.9345300,0.9107300,0.9301099,0.8789435,0.9466093,0.9411332,0.9281025,0.9412834,0.9365483,0.9313111,0.9082247,0.9122782,0.9317601,0.9426133,0.9382428,0.9101236,0.9236676,0.8998284,0.9035745,0.9371143,0.9397037,0.8885258,0.9267054,0.9153058,0.9332505,0.9358700,0.9017200,0.9206650,0.9382196,0.9428744,0.9193965,0.9460213,0.9160089,0.9356055,0.9431392,0.9205122,0.9280434,0.9227227,0.9225047,0.9029448,0.9212718,0.9282422,0.9311121,0.9422376,0.9219033,0.9320939,0.9310844,0.9381068,0.9287523,0.9152169,0.9249965,0.9172809,0.9414600,0.9338607,0.9425275,0.9305540,0.9389519,0.9119305,0.9351670,0.9198046,0.9187372,0.9157485,0.9225778,0.9266584,0.9354760,0.9352077,0.8541530,0.9133167,0.9353824,0.9184188,0.9318729,0.9224009,0.9383934,0.9470861,0.9386996,0.8962247,0.8550228,0.8519963,0.8899376,0.9172214,0.9176901,0.9283196,0.8943964,0.9214358,0.8975885,0.8693222,0.8546563,0.9022641,0.9159225,0.8987325,0.9398114,0.9101640,0.9407716,0.8858262,0.9370181,0.8564681,0.8747939,0.8490658,0.8868354,0.8788875,0.8945245,0.9327530,0.9283312,0.9211091,0.9086812,0.8997323,0.8841786,0.9128619,0.8929996,0.8982417,0.9350461,0.8660832,0.9038981,0.9000306,0.8890286,0.8670077,0.8746625,0.9409561,0.8520406,0.8919985,0.9048730,0.8433618,0.8805191,0.8636045,0.9038010,0.9023337,0.9434552,0.8788990,0.8927260,0.8948767,0.8754942,0.9014378,0.8681889,0.8935816,0.8849625,0.9208982,0.9232843,0.8715749,0.8820323,0.9407211,0.8800948,0.9131071,0.8834555,0.9000132,0.8783719,0.9194339,0.9123758,0.8988560,0.9149463,0.8728946,0.8765540,0.9210840,0.8928635,0.9031307,0.8996683,0.8875811,0.8964605,0.8643097,0.9059545,0.9245375,0.8746781,0.8747230,0.8768163,0.8877175,0.8891209,0.8913277,0.8009505,0.8670886,0.9076881,0.9280430,0.8813076,0.9240698,0.9062981,0.9047717,0.8947867,0.9046592,0.8608145,0.8595820,0.8935078,0.9207027,0.9129245,0.9251303,0.9046157,0.9200366,0.8982054,0.8785004,0.8648180,0.9108951,0.9264270,0.9046574,0.9420665,0.9122709,0.9477888,0.8953583,0.9456847,0.8613405,0.8795249,0.8684295,0.8845638,0.8855835,0.9000980,0.9420977,0.9253305,0.9213302,0.9088284,0.9145428,0.8917522,0.9203358,0.8966819,0.9001517,0.9472320,0.8740202,0.9104900,0.8986307,0.8969320,0.8758635,0.8873909,0.9392582,0.8661109,0.9022292,0.9136086,0.8535736,0.8866477,0.8743346,0.9119711,0.9068980,0.9419690,0.8872177,0.9050331,0.9010157,0.8812292,0.9068343,0.8826329,0.8988113,0.8935914,0.9288647,0.9261284,0.8797167,0.8923758,0.9359714,0.8860452,0.9209123,0.8889466,0.9073233,0.8798058,0.9221713,0.9203365,0.9055328,0.9195867,0.8737514,0.8790637,0.9297145,0.8952439,0.9159806,0.9117399,0.8962840,0.9003026,0.8700175,0.8979203,0.9360321,0.8842277,0.8794683,0.8817612,0.8888984,0.8926424,0.8939837,0.8079224,0.8767627,0.9092387,0.9388829,0.8891087,0.9264993,0.9119256,0.9130618,0.9036399,0.9319601,0.8853934,0.8836372,0.9146996,0.9507707,0.9376409,0.9047886,0.9257982,0.9403551,0.9229379,0.9009891,0.8855833,0.9370501,0.9450511,0.9333750,0.9223669,0.9337833,0.9269653,0.9218026,0.9312358,0.8905540,0.9088239,0.8934702,0.9124390,0.9057476,0.9210964,0.9278980,0.9438692,0.8955772,0.9352674,0.9304951,0.9211944,0.9388420,0.9210160,0.9273643,0.9293469,0.8990557,0.9371977,0.9367215,0.9241769,0.9001663,0.9134917,0.9242843,0.8888792,0.9296077,0.9333656,0.8743657,0.9173542,0.9003817,0.9374314,0.9313334,0.9172697,0.9099020,0.9252490,0.9314634,0.9135018,0.9354451,0.9084681,0.9275650,0.9210198,0.9390557,0.9480351,0.9018382,0.9187187,0.9265788,0.9083677,0.9372914,0.9159429,0.9282739,0.9092089,0.9532030,0.9438935,0.9308887,0.9443745,0.9024598,0.9065401,0.9421947,0.9279273,0.9344892,0.9328758,0.9199186,0.9229165,0.9025265,0.9372172,0.9387377,0.8970955,0.9084265,0.9081706,0.9181099,0.9209108,0.9181579,0.8410049,0.8986361,0.9322074,0.9369722,0.9161750,0.9370587,0.9377977,0.9309369,0.9367355,0.9278496,0.8890275,0.8829462,0.9197221,0.9472200,0.9399195,0.9062140,0.9262636,0.9430941,0.9213996,0.9020059,0.8893718,0.9315905,0.9468616,0.9297048,0.9272884,0.9341609,0.9300614,0.9140421,0.9314444,0.8905287,0.9061333,0.8901905,0.9114259,0.9122849,0.9200370,0.9320425,0.9412580,0.8967115,0.9315203,0.9331156,0.9153544,0.9422047,0.9206233,0.9213108,0.9274058,0.8909718,0.9327890,0.9296601,0.9223879,0.8999367,0.9127804,0.9246329,0.8897727,0.9305917,0.9351654,0.8717679,0.9175439,0.8958630,0.9380401,0.9334589,0.9197505,0.9116982,0.9283207,0.9269146,0.9090463,0.9290303,0.9012117,0.9236674,0.9185868,0.9441922,0.9475424,0.9049621,0.9168312,0.9244731,0.9140373,0.9377759,0.9161596,0.9348655,0.9114148,0.9513839,0.9400719,0.9246404,0.9439951,0.9024291,0.9092543,0.9431646,0.9245426,0.9328490,0.9373101,0.9150403,0.9272471,0.9006587,0.9297889,0.9450524,0.8991197,0.9082949,0.9024526,0.9137400,0.9143539,0.9203691,0.8401727,0.9002130,0.9399206,0.9378715,0.9171144,0.9453757,0.9361942,0.9332102,0.9277641,0.9418657,0.8989545,0.8999019,0.9331079,0.9298916,0.9289415,0.8884878,0.9423227,0.9269941,0.9365955,0.9129572,0.9006071,0.9417174,0.9180885,0.9339814,0.9087525,0.9323643,0.9162010,0.9321646,0.9040556,0.9045976,0.9212631,0.9065515,0.9295423,0.9224827,0.9411037,0.9118039,0.9248373,0.8815993,0.9458109,0.9335531,0.9319516,0.9360127,0.9350896,0.9379194,0.9111462,0.9143559,0.9436070,0.9408365,0.9418666,0.9152046,0.9256868,0.9114021,0.9039916,0.9433421,0.9456004,0.8905821,0.9289293,0.9135030,0.9380342,0.9410659,0.9009832,0.9254943,0.9379552,0.9420493,0.9272543,0.9426511,0.9230355,0.9417734,0.9348623,0.9212002,0.9372154,0.9178481,0.9348995,0.9082842,0.9273352,0.9305257,0.9271389,0.9409008,0.9228690,0.9294233,0.9410290,0.9456247,0.9259136,0.9133999,0.9172487,0.9279999,0.9442652,0.9418394,0.9353516,0.9306456,0.9461981,0.9120428,0.9444288,0.9261917,0.9134235,0.9221468,0.9268680,0.9299942,0.9410947,0.9311216,0.8531980,0.9195330,0.9428920,0.9182355,0.9298785,0.9291454,0.9413903,0.9473329,0.9450126,0.9244527,0.9233499,0.9252443,0.9427113,0.9146360,0.9167987,0.8646052,0.9264615,0.9136045,0.9267083,0.9406629,0.9231414,0.9332108,0.9135517,0.9185995,0.8898477,0.9170699,0.8863843,0.9370966,0.8906094,0.9255950,0.9354178,0.9241023,0.9359083,0.9346141,0.9386938,0.8956537,0.9106883,0.8606199,0.9300732,0.9216099,0.9372995,0.9188982,0.9287956,0.9370079,0.8908120,0.9288846,0.9178886,0.9298670,0.9410440,0.9355810,0.9381543,0.8869428,0.9199314,0.9307575,0.9268033,0.9091761,0.9485645,0.9297772,0.9254957,0.9275126,0.8802429,0.9462737,0.9283691,0.9280790,0.9391584,0.9270855,0.9276816,0.9285827,0.9340272,0.9062003,0.9127253,0.9426764,0.9413182,0.8820663,0.9416080,0.9102886,0.9458919,0.9237903,0.9429557,0.9129157,0.9143693,0.9302893,0.9181583,0.9379540,0.9474054,0.9045847,0.9312507,0.9159101,0.9230544,0.9414040,0.9290013,0.9347239,0.9301550,0.9003678,0.9321544,0.9308649,0.9397558,0.9458266,0.9308956,0.9366243,0.8799180,0.9318982,0.9204334,0.9025390,0.9418448,0.9077577,0.9254611,0.9275637,0.9292262,0.9379037,0.8996545,0.8981794,0.9355075,0.9366821,0.9285465,0.8892768,0.9389396,0.9291704,0.9351731,0.9163757,0.9032845,0.9433927,0.9207379,0.9374219,0.9115161,0.9344985,0.9162886,0.9320741,0.9085454,0.9025283,0.9177151,0.9037841,0.9260393,0.9232187,0.9346955,0.9125685,0.9314250,0.8768260,0.9447168,0.9434787,0.9270535,0.9399921,0.9386836,0.9327083,0.9060668,0.9097083,0.9339198,0.9420666,0.9421143,0.9140322,0.9253566,0.9035169,0.9083668,0.9421882,0.9453609,0.8891737,0.9313480,0.9148689,0.9340470,0.9423768,0.9064833,0.9218602,0.9434939,0.9394189,0.9253648,0.9475066,0.9227709,0.9405955,0.9333696,0.9302664,0.9344980,0.9219474,0.9258227,0.9076624,0.9277345,0.9282934,0.9291268,0.9438592,0.9185618,0.9336327,0.9374520,0.9381991,0.9294446,0.9109747,0.9222301,0.9280245,0.9417267,0.9352095,0.9448515,0.9311809,0.9422397,0.9113128,0.9383248,0.9288283,0.9189713,0.9166455,0.9219319,0.9286882,0.9357808,0.9374043,0.8530840,0.9221423,0.9389867,0.9214431,0.9299394,0.9280818,0.9354338,0.9435693,0.9407452,0.9387169,0.8902022,0.8980480,0.9271352,0.9376973,0.9164503,0.8887102,0.9351251,0.9338028,0.9310444,0.9127122,0.8977691,0.9446573,0.9262264,0.9286947,0.9105551,0.9327388,0.9143077,0.9256094,0.9137826,0.8980471,0.9161285,0.8983476,0.9183644,0.9125770,0.9263818,0.9150621,0.9312360,0.8721069,0.9438839,0.9358761,0.9208259,0.9329527,0.9286975,0.9321340,0.9154770,0.9014315,0.9405166,0.9329630,0.9318578,0.9108686,0.9176164,0.9057282,0.9018456,0.9375731,0.9388524,0.8872286,0.9233617,0.9114148,0.9361608,0.9424924,0.9016084,0.9166736,0.9339847,0.9330464,0.9167328,0.9397929,0.9201724,0.9364130,0.9241700,0.9273636,0.9314350,0.9136255,0.9187601,0.9094539,0.9221245,0.9300737,0.9246797,0.9462778,0.9092040,0.9294417,0.9383275,0.9325886,0.9335157,0.9045428,0.9133238,0.9276185,0.9296234,0.9319622,0.9368990,0.9276108,0.9314718,0.9077271,0.9412942,0.9235538,0.9145481,0.9149006,0.9172184,0.9307338,0.9265331,0.9249520,0.8498408,0.9074013,0.9348798,0.9206072,0.9240142,0.9251926,0.9382546,0.9366482,0.9304202,0.9337160,0.9104239,0.9017303,0.9403213,0.9225286,0.9224377,0.8723937,0.9321649,0.9222458,0.9355128,0.9188721,0.9086958,0.9418667,0.9120659,0.9333661,0.9028656,0.9323932,0.9004171,0.9372407,0.9024337,0.9064452,0.9265020,0.9146563,0.9349395,0.9264631,0.9480339,0.9044085,0.9200041,0.8655847,0.9370036,0.9361314,0.9317637,0.9370583,0.9443900,0.9502548,0.8971827,0.9154524,0.9316128,0.9450299,0.9404557,0.9174122,0.9326796,0.8979035,0.9042054,0.9393182,0.9313085,0.9002656,0.9378614,0.9178252,0.9321456,0.9314996,0.8920428,0.9307707,0.9417820,0.9381476,0.9315192,0.9375264,0.9167336,0.9338440,0.9409031,0.9138061,0.9233664,0.9251596,0.9349322,0.8933527,0.9358502,0.9196886,0.9352997,0.9292433,0.9298930,0.9221389,0.9230669,0.9414311,0.9221658,0.9246062,0.9278692,0.9103133,0.9413889,0.9288763,0.9302793,0.9400059,0.9446237,0.9189525,0.9378820,0.9124459,0.9225060,0.9270110,0.9286376,0.9349669,0.9411665,0.9371243,0.8612721,0.9194138,0.9320074,0.9123686,0.9352742,0.9211550,0.9356318,0.9368385,0.9331641,0.9415196,0.9086009,0.9125572,0.9446775,0.9303711,0.9334389,0.8786522,0.9401917,0.9249647,0.9389714,0.9232660,0.9057150,0.9419563,0.9272083,0.9316271,0.9009734,0.9320589,0.9088324,0.9416032,0.9058387,0.9102346,0.9250950,0.9047513,0.9362515,0.9336316,0.9423168,0.9064340,0.9241368,0.8690873,0.9423486,0.9423854,0.9369284,0.9336866,0.9408142,0.9463756,0.9047355,0.9190776,0.9346364,0.9444341,0.9412088,0.9202924,0.9275932,0.9025397,0.9070999,0.9433009,0.9403661,0.8960470,0.9332011,0.9160169,0.9336077,0.9397893,0.9009303,0.9304844,0.9456842,0.9432263,0.9281648,0.9433766,0.9197307,0.9425062,0.9385453,0.9212829,0.9231201,0.9240763,0.9411703,0.8992077,0.9332947,0.9284330,0.9352064,0.9371620,0.9295939,0.9237699,0.9311097,0.9414602,0.9292920,0.9240585,0.9258847,0.9187349,0.9453264,0.9313299,0.9415233,0.9378479,0.9476827,0.9207946,0.9442996,0.9242438,0.9193489,0.9308137,0.9237090,0.9359810,0.9367962,0.9355439,0.8590187,0.9196754,0.9409634,0.9143811,0.9369415,0.9232513,0.9383678,0.9415556,0.9395039,0.9070850,0.8716371,0.8692675,0.9018068,0.9287872,0.9274018,0.9215392,0.9114259,0.9291868,0.9120296,0.8841367,0.8700669,0.9153027,0.9278419,0.9146662,0.9423087,0.9219107,0.9443001,0.9015938,0.9394324,0.8728502,0.8861335,0.8675228,0.9017525,0.8877883,0.9072493,0.9420730,0.9370012,0.9191402,0.9190518,0.9158874,0.8992383,0.9305646,0.9088722,0.9065317,0.9452306,0.8799289,0.9153961,0.9095441,0.9042662,0.8834612,0.8912059,0.9392795,0.8683345,0.9094378,0.9208454,0.8601185,0.8949222,0.8753003,0.9216333,0.9157313,0.9335615,0.8962563,0.9051632,0.9064902,0.8934817,0.9129597,0.8838651,0.9036799,0.9013889,0.9346664,0.9313923,0.8842387,0.8968008,0.9373398,0.8918494,0.9238926,0.8952839,0.9172060,0.8913200,0.9266713,0.9289448,0.9167785,0.9293411,0.8859003,0.8929853,0.9311412,0.9039701,0.9234772,0.9162090,0.9039772,0.9114948,0.8809031,0.9106900,0.9340290,0.8847586,0.8865611,0.8914762,0.8993432,0.9017487,0.9013532,0.8192097,0.8889014,0.9242705,0.9400647,0.8963067,0.9384875,0.9154556,0.9207420,0.9089327,0.9021986,0.8622435,0.8634084,0.8960183,0.9255748,0.9183827,0.9250063,0.9001943,0.9200044,0.9050371,0.8769117,0.8677724,0.9078000,0.9246904,0.9087297,0.9363191,0.9076806,0.9401757,0.8924008,0.9354510,0.8644699,0.8837391,0.8621980,0.8922374,0.8867915,0.9002692,0.9395495,0.9298140,0.9232955,0.9127606,0.9172041,0.8937413,0.9209294,0.9005925,0.8977952,0.9375537,0.8722727,0.9118089,0.9082337,0.9061007,0.8794225,0.8881012,0.9382655,0.8651638,0.9037765,0.9218767,0.8451860,0.8933027,0.8743221,0.9186077,0.9142522,0.9355401,0.8882699,0.9031541,0.9075511,0.8891453,0.9077039,0.8765561,0.8984827,0.9056033,0.9284333,0.9291581,0.8772063,0.8921640,0.9343340,0.8890301,0.9169373,0.8933845,0.9102579,0.8878366,0.9225664,0.9280961,0.9023503,0.9252551,0.8744634,0.8821155,0.9317332,0.8999258,0.9130654,0.9177859,0.8956001,0.9043040,0.8755864,0.9020335,0.9340704,0.8788567,0.8782761,0.8912014,0.8923774,0.8934850,0.9003731,0.8126500,0.8815537,0.9161451,0.9344102,0.8868929,0.9318451,0.9060274,0.9147376,0.9078747,0.9208115,0.8858132,0.8833907,0.9172888,0.9387571,0.9393314,0.8944674,0.9234136,0.9448719,0.9170497,0.9031745,0.8856167,0.9284816,0.9393646,0.9279804,0.9179661,0.9309724,0.9235400,0.9132086,0.9227581,0.8838911,0.9027775,0.8828749,0.9099919,0.9084532,0.9202489,0.9218590,0.9397177,0.8918494,0.9340006,0.9305473,0.9115246,0.9397463,0.9157647,0.9172377,0.9225400,0.8956392,0.9250168,0.9303354,0.9205902,0.8953989,0.9112694,0.9159963,0.8861002,0.9290732,0.9303358,0.8717476,0.9074307,0.8950935,0.9312341,0.9327269,0.9132342,0.9084745,0.9232836,0.9190958,0.9072211,0.9222944,0.8949804,0.9193999,0.9226917,0.9375942,0.9442124,0.9067444,0.9106769,0.9111521,0.9081648,0.9419101,0.9120978,0.9248582,0.9038165,0.9381472,0.9347199,0.9220157,0.9403313,0.9014924,0.9088962,0.9357612,0.9250439,0.9326391,0.9328556,0.9122847,0.9282734,0.8963653,0.9308572,0.9341812,0.9032791,0.9042633,0.9065086,0.9106911,0.9128324,0.9194965,0.8396320,0.8931170,0.9323635,0.9355251,0.9191341,0.9324784,0.9328653,0.9304807,0.9169209,0.9045271,0.8624521,0.8660044,0.8931369,0.9245028,0.9233110,0.9263117,0.9080427,0.9228870,0.9021177,0.8797702,0.8604049,0.9088285,0.9280332,0.9075110,0.9453158,0.9144789,0.9463440,0.8938825,0.9414410,0.8631273,0.8834549,0.8641659,0.8943841,0.8846117,0.8965572,0.9397510,0.9334358,0.9204473,0.9188994,0.9136257,0.8926338,0.9228095,0.8975283,0.8998615,0.9429916,0.8771945,0.9101527,0.9071915,0.9028184,0.8751532,0.8907716,0.9408272,0.8644623,0.9013999,0.9163709,0.8460941,0.8920648,0.8773695,0.9185669,0.9104813,0.9431335,0.8846051,0.9001551,0.9014192,0.8830206,0.9069575,0.8784366,0.9036268,0.9011289,0.9283262,0.9321653,0.8778266,0.8929459,0.9434615,0.8857652,0.9168844,0.8949023,0.9054014,0.8874463,0.9282334,0.9201384,0.9054969,0.9228221,0.8763847,0.8852740,0.9248926,0.9067380,0.9093942,0.9199320,0.8955182,0.9014023,0.8755637,0.9046489,0.9360475,0.8820379,0.8849867,0.8902376,0.8890976,0.8980934,0.8962276,0.8105874,0.8777248,0.9122100,0.9370998,0.8896425,0.9267955,0.9122638,0.9125786,0.9094424,0.9286804,0.8871095,0.8868419,0.9230553,0.9416388,0.9341247,0.9089637,0.9344346,0.9426287,0.9307765,0.9051975,0.8914371,0.9347608,0.9438014,0.9300426,0.9211886,0.9347619,0.9256112,0.9198990,0.9246912,0.8871379,0.9076661,0.8889795,0.9169654,0.9111759,0.9243268,0.9280009,0.9408994,0.8969386,0.9407348,0.9353180,0.9202217,0.9409247,0.9264726,0.9214181,0.9273266,0.8985336,0.9382609,0.9323808,0.9275205,0.9049293,0.9135454,0.9170071,0.8898469,0.9319499,0.9402540,0.8767415,0.9180142,0.9020443,0.9364320,0.9386435,0.9162704,0.9118774,0.9309486,0.9273456,0.9128972,0.9318971,0.9130603,0.9266610,0.9163210,0.9439021,0.9441599,0.9075821,0.9186941,0.9170471,0.9069274,0.9443844,0.9172797,0.9371446,0.9114662,0.9441892,0.9449943,0.9275814,0.9408576,0.8992620,0.9121854,0.9457209,0.9290934,0.9401402,0.9390666,0.9176270,0.9288912,0.9024510,0.9256256,0.9394063,0.9083865,0.9111829,0.9158808,0.9160681,0.9218517,0.9198004,0.8412570,0.9088323,0.9379317,0.9355245,0.9183123,0.9374531,0.9409678,0.9326568,0.9312359,0.9307924,0.9227608,0.9195465,0.9445965,0.9153258,0.9136981,0.8650460,0.9335127,0.9144434,0.9255671,0.9280910,0.9212817,0.9350611,0.9046056,0.9219289,0.8859096,0.9175873,0.8925308,0.9425077,0.8856506,0.9277785,0.9356002,0.9240136,0.9312118,0.9303056,0.9365930,0.8945098,0.9044885,0.8574394,0.9227738,0.9222995,0.9392325,0.9174242,0.9316887,0.9374422,0.8913250,0.9262968,0.9261422,0.9335297,0.9374991,0.9335656,0.9442537,0.8845947,0.9274166,0.9302828,0.9257213,0.9111169,0.9466482,0.9307630,0.9233939,0.9292112,0.8875108,0.9464292,0.9309784,0.9266435,0.9435272,0.9283869,0.9407343,0.9245530,0.9266884,0.9046528,0.9146117,0.9350051,0.9455802,0.8881643,0.9371323,0.9068013,0.9434731,0.9279398,0.9412707,0.9102923,0.9203159,0.9292696,0.9138463,0.9387682,0.9391610,0.9041203,0.9313776,0.9201759,0.9227027,0.9384255,0.9235530,0.9314121,0.9243646,0.9094796,0.9291241,0.9393071,0.9394227,0.9472864,0.9404164,0.9325360,0.8767104,0.9312971,0.9211914,0.9024789,0.9418796,0.9033058,0.9143871,0.9236343,0.9378940,0.9411646,0.9173718,0.9133502,0.9490643,0.9297895,0.9208865,0.8813893,0.9407480,0.9209137,0.9352581,0.9280102,0.9150502,0.9399616,0.9226889,0.9356881,0.8944578,0.9290475,0.9018235,0.9494649,0.9015769,0.9175273,0.9339506,0.9199027,0.9369096,0.9362173,0.9436861,0.8997569,0.9162794,0.8631867,0.9349882,0.9317323,0.9461149,0.9326213,0.9452999,0.9444646,0.9000141,0.9264140,0.9340426,0.9442749,0.9452951,0.9282071,0.9415558,0.8946461,0.9204913,0.9390272,0.9369757,0.9066508,0.9433571,0.9252379,0.9291319,0.9369196,0.8924066,0.9346883,0.9412681,0.9399163,0.9379227,0.9395326,0.9310160,0.9326966,0.9379556,0.9175439,0.9257656,0.9352678,0.9451890,0.9015096,0.9413890,0.9189223,0.9432989,0.9335008,0.9353156,0.9248060,0.9327518,0.9375878,0.9212870,0.9289756,0.9327169,0.9161378,0.9424127,0.9301340,0.9328542,0.9448863,0.9363741,0.9255742,0.9368810,0.9138262,0.9282873,0.9335175,0.9301274,0.9403655,0.9455735,0.9412344,0.8672130,0.9285412,0.9308016,0.9090215,0.9439259,0.9185218,0.9266014,0.9343878,0.9406245,0.9235123,0.8811077,0.8804760,0.9162351,0.9389293,0.9404548,0.9027046,0.9199779,0.9435537,0.9184902,0.8990092,0.8832224,0.9293084,0.9379299,0.9287847,0.9329549,0.9345890,0.9329728,0.9091347,0.9332016,0.8814271,0.9050767,0.8889123,0.9121805,0.9009316,0.9201664,0.9360320,0.9438734,0.9016812,0.9292936,0.9313999,0.9078870,0.9433902,0.9166614,0.9245790,0.9295995,0.8924962,0.9301320,0.9216537,0.9188713,0.8924455,0.9078022,0.9274845,0.8817366,0.9207673,0.9290919,0.8734626,0.9088355,0.8983159,0.9333498,0.9270599,0.9253146,0.9075761,0.9202985,0.9221387,0.9063362,0.9271423,0.8981257,0.9213985,0.9161903,0.9452272,0.9471755,0.8990768,0.9122716,0.9233050,0.9134738,0.9396541,0.9130851,0.9264480,0.9020600,0.9453307,0.9375487,0.9251995,0.9394035,0.8978988,0.9032692,0.9384905,0.9216068,0.9336963,0.9271657,0.9178750,0.9274602,0.8969688,0.9228765,0.9434781,0.8956826,0.9003576,0.9012875,0.9145387,0.9202597,0.9165049,0.8345373,0.8961730,0.9293915,0.9424657,0.9087325,0.9482440,0.9302429,0.9232165,0.9234157,0.9324029,0.8904573,0.8924657,0.9242768,0.9448352,0.9417167,0.8929400,0.9298831,0.9328327,0.9307178,0.9103965,0.8933847,0.9389461,0.9342833,0.9397521,0.9157115,0.9439717,0.9160782,0.9221767,0.9154900,0.8992682,0.9114057,0.8921590,0.9249724,0.9158259,0.9285031,0.9179399,0.9364380,0.8932216,0.9442502,0.9446888,0.9294352,0.9525604,0.9316919,0.9279301,0.9189658,0.9077538,0.9399379,0.9373744,0.9326115,0.9043371,0.9195329,0.9184489,0.8928490,0.9352279,0.9416351,0.8812183,0.9210786,0.9093210,0.9416029,0.9390988,0.9051672,0.9180980,0.9362229,0.9354678,0.9143087,0.9432427,0.9103114,0.9277001,0.9291338,0.9404303,0.9425400,0.9081513,0.9263383,0.9162578,0.9183331,0.9477638,0.9192191,0.9293078,0.9144037,0.9449044,0.9417297,0.9370689,0.9362857,0.9085666,0.9105663,0.9386064,0.9288618,0.9369051,0.9398835,0.9241486,0.9345357,0.9026553,0.9355282,0.9286230,0.9116449,0.9131811,0.9167636,0.9195751,0.9227500,0.9281766,0.8429603,0.9059279,0.9394853,0.9276957,0.9284771,0.9367189,0.9423182,0.9415205,0.9380844,0.9377350,0.9121690,0.9080660,0.9439484,0.9190626,0.9193402,0.8753188,0.9403137,0.9183303,0.9322403,0.9309826,0.9138848,0.9382821,0.9160227,0.9296380,0.8953411,0.9280324,0.8985589,0.9438217,0.8990584,0.9160520,0.9368743,0.9220833,0.9367437,0.9328552,0.9423805,0.9021541,0.9177700,0.8702339,0.9350095,0.9270722,0.9367447,0.9330510,0.9437198,0.9398349,0.9018908,0.9256934,0.9274998,0.9355587,0.9453163,0.9280722,0.9358115,0.8904150,0.9142095,0.9355755,0.9344600,0.9056559,0.9372202,0.9309657,0.9294207,0.9356584,0.8882937,0.9373344,0.9376325,0.9385223,0.9356203,0.9406987,0.9303590,0.9370070,0.9427191,0.9150529,0.9178989,0.9333850,0.9381813,0.8950457,0.9389373,0.9228127,0.9413495,0.9340698,0.9277672,0.9193228,0.9266171,0.9392570,0.9215241,0.9269025,0.9343259,0.9171419,0.9394501,0.9251058,0.9289961,0.9441075,0.9396158,0.9282875,0.9329471,0.9143710,0.9344478,0.9303526,0.9376192,0.9424232,0.9456153,0.9457913,0.8679871,0.9280686,0.9311611,0.9091754,0.9434033,0.9138738,0.9232990,0.9373535,0.9343415,0.9288922,0.8820515,0.8856521,0.9164058,0.9515133,0.9347001,0.9080847,0.9244108,0.9457001,0.9192829,0.9024298,0.8834871,0.9337720,0.9402627,0.9302483,0.9199094,0.9361421,0.9317802,0.9215563,0.9262122,0.8876862,0.9046429,0.8876760,0.9098882,0.9067589,0.9222296,0.9251083,0.9500650,0.8913276,0.9376707,0.9316069,0.9116620,0.9412974,0.9214495,0.9237598,0.9253908,0.8946210,0.9346788,0.9294457,0.9191801,0.8945254,0.9080302,0.9266876,0.8895641,0.9252029,0.9374568,0.8714715,0.9114495,0.8967062,0.9333711,0.9358171,0.9198869,0.9095114,0.9215931,0.9272144,0.9097588,0.9280787,0.8968069,0.9261681,0.9201694,0.9377991,0.9442851,0.9040364,0.9097151,0.9261023,0.9109063,0.9321742,0.9175542,0.9339801,0.9014262,0.9481444,0.9474442,0.9256485,0.9427178,0.8998004,0.9080914,0.9456004,0.9271885,0.9357479,0.9304694,0.9179131,0.9235439,0.8987220,0.9319258,0.9387842,0.9034636,0.9042947,0.9130411,0.9157439,0.9194132,0.9204617,0.8395367,0.8984014,0.9312291,0.9366888,0.9143127,0.9439888,0.9338350,0.9322387,0.9251233,0.9252694,0.8819786,0.8820848,0.9161452,0.9454866,0.9394026,0.9040173,0.9247044,0.9407057,0.9226393,0.9005596,0.8807707,0.9279368,0.9385326,0.9324669,0.9251101,0.9350777,0.9354120,0.9153219,0.9322021,0.8842572,0.9011872,0.8864890,0.9154678,0.9021307,0.9183563,0.9322996,0.9545464,0.8947958,0.9357238,0.9343548,0.9086782,0.9396393,0.9181837,0.9193348,0.9301477,0.8996281,0.9315898,0.9265221,0.9227771,0.8961794,0.9063337,0.9269029,0.8842786,0.9215274,0.9308686,0.8697172,0.9132423,0.8993651,0.9363312,0.9317696,0.9258707,0.9079635,0.9205764,0.9212315,0.9051294,0.9290747,0.9027060,0.9243239,0.9213521,0.9423970,0.9493343,0.9040087,0.9108869,0.9329009,0.9106100,0.9305680,0.9146306,0.9274767,0.9012288,0.9435889,0.9427719,0.9251137,0.9413055,0.8951395,0.9002993,0.9467581,0.9237583,0.9291273,0.9305001,0.9136804,0.9216800,0.8937886,0.9305547,0.9486815,0.9001797,0.9013376,0.9090957,0.9121324,0.9222303,0.9155717,0.8289122,0.8990362,0.9303136,0.9379738,0.9139091,0.9468894,0.9348580,0.9276285,0.9241347,0.9118656,0.8765486,0.8748704,0.9035764,0.9349834,0.9277420,0.9140267,0.9149605,0.9279421,0.9165257,0.8876078,0.8708776,0.9221189,0.9304512,0.9215249,0.9371225,0.9267135,0.9394857,0.9124451,0.9390077,0.8772236,0.8907571,0.8762118,0.9030104,0.8930163,0.9113155,0.9414285,0.9411742,0.9080230,0.9191474,0.9227402,0.9089439,0.9341263,0.9052146,0.9124550,0.9376823,0.8812744,0.9268699,0.9230522,0.9164120,0.8868554,0.8989190,0.9353205,0.8718031,0.9125281,0.9179563,0.8605287,0.9071921,0.8805328,0.9282278,0.9203980,0.9264728,0.9013511,0.9135003,0.9155889,0.8950266,0.9228765,0.8953407,0.9108278,0.9123312,0.9392979,0.9464609,0.8948867,0.9070408,0.9360638,0.8985750,0.9259898,0.9027349,0.9194350,0.8951254,0.9333596,0.9336977,0.9213022,0.9348637,0.8920812,0.8945139,0.9352061,0.9096257,0.9221365,0.9173650,0.9129400,0.9113975,0.8835066,0.9140333,0.9400886,0.8907268,0.8899934,0.8979382,0.9069950,0.9112754,0.9084234,0.8234875,0.8922616,0.9191258,0.9413389,0.9039935,0.9454362,0.9270347,0.9256349,0.9215325,0.9338151,0.9167472,0.9093404,0.9440428,0.9204379,0.9211949,0.8748222,0.9431454,0.9177452,0.9378566,0.9234355,0.9124125,0.9426008,0.9159426,0.9314995,0.8976589,0.9303338,0.8983479,0.9393982,0.8986982,0.9157581,0.9304381,0.9164423,0.9322890,0.9320006,0.9458050,0.9004725,0.9153693,0.8686572,0.9349735,0.9322074,0.9405267,0.9277573,0.9424286,0.9377797,0.8966574,0.9264933,0.9231959,0.9373319,0.9426590,0.9229664,0.9359331,0.8926389,0.9173587,0.9397107,0.9367000,0.9047303,0.9372575,0.9209463,0.9240429,0.9393903,0.8895762,0.9338806,0.9362773,0.9472306,0.9351106,0.9387348,0.9257021,0.9379118,0.9344208,0.9190564,0.9190658,0.9266829,0.9407040,0.8957051,0.9225856,0.9207981,0.9399380,0.9353860,0.9262184,0.9257942,0.9260070,0.9397480,0.9172885,0.9262167,0.9285671,0.9161699,0.9387842,0.9278775,0.9316295,0.9388170,0.9412080,0.9279832,0.9368977,0.9140528,0.9213836,0.9259121,0.9327731,0.9445348,0.9365892,0.9392875,0.8659928,0.9315240,0.9332757,0.9138477,0.9443218,0.9146288,0.9282060,0.9442901,0.9398244,0.9429209,0.9082897,0.9021616,0.9367002,0.9335984,0.9308619,0.8907520,0.9454034,0.9322546,0.9401787,0.9207078,0.9077662,0.9430838,0.9348517,0.9393717,0.9085335,0.9408850,0.9109982,0.9376829,0.9134383,0.9099449,0.9270694,0.9096034,0.9282436,0.9273392,0.9429742,0.9120752,0.9277697,0.8779540,0.9482472,0.9415516,0.9374742,0.9374444,0.9439990,0.9417462,0.9139625,0.9162404,0.9340263,0.9431921,0.9350894,0.9215732,0.9295163,0.9049552,0.9117761,0.9497512,0.9455538,0.8975948,0.9325091,0.9178260,0.9392070,0.9480442,0.9026940,0.9262123,0.9454884,0.9456197,0.9291109,0.9444214,0.9240744,0.9422223,0.9321213,0.9290208,0.9266420,0.9194000,0.9376931,0.9097297,0.9235630,0.9352433,0.9346815,0.9473103,0.9283390,0.9354697,0.9381858,0.9428012,0.9332215,0.9210145,0.9285294,0.9280054,0.9386908,0.9364673,0.9494450,0.9378562,0.9429863,0.9216239,0.9443111,0.9276766,0.9142212,0.9227425,0.9239928,0.9314070,0.9347657,0.9308022,0.8630777,0.9263842,0.9476957,0.9251685,0.9362044,0.9303724,0.9423127,0.9501350,0.9490243,0.9163702,0.8766898,0.8751561,0.9079501,0.9372865,0.9356301,0.9185982,0.9188390,0.9326743,0.9145705,0.8891735,0.8753321,0.9234586,0.9344878,0.9193866,0.9369459,0.9268934,0.9395306,0.9098205,0.9325830,0.8744169,0.8932949,0.8783911,0.9072320,0.9017311,0.9116596,0.9380900,0.9455948,0.9033157,0.9282447,0.9272593,0.9025452,0.9309693,0.9130004,0.9140493,0.9372695,0.8878207,0.9201653,0.9236146,0.9171673,0.8897422,0.8991348,0.9320430,0.8772428,0.9140869,0.9275853,0.8639511,0.9052555,0.8875720,0.9283343,0.9223792,0.9328621,0.8978006,0.9142660,0.9158393,0.8980897,0.9221074,0.8929131,0.9114411,0.9109879,0.9410926,0.9445461,0.8934524,0.9066574,0.9339223,0.9012642,0.9288285,0.9047572,0.9221795,0.8982203,0.9349288,0.9326346,0.9167102,0.9321868,0.8887842,0.8931914,0.9413845,0.9132181,0.9292123,0.9250300,0.9067797,0.9200605,0.8862388,0.9166075,0.9484347,0.8940585,0.8996225,0.8968868,0.9059925,0.9117954,0.9101227,0.8246544,0.8948923,0.9253750,0.9445538,0.9021962,0.9432878,0.9256616,0.9246058,0.9146176,0.9388991,0.9224283,0.9181905,0.9434941,0.9153720,0.9170556,0.8685312,0.9339161,0.9216076,0.9336689,0.9331672,0.9247467,0.9365151,0.9142949,0.9235905,0.8925436,0.9250842,0.8929753,0.9346005,0.8956503,0.9133919,0.9361484,0.9234821,0.9388257,0.9387645,0.9405201,0.8995518,0.9080240,0.8643677,0.9340756,0.9323411,0.9458000,0.9301008,0.9386279,0.9400372,0.8967291,0.9243596,0.9289486,0.9363699,0.9359053,0.9392406,0.9483549,0.8865855,0.9243581,0.9364769,0.9320433,0.9138826,0.9479345,0.9305551,0.9258011,0.9327549,0.8884593,0.9374930,0.9385623,0.9322481,0.9404570,0.9307250,0.9352478,0.9327861,0.9365340,0.9143383,0.9164900,0.9347035,0.9460348,0.8824491,0.9412591,0.9228479,0.9428200,0.9337174,0.9382486,0.9136025,0.9228845,0.9332517,0.9185077,0.9300947,0.9382289,0.9137957,0.9334502,0.9251878,0.9315430,0.9426960,0.9345715,0.9294682,0.9305752,0.9118571,0.9292199,0.9355811,0.9368290,0.9470748,0.9415426,0.9353179,0.8711874,0.9320583,0.9242820,0.9043158,0.9450833,0.9149925,0.9239763,0.9303601,0.9331324,0.9383238,0.8996406,0.9073760,0.9374139,0.9255466,0.9184919,0.8823739,0.9415450,0.9251430,0.9407537,0.9162594,0.9051323,0.9415553,0.9229335,0.9330079,0.9048533,0.9321602,0.9045767,0.9319224,0.8987677,0.9000383,0.9213488,0.9027684,0.9371662,0.9214680,0.9384944,0.9112557,0.9235758,0.8814471,0.9394514,0.9324457,0.9370666,0.9356261,0.9341833,0.9335201,0.8980867,0.9125103,0.9373843,0.9419610,0.9412377,0.9205437,0.9240439,0.8978969,0.9030934,0.9353383,0.9447755,0.8910987,0.9330727,0.9145815,0.9336765,0.9425765,0.8960411,0.9276138,0.9409336,0.9444157,0.9255616,0.9448054,0.9223179,0.9388229,0.9387469,0.9182155,0.9254252,0.9187614,0.9352604,0.8967072,0.9321898,0.9269227,0.9368575,0.9387309,0.9216228,0.9292508,0.9309038,0.9445347,0.9224195,0.9145274,0.9224573,0.9180918,0.9388717,0.9328685,0.9369144,0.9287177,0.9422377,0.9197590,0.9370078,0.9180499,0.9196623,0.9173927,0.9294238,0.9291889,0.9338229,0.9352887,0.8536138,0.9205294,0.9326149,0.9144840,0.9362526,0.9217847,0.9286069,0.9434781,0.9410986,0.9423939,0.9227692,0.9095523,0.9428277,0.9267753,0.9257541,0.8761957,0.9447666,0.9226949,0.9325536,0.9302409,0.9145541,0.9389494,0.9218152,0.9355987,0.8994032,0.9337100,0.9011391,0.9400894,0.8999465,0.9182691,0.9349063,0.9119053,0.9345153,0.9313630,0.9442893,0.9048270,0.9179381,0.8702291,0.9401840,0.9382285,0.9425437,0.9357654,0.9465280,0.9514490,0.9032774,0.9192651,0.9357541,0.9486626,0.9453974,0.9265753,0.9398632,0.8966778,0.9166218,0.9462122,0.9400682,0.9057568,0.9416852,0.9274655,0.9351438,0.9416600,0.8942528,0.9346280,0.9425052,0.9470401,0.9421236,0.9417850,0.9209843,0.9412176,0.9427565,0.9233970,0.9248217,0.9295131,0.9456043,0.8972354,0.9355658,0.9239425,0.9428161,0.9385543,0.9281022,0.9276830,0.9311638,0.9431699,0.9253284,0.9287029,0.9359389,0.9171496,0.9429532,0.9281354,0.9398393,0.9416826,0.9455842,0.9276600,0.9424971,0.9182362,0.9148674,0.9313404,0.9302085,0.9438631,0.9410516,0.9401089,0.8687113,0.9234263,0.9341988,0.9143800,0.9431739,0.9241723,0.9344505,0.9359179,0.9458032,0.9031937,0.9340407,0.9400945,0.9149876,0.8908035,0.8820701,0.8373965,0.9036474,0.8824784,0.9047822,0.9272797,0.9395050,0.9079224,0.8878584,0.8912200,0.8539184,0.8960861,0.8624997,0.9065303,0.8644267,0.9415271,0.9298019,0.9401091,0.9076408,0.9293157,0.9058685,0.8659332,0.8807585,0.8259266,0.9008242,0.9009208,0.9116891,0.8886458,0.9018172,0.9063697,0.8575309,0.9311879,0.8894820,0.9021030,0.9103859,0.9322638,0.9219266,0.8491963,0.9463978,0.9008248,0.9025092,0.9376992,0.9228673,0.9321176,0.8885291,0.8992761,0.8540506,0.9152861,0.9074418,0.9040569,0.9116489,0.9037482,0.9171617,0.9001676,0.9031352,0.8803165,0.8851175,0.9341807,0.9205549,0.8534762,0.9123352,0.8870505,0.9177650,0.8957930,0.9160423,0.8866366,0.8892914,0.9054605,0.8869615,0.9315357,0.9274830,0.8781095,0.9021072,0.8935848,0.8998281,0.9098319,0.9011328,0.9303777,0.9017746,0.8752230,0.9187273,0.9120264,0.9114449,0.9196213,0.9057604,0.9063118,0.9059180,0.9267298,0.8943694,0.8752875,0.9147502,0.8791118,0.8929991,0.9008891,0.9044639,0.8788039,0.9185444,0.9174149,0.8855826,0.8670574,0.8568498,0.8108181,0.8818607,0.8526104,0.8739705,0.9071155,0.9193560,0.8754683,0.8531034,0.8677715,0.8313363,0.8622111,0.8356472,0.8792372,0.8315971,0.9200897,0.9068304,0.9145982,0.8849278,0.9061913,0.8846820,0.8367427,0.8540101,0.8028074,0.8739788,0.8705609,0.8862326,0.8661311,0.8853220,0.8805410,0.8323150,0.9018070,0.8644095,0.8863107,0.8864481,0.9066108,0.8958725,0.8267360,0.9175722,0.8749422,0.8787052,0.9261726,0.8975449,0.9038943,0.8662288,0.8798995,0.8279892,0.8961197,0.8783015,0.8789173,0.8954014,0.8750869,0.8971472,0.8758470,0.8794532,0.8550614,0.8582844,0.9033379,0.8900803,0.8305937,0.8835465,0.8575818,0.8941167,0.8744099,0.8973965,0.8659765,0.8646200,0.8802052,0.8626931,0.9047258,0.9029530,0.8569404,0.8779793,0.8648209,0.8732134,0.8851113,0.8785078,0.9130206,0.8792730,0.8510954,0.8948888,0.8912820,0.8944476,0.8986848,0.8885108,0.8783805,0.9335761,0.9070854,0.8685096,0.8483759,0.8919845,0.8523771,0.8669347,0.8754376,0.8817965,0.9461828,0.9126777,0.9148958,0.9367032,0.9342239,0.9285924,0.8833349,0.9445646,0.9294941,0.9452353,0.9266208,0.9115586,0.9451286,0.9197072,0.9386460,0.9008838,0.9332240,0.9022563,0.9363335,0.9064661,0.9151747,0.9283850,0.9091882,0.9355248,0.9250206,0.9435511,0.9120497,0.9243419,0.8773650,0.9382495,0.9380940,0.9386829,0.9352026,0.9425208,0.9406381,0.9073348,0.9159082,0.9386740,0.9513077,0.9435876,0.9231626,0.9327396,0.9000728,0.9106322,0.9435025,0.9424300,0.8946493,0.9369967,0.9135567,0.9416139,0.9433086,0.8967227,0.9383313,0.9404282,0.9395940,0.9338807,0.9420786,0.9253853,0.9430368,0.9350321,0.9215827,0.9277555,0.9286583,0.9399638,0.9006414,0.9258613,0.9238124,0.9406098,0.9479492,0.9254255,0.9292438,0.9358068,0.9467838,0.9295772,0.9230684,0.9327855,0.9269057,0.9399772,0.9370124,0.9355248,0.9408977,0.9469242,0.9223613,0.9441388,0.9208888,0.9206680,0.9282669,0.9285960,0.9400449,0.9358948,0.9358524,0.8633004,0.9244964,0.9358433,0.9152059,0.9403352,0.9253219,0.9364878,0.9440235,0.9446603,0.9355185,0.9220162,0.9183826,0.9457093,0.9140775,0.9145447,0.8683156,0.9291613,0.9089928,0.9207438,0.9339259,0.9218084,0.9292168,0.9060366,0.9238362,0.8860003,0.9147419,0.8873604,0.9404380,0.8869861,0.9235932,0.9379659,0.9271109,0.9303664,0.9356954,0.9364396,0.8924962,0.9043782,0.8623251,0.9285034,0.9222640,0.9386575,0.9235188,0.9334322,0.9370485,0.8926966,0.9295347,0.9256164,0.9326822,0.9316925,0.9315181,0.9483151,0.8853798,0.9271479,0.9347997,0.9272971,0.9166838,0.9478930,0.9339922,0.9241581,0.9262687,0.8847747,0.9410568,0.9340320,0.9320161,0.9467626,0.9260705,0.9366483,0.9282826,0.9307502,0.9100712,0.9137082,0.9358724,0.9449078,0.8890288,0.9440578,0.9111245,0.9435055,0.9241493,0.9416559,0.9154688,0.9226934,0.9220090,0.9121654,0.9361079,0.9399834,0.9097901,0.9379270,0.9218876,0.9289044,0.9373328,0.9311750,0.9304723,0.9270564,0.9058424,0.9329435,0.9462862,0.9376596,0.9400626,0.9419961,0.9324718,0.8804044,0.9311366,0.9232460,0.9005025,0.9398215,0.9052149,0.9143527,0.9186891,0.9298605,0.9024233,0.9318591,0.9396211,0.9124793,0.8860194,0.8806794,0.8350005,0.9038341,0.8835254,0.9009123,0.9236075,0.9369322,0.9025342,0.8841864,0.8883997,0.8580023,0.8921937,0.8600257,0.9081542,0.8559655,0.9402544,0.9190933,0.9379474,0.9011928,0.9228682,0.9117140,0.8637603,0.8740852,0.8270714,0.8985048,0.8952236,0.9155418,0.8897930,0.9058963,0.9056834,0.8623648,0.9302899,0.8914634,0.9045830,0.9123289,0.9361028,0.9186045,0.8507839,0.9440564,0.8987092,0.9003457,0.9430801,0.9236525,0.9321001,0.8912695,0.9038766,0.8558411,0.9232800,0.9054544,0.9069173,0.9088684,0.8993266,0.9225577,0.9041455,0.9001845,0.8799058,0.8841793,0.9289104,0.9173061,0.8548200,0.9148225,0.8880346,0.9161895,0.8971475,0.9162154,0.8828126,0.8930349,0.9041052,0.8841743,0.9271824,0.9199870,0.8796986,0.9019133,0.8832477,0.8967776,0.9108869,0.9030710,0.9271620,0.8981340,0.8751401,0.9215832,0.9166190,0.9133439,0.9146950,0.9065373,0.9021482,0.9046188,0.9294110,0.8884879,0.8691570,0.9160479,0.8808014,0.8885809,0.9010004,0.9036875,0.9313133,0.8980365,0.8989766,0.9330222,0.9442005,0.9376827,0.8968644,0.9341978,0.9391350,0.9369081,0.9141623,0.8993013,0.9405694,0.9299968,0.9419908,0.9178516,0.9388093,0.9181898,0.9354381,0.9102607,0.8971534,0.9170115,0.8947035,0.9261429,0.9176411,0.9370322,0.9162289,0.9396923,0.8849716,0.9451380,0.9374830,0.9293497,0.9401236,0.9350003,0.9317848,0.9167332,0.9073140,0.9443382,0.9371692,0.9344888,0.9123857,0.9149599,0.9171880,0.9006109,0.9308608,0.9450455,0.8875388,0.9244023,0.9079601,0.9440059,0.9470518,0.9055235,0.9211085,0.9341703,0.9338380,0.9199763,0.9379653,0.9155769,0.9348190,0.9284269,0.9349015,0.9389067,0.9169709,0.9305992,0.9189090,0.9235328,0.9385667,0.9255166,0.9395888,0.9189907,0.9406723,0.9468480,0.9347675,0.9413965,0.9126076,0.9171871,0.9356814,0.9391734,0.9469384,0.9473389,0.9322231,0.9361755,0.9108357,0.9419437,0.9308756,0.9107863,0.9197169,0.9201755,0.9229474,0.9304284,0.9283668,0.8488901,0.9128590,0.9490794,0.9294235,0.9273892,0.9336144,0.9451235,0.9414679,0.9419002,0.9404995,0.8987761,0.8962829,0.9208674,0.9463228,0.9321370,0.8989097,0.9317360,0.9435031,0.9332604,0.9119496,0.8956064,0.9422011,0.9414500,0.9324638,0.9104687,0.9435983,0.9198928,0.9278084,0.9202600,0.8993285,0.9123393,0.8946045,0.9256917,0.9171623,0.9318071,0.9214300,0.9354064,0.8911114,0.9450459,0.9382378,0.9260800,0.9433836,0.9360804,0.9345594,0.9204720,0.9026681,0.9433581,0.9344019,0.9331028,0.9093956,0.9235644,0.9089464,0.9018156,0.9360347,0.9483378,0.8804738,0.9240160,0.9088672,0.9476314,0.9455701,0.9113904,0.9134483,0.9346628,0.9377461,0.9155701,0.9376316,0.9109425,0.9375137,0.9272168,0.9381888,0.9433413,0.9097494,0.9288325,0.9143107,0.9167351,0.9441927,0.9248240,0.9394127,0.9135923,0.9397259,0.9494802,0.9334732,0.9403601,0.9111753,0.9142425,0.9383291,0.9299234,0.9367346,0.9509711,0.9223823,0.9327680,0.9110904,0.9434206,0.9327030,0.9054505,0.9059816,0.9180042,0.9269557,0.9232645,0.9248993,0.8467850,0.9130862,0.9420903,0.9276948,0.9257909,0.9359806,0.9360915,0.9402474,0.9410389,0.9235953,0.9149654,0.9248926,0.9338956,0.9041439,0.9014645,0.8615077,0.9164179,0.9031388,0.9184163,0.9359105,0.9278139,0.9216759,0.9031100,0.9113716,0.8774637,0.9100307,0.8817198,0.9331973,0.8784247,0.9249141,0.9412626,0.9261485,0.9301617,0.9381772,0.9276661,0.8885260,0.9067212,0.8433471,0.9179470,0.9111909,0.9342020,0.9083343,0.9285870,0.9292003,0.8806815,0.9298397,0.9147135,0.9230767,0.9311421,0.9344864,0.9328312,0.8766639,0.9269856,0.9247408,0.9177928,0.9170304,0.9363871,0.9358990,0.9132240,0.9245907,0.8686371,0.9420004,0.9244704,0.9234738,0.9360127,0.9142177,0.9415592,0.9210280,0.9253133,0.8969570,0.8995920,0.9425084,0.9390231,0.8757719,0.9390686,0.9002461,0.9384032,0.9181495,0.9359961,0.9044978,0.9115873,0.9213786,0.9023831,0.9337155,0.9427355,0.9030981,0.9309766,0.9138998,0.9178222,0.9238376,0.9213892,0.9409723,0.9191683,0.8961850,0.9332996,0.9459288,0.9330438,0.9379034,0.9248161,0.9249835,0.8871271,0.9357288,0.9158374,0.8961637,0.9388374,0.8910840,0.9100184,0.9179577,0.9187346,0.9160456,0.9365678,0.9379850,0.9213038,0.8901860,0.8932635,0.8484104,0.9147383,0.8827082,0.9083999,0.9346445,0.9321904,0.9094600,0.8965716,0.8996615,0.8666635,0.9016330,0.8655863,0.9155404,0.8683328,0.9345930,0.9317315,0.9376816,0.9162596,0.9363304,0.9136100,0.8706292,0.8797639,0.8378290,0.9090919,0.9044599,0.9213294,0.8986324,0.9132304,0.9140207,0.8671375,0.9392640,0.9030374,0.9091897,0.9159113,0.9380186,0.9321264,0.8614186,0.9378644,0.9112645,0.9126672,0.9298246,0.9287114,0.9388771,0.9042249,0.9053218,0.8580221,0.9279307,0.9111408,0.9102246,0.9274305,0.9050614,0.9210030,0.9086507,0.9185296,0.8885597,0.8917640,0.9388338,0.9236300,0.8609262,0.9209189,0.8910683,0.9244422,0.9060018,0.9301788,0.8951451,0.8973862,0.9147881,0.8863811,0.9375936,0.9310247,0.8874699,0.9182935,0.8996622,0.9072295,0.9159054,0.9111741,0.9396826,0.9049519,0.8848365,0.9254235,0.9277626,0.9209510,0.9230991,0.9156394,0.9183699,0.8899482,0.9389588,0.9060910,0.8832455,0.9270724,0.8869709,0.8998994,0.9067843,0.9125469,0.9146840,0.9394461,0.9337615,0.9176404,0.9034726,0.8954595,0.8511053,0.9102419,0.8959846,0.9102872,0.9456896,0.9360067,0.9164824,0.8946277,0.9012430,0.8706546,0.9017256,0.8764371,0.9211531,0.8733225,0.9361302,0.9427249,0.9359055,0.9207742,0.9434292,0.9271409,0.8763713,0.8939838,0.8414317,0.9144214,0.9124112,0.9167296,0.9048335,0.9232070,0.9233639,0.8734970,0.9395740,0.9060761,0.9217046,0.9223522,0.9468250,0.9311189,0.8760411,0.9373141,0.9130750,0.9179228,0.9226090,0.9363745,0.9499247,0.9127366,0.9202842,0.8652084,0.9356204,0.9107629,0.9189178,0.9298652,0.9149941,0.9287434,0.9204267,0.9200682,0.8909988,0.9030338,0.9424323,0.9252861,0.8673373,0.9310996,0.8993685,0.9309315,0.9123984,0.9325033,0.8997828,0.9069378,0.9165198,0.9018942,0.9421320,0.9368182,0.8954505,0.9220247,0.9110730,0.9141501,0.9283405,0.9226185,0.9493237,0.9183948,0.8880724,0.9314505,0.9325655,0.9331950,0.9318299,0.9236604,0.9219206,0.8932844,0.9375642,0.9084073,0.8902039,0.9298640,0.8916971,0.9041468,0.9120466,0.9191662,0.9237417,0.8820092,0.8817635,0.9154319,0.9347686,0.9399019,0.9098263,0.9245114,0.9427211,0.9233551,0.8935366,0.8774866,0.9256619,0.9384051,0.9276801,0.9277414,0.9321440,0.9314396,0.9098944,0.9253175,0.8814713,0.8989218,0.8804868,0.9088695,0.8910773,0.9179636,0.9309331,0.9479627,0.9051312,0.9284019,0.9198361,0.9111142,0.9347052,0.9220317,0.9167509,0.9306882,0.8905061,0.9280339,0.9253012,0.9185901,0.8947542,0.9000956,0.9260667,0.8805495,0.9188742,0.9258557,0.8683818,0.9078163,0.8922560,0.9313787,0.9283569,0.9279281,0.9040169,0.9121350,0.9204702,0.9062059,0.9179255,0.8972650,0.9173005,0.9123667,0.9483637,0.9461699,0.8925091,0.9121315,0.9275596,0.8974739,0.9272270,0.9088895,0.9295038,0.9006669,0.9387259,0.9343452,0.9208813,0.9316436,0.8931796,0.9032207,0.9453364,0.9161842,0.9298530,0.9314199,0.9071020,0.9185407,0.8873783,0.9193515,0.9490276,0.8911077,0.9009873,0.9028680,0.9075347,0.9152729,0.9102805,0.8325226,0.8965987,0.9251537,0.9419172,0.9126170,0.9428376,0.9264145,0.9204233,0.9267732,0.9082257,0.8616758,0.8626029,0.8969696,0.9266549,0.9198271,0.9217436,0.9067759,0.9251453,0.9051133,0.8829249,0.8621449,0.9120806,0.9238173,0.9051151,0.9419877,0.9157267,0.9523094,0.8937311,0.9435376,0.8647970,0.8835309,0.8681568,0.8964004,0.8887375,0.9043486,0.9454646,0.9286504,0.9223110,0.9098584,0.9161098,0.8928304,0.9250665,0.8972903,0.9065302,0.9451664,0.8743853,0.9196633,0.9040282,0.9008573,0.8781420,0.8949487,0.9452238,0.8606246,0.9072309,0.9167938,0.8511809,0.8950332,0.8746594,0.9158533,0.9145865,0.9341243,0.8857398,0.9053991,0.9066839,0.8915970,0.9083420,0.8830547,0.9051411,0.9002729,0.9266422,0.9340280,0.8771122,0.8957511,0.9409674,0.8927186,0.9202358,0.8964577,0.9079925,0.8873077,0.9260262,0.9271919,0.9072146,0.9276715,0.8803868,0.8887563,0.9295125,0.9059893,0.9188060,0.9127656,0.8996223,0.9065416,0.8788172,0.9081125,0.9347181,0.8797141,0.8828671,0.8865849,0.8977313,0.9012441,0.8996124,0.8159160,0.8762648,0.9150341,0.9340128,0.8917406,0.9326868,0.9178212,0.9157065,0.9065591,0.9433699,0.9086185,0.9059765,0.9452506,0.9260520,0.9185301,0.8813361,0.9412564,0.9263293,0.9337135,0.9264523,0.9083836,0.9462172,0.9263212,0.9316952,0.8992400,0.9315336,0.9048817,0.9372954,0.9023428,0.9077982,0.9246557,0.9064877,0.9343217,0.9284663,0.9421920,0.9070377,0.9208746,0.8800506,0.9382346,0.9388965,0.9335973,0.9374330,0.9381149,0.9461006,0.9068869,0.9123187,0.9364729,0.9435500,0.9418523,0.9203697,0.9356677,0.9009751,0.9050686,0.9417785,0.9393131,0.8955686,0.9366483,0.9171587,0.9342035,0.9404251,0.8959412,0.9294488,0.9455152,0.9374904,0.9315394,0.9432305,0.9200190,0.9412165,0.9391441,0.9176356,0.9252806,0.9187584,0.9383230,0.8976567,0.9370420,0.9269567,0.9391962,0.9388093,0.9243471,0.9270839,0.9270624,0.9369708,0.9296735,0.9215168,0.9285624,0.9177995,0.9390620,0.9353652,0.9328891,0.9382409,0.9384297,0.9231953,0.9387619,0.9103974,0.9285781,0.9243059,0.9268381,0.9349443,0.9351840,0.9412282,0.8617144,0.9206984,0.9368098,0.9151104,0.9343702,0.9192792,0.9334242,0.9417975,0.9430840,0.9426158,0.9078639,0.9060886,0.9361068,0.9429743,0.9395075,0.8933749,0.9432021,0.9413908,0.9403858,0.9228105,0.9064210,0.9460831,0.9346280,0.9410030,0.9081118,0.9458653,0.9157633,0.9341194,0.9163062,0.9081606,0.9197467,0.9014553,0.9322996,0.9279781,0.9398331,0.9205023,0.9335097,0.8806871,0.9500030,0.9483642,0.9353752,0.9441294,0.9395123,0.9418958,0.9177573,0.9132089,0.9456622,0.9424488,0.9391577,0.9199072,0.9266111,0.9103199,0.9052171,0.9411847,0.9531479,0.8914555,0.9295391,0.9126595,0.9462058,0.9506879,0.9072837,0.9293463,0.9392704,0.9439929,0.9249519,0.9454073,0.9229498,0.9448797,0.9318788,0.9315606,0.9378212,0.9206039,0.9346305,0.9081525,0.9234895,0.9385542,0.9349490,0.9489597,0.9249515,0.9394782,0.9464327,0.9411692,0.9409635,0.9195729,0.9298190,0.9352860,0.9377851,0.9411704,0.9505430,0.9332670,0.9416276,0.9201775,0.9505472,0.9329365,0.9175708,0.9237185,0.9243470,0.9337270,0.9276721,0.9310552,0.8537427,0.9156844,0.9483159,0.9252897,0.9338170,0.9302419,0.9441378,0.9475988,0.9438738)
plot1 = ggplot(data = NULL, aes(x = dist1, y = ..density..)) + geom_histogram(bins = 100) + geom_vline(aes(xintercept = 0.8066626), color = "red") + ggtitle("Method 1", paste("p =", as.character(.0004))) + xlab("Sorensen Index")

## ---- fig.width=8, echo=FALSE-------------------------------------------------
dist2 = c(0.9129066,0.9184350,0.8988366,0.8321705,0.9153036,0.8854942,0.8811304,0.8913535,0.7341722,0.8515664,0.8536195,0.8901306,0.8855479,0.8881165,0.9177803,0.8765598,0.8086373,0.9124544,0.8172614,0.8192190,0.7954984,0.8842433,0.9189611,0.8722292,0.9348788,0.8409329,0.8639109,0.9001527,0.9222463,0.9231293,0.8804507,0.8528016,0.8609889,0.8969811,0.9078839,0.8641836,0.8510093,0.8635549,0.8131291,0.8841868,0.8511895,0.9257771,0.9017789,0.8974790,0.9193449,0.9207025,0.8708324,0.7588041,0.8525127,0.8619684,0.9065372,0.8822649,0.8139026,0.8398272,0.8365755,0.9009274,0.8202223,0.8768611,0.8856939,0.8923981,0.9226053,0.8532152,0.8828919,0.8436272,0.8773192,0.8696274,0.8680345,0.8501513,0.9165553,0.9079160,0.8944038,0.8981102,0.8165778,0.8530352,0.9166089,0.9166408,0.9247092,0.8822670,0.8889894,0.9043699,0.8703746,0.8533176,0.9034248,0.9212109,0.9107388,0.8772738,0.8941240,0.8973858,0.8699977,0.8722497,0.8548561,0.8233300,0.9077235,0.8812553,0.8303347,0.7457686,0.9172385,0.8743793,0.8722693,0.8834681,0.9023051,0.9255538,0.8943731,0.8126204,0.9179851,0.8599187,0.8782863,0.8991254,0.8893856,0.8995022,0.8681141,0.9000330,0.9171313,0.8460199,0.9016184,0.9215975,0.8845232,0.9206581,0.8415141,0.9124383,0.9026442,0.9135124,0.9247522,0.8989686,0.8654985,0.7817132,0.8810075,0.9070230,0.8064246,0.8165293,0.9157421,0.8582443,0.8314655,0.8728964,0.9272314,0.8066312,0.8971707,0.9332922,0.9144443,0.8762348,0.8647258,0.8939648,0.8970488,0.8453535,0.9272905,0.9073762,0.9157268,0.9221617,0.8813266,0.8892192,0.8709757,0.8934090,0.8789599,0.8881161,0.9045429,0.8844677,0.8747692,0.8836371,0.8761079,0.8657741,0.8924904,0.8731360,0.8217270,0.8785967,0.8188138,0.8083955,0.9004983,0.8150386,0.9210491,0.9066743,0.8327136,0.8989861,0.8995224,0.9156702,0.8496692,0.8951462,0.8206213,0.8734960,0.9204095,0.8970995,0.7852508,0.8771157,0.8970985,0.8981131,0.8975687,0.8865763,0.9163636,0.8431235,0.9016682,0.8842635,0.8806267,0.9210598,0.9161154,0.8662424,0.8974157,0.8996730,0.8654716,0.8447247,0.8897276,0.8928042,0.8302623,0.8402384,0.8155379,0.9049617,0.9277148,0.9036028,0.8847155,0.8491631,0.8701814,0.9095347,0.8841084,0.9419762,0.9068022,0.8868394,0.9202481,0.8986448,0.8465106,0.8510692,0.8587290,0.7791825,0.8687087,0.8932363,0.8438418,0.8782578,0.8988284,0.8452467,0.8980772,0.9058667,0.8393731,0.8190500,0.9191493,0.8811299,0.9063829,0.8812708,0.8391746,0.8839000,0.9081181,0.9228986,0.8984239,0.8694600,0.8620391,0.7868720,0.9140732,0.9167341,0.9236110,0.8321634,0.8255969,0.9018618,0.8517594,0.9051024,0.9079694,0.8789241,0.9095442,0.9286956,0.8724392,0.9320868,0.9006965,0.8992661,0.9086275,0.8593543,0.8893116,0.8867643,0.8194355,0.9049241,0.8024710,0.9089080,0.8455733,0.8445795,0.7723578,0.9364774,0.8848350,0.8087675,0.8286463,0.9189434,0.8631931,0.8922125,0.8997205,0.9172660,0.8320710,0.8691202,0.9283195,0.9036343,0.8623986,0.8575824,0.8092632,0.8614649,0.8621981,0.8882103,0.9242715,0.8847128,0.9181240,0.9186782,0.8446683,0.8967821,0.8809292,0.8201989,0.8589315,0.8635584,0.8836393,0.8621330,0.8630467,0.9223854,0.9059094,0.9034469,0.8562337,0.9133220,0.8464521,0.9110211,0.8807768,0.8397743,0.8614001,0.9178480,0.8783585,0.9256644,0.8291270,0.7827929,0.9204578,0.8725464,0.9111130,0.8331111,0.9278623,0.8812108,0.8917470,0.8771642,0.8756347,0.9302684,0.8879572,0.8509430,0.9103969,0.9100854,0.8485758,0.8795773,0.7857189,0.8835210,0.8313734,0.8941296,0.8407887,0.8763393,0.8249285,0.8696335,0.8795015,0.8984424,0.8273574,0.8343457,0.8535855,0.9041251,0.9262930,0.8917076,0.8803669,0.8832129,0.8928778,0.8572301,0.8865375,0.9000152,0.7320970,0.8904403,0.8043555,0.8819148,0.8310343,0.9234766,0.8702087,0.8725707,0.8759373,0.9013806,0.8979669,0.9098459,0.8933298,0.8774060,0.8930575,0.8653992,0.9002511,0.9091046,0.9141858,0.7602055,0.9245891,0.8205997,0.8991580,0.8874364,0.8830779,0.8916867,0.9077373,0.8578970,0.8924702,0.9030896,0.8844384,0.8950204,0.8693999,0.8858988,0.8626382,0.9214483,0.7681160,0.9014305,0.8978316,0.9141171,0.8451553,0.8700584,0.9128327,0.8790429,0.8195849,0.9047377,0.9067659,0.7862385,0.9088877,0.8482408,0.8994887,0.8202047,0.9236280,0.8910771,0.8579969,0.8951778,0.8888479,0.8374230,0.9053305,0.8611888,0.9014138,0.8618401,0.8942473,0.9018069,0.8708855,0.8813555,0.8785585,0.8721553,0.8474228,0.9331582,0.9137092,0.8899248,0.8269351,0.9039741,0.8672410,0.8403056,0.9071666,0.8743358,0.9080244,0.8546032,0.9224185,0.8741164,0.8556744,0.9199970,0.8556275,0.8638040,0.8555870,0.9121923,0.8667182,0.8292916,0.9175514,0.9197871,0.8973968,0.9259639,0.8972217,0.8983607,0.9111722,0.9113415,0.8373630,0.9231320,0.8294235,0.9076928,0.9183250,0.9225609,0.9108802,0.9027458,0.8771066,0.9029692,0.9197129,0.9052539,0.8555481,0.8936445,0.8433432,0.9207376,0.9239105,0.9193745,0.8433862,0.9141318,0.8528167,0.8043405,0.8955933,0.8748893,0.8937691,0.9234837,0.9099860,0.9207327,0.9279976,0.9259952,0.8829991,0.9025666,0.8835065,0.9165830,0.9199702,0.9239809,0.9065267,0.8623827,0.8047926,0.8238191,0.8338891,0.9179576,0.8646351,0.9131616,0.7682424,0.9196837,0.8200771,0.8220316,0.8353000,0.8198585,0.8572706,0.8030054,0.8550805,0.9178803,0.9249295,0.8851103,0.8664365,0.8728052,0.8069446,0.8859525,0.8668338,0.8668875,0.8432703,0.8565701,0.8266942,0.9036652,0.9234408,0.8544421,0.8969158,0.7956162,0.8665732,0.9143458,0.8805132,0.9053416,0.8303862,0.8115095,0.9110813,0.8760744,0.8522217,0.9019430,0.8595556,0.9279954,0.9048120,0.8984280,0.8931772,0.8995327,0.8854984,0.8868690,0.9106454,0.8303711,0.8885690,0.9208059,0.8608064,0.8907562,0.8795607,0.8953092,0.8749433,0.9066334,0.9037546,0.9122872,0.8377439,0.7978160,0.8635937,0.8559753,0.8896606,0.9229724,0.8019911,0.8840099,0.8467497,0.8645563,0.8992253,0.9014196,0.8955220,0.9139304,0.8074411,0.9201148,0.8705478,0.8904925,0.9137119,0.8946540,0.7896141,0.8837669,0.9060744,0.8820038,0.8217656,0.8541842,0.8246093,0.8638431,0.9207165,0.9131910,0.8007491,0.8174008,0.9037258,0.8690737,0.9185550,0.8385425,0.8466492,0.9049468,0.8875263,0.8327227,0.9039441,0.8517809,0.8571980,0.8387437,0.9026439,0.8541121,0.8612640,0.8228375,0.9268128,0.8199356,0.8778378,0.9210269,0.8584026,0.8872756,0.8701419,0.8891453,0.8344815,0.8316347,0.9022232,0.9232668,0.8940532,0.9076830,0.8402887,0.8578757,0.8862573,0.9204284,0.8389121,0.9148324,0.8824902,0.7847660,0.8930700,0.9068073,0.8232915,0.9153950,0.8717059,0.9133693,0.9037900,0.8307594,0.8525517,0.8804848,0.9103615,0.8452223,0.8723979,0.9140533,0.8407751,0.8985457,0.9135978,0.9237265,0.9279558,0.8990298,0.8470296,0.8749050,0.9024095,0.7991491,0.9266391,0.9125389,0.9039187,0.8842822,0.9116180,0.8975576,0.8291339,0.8106791,0.8797949,0.9141150,0.8433131,0.8955384,0.9016728,0.8707043,0.8917508,0.9108071,0.9308324,0.8630132,0.8860737,0.8901027,0.9112844,0.8661313,0.9110144,0.9204920,0.8252655,0.9166775,0.8495316,0.7543033,0.8913392,0.8716820,0.9056028,0.8698918,0.9095464,0.8576023,0.9212102,0.8610501,0.8226493,0.8150610,0.8755970,0.8828484,0.8756923,0.9138887,0.9016029,0.8260946,0.8795669,0.8271733,0.8790730,0.8778017,0.8995787,0.8683425,0.8720250,0.9100084,0.8504954,0.9045482,0.8759443,0.9066013,0.9155820,0.8553922,0.9052396,0.8381718,0.8647514,0.9022164,0.8647310,0.8421876,0.9023959,0.8953395,0.8881892,0.8962110,0.8975761,0.8730488,0.9171608,0.9061098,0.9053552,0.8698935,0.8062397,0.9136102,0.9236911,0.8983280,0.8726989,0.8980953,0.8347730,0.9211302,0.9016442,0.8829478,0.8981626,0.8730196,0.9170173,0.8922785,0.9051066,0.8133443,0.8830517,0.9001347,0.9280588,0.8980840,0.8974557,0.8322118,0.8672578,0.8777120,0.9129954,0.8096506,0.9206314,0.8885439,0.8774389,0.8620311,0.8201082,0.8625611,0.9269388,0.8823691,0.9015690,0.9076576,0.8677989,0.8785103,0.9023614,0.9146303,0.8965389,0.8679449,0.8735278,0.8990792,0.9042356,0.8177480,0.8042053,0.8602451,0.8489539,0.8865212,0.8640161,0.9277293,0.8365528,0.9036649,0.8234103,0.8619268,0.8956715,0.7529967,0.8957099,0.8880197,0.8523358,0.9015373,0.9161577,0.8715423,0.9266325,0.8529342,0.9061223,0.8891175,0.8802987,0.8632758,0.9185658,0.9166642,0.9212089,0.8632162,0.8139115,0.8988379,0.8545600,0.8526004,0.9126216,0.8894504,0.8519656,0.8322558,0.8577425,0.8655018,0.8341529,0.8962311,0.8876096,0.8389856,0.8256057,0.8865322,0.7751498,0.9123791,0.9054228,0.8539367,0.8009683,0.8066053,0.8618157,0.9061622,0.8452553,0.8759936,0.8235147,0.9265626,0.8534243,0.8213616,0.9122879,0.8998365,0.8984294,0.8647140,0.8328117,0.8800182,0.8524364,0.8531940,0.8623241,0.8318089,0.8778297,0.9268388,0.8289879,0.8372420,0.8396954,0.8499195,0.8959959,0.9194502,0.8504959,0.8718010,0.8592872,0.8265922,0.9270922,0.8727373,0.9260113,0.8954320,0.8719597,0.8104344,0.9121337,0.8493027,0.9096867,0.8803901,0.8644542,0.8236452,0.8753530,0.8861975,0.8718966,0.8394446,0.8539811,0.9007645,0.8594661,0.8689079,0.9204117,0.9112290,0.8062456,0.8133076,0.8490288,0.9116163,0.8754465,0.8866371,0.8235968,0.9149397,0.8080404,0.9094683,0.8615789,0.8991379,0.8752557,0.8443140,0.9099127,0.8720477,0.8949682,0.8884433,0.8209941,0.8964813,0.8469413,0.8090245,0.9106547,0.8287546,0.9182300,0.8558695,0.8542787,0.9261381,0.8786324,0.8007477,0.8372771,0.8238876,0.8239715,0.7799288,0.8912530,0.9179266,0.9210725,0.8456676,0.9129093,0.9249755,0.8777784,0.8085838,0.8102979,0.8793851,0.8837045,0.8650219,0.8979129,0.8648489,0.8728783,0.8942167,0.9001836,0.8097858,0.9126764,0.9072934,0.9040253,0.8713677,0.8948654,0.9087600,0.8737351,0.8635566,0.8658457,0.8662169,0.8729999,0.8543121,0.8816821,0.9003339,0.8545241,0.8544277,0.8849173,0.8498069,0.9070210,0.9120922,0.9220377,0.9014748,0.8872154,0.9131127,0.8725162,0.9053729,0.8494433,0.9038477,0.8327036,0.9225957,0.8424233,0.8392175,0.9153210,0.7851127,0.9248109,0.8587314,0.9047638,0.9255434,0.8400176,0.9009229,0.8792677,0.8195692,0.9080718,0.8986578,0.9200051,0.8870988,0.9036209,0.8839412,0.8025023,0.8530422,0.8441817,0.8624812,0.8673264,0.9367351,0.9033957,0.8421643,0.9148444,0.8365065,0.8742826,0.8255556,0.8435162,0.8945005,0.9050418,0.9127301,0.9112899,0.9023027,0.8590205,0.9129079,0.8458223,0.8696807,0.9010374,0.8781084,0.9185313,0.9064956,0.9195779,0.9123540,0.8885751,0.8730800,0.9226342,0.8652564,0.9216312,0.8758398,0.9100578,0.9225924,0.8634018,0.9185112,0.9127250,0.8496913,0.8389746,0.8527036,0.9000387,0.8870645,0.8967375,0.8547338,0.8969387,0.9114068,0.9176384,0.8779516,0.8924016,0.8995697,0.9320458,0.7909506,0.8574350,0.8880606,0.8135595,0.8876969,0.7918274,0.8149934,0.8933421,0.9062216,0.8981051,0.7975811,0.8056442,0.8570258,0.8838222,0.8461073,0.8794603,0.8450369,0.8500565,0.8077214,0.9171728,0.8324738,0.8596908,0.8177310,0.8789765,0.9156835,0.9057372,0.8949883,0.8312889,0.9229161,0.9139789,0.8768593,0.8851122,0.8591214,0.8209883,0.9014003,0.9183134,0.7468275,0.8857503,0.9123025,0.8867439,0.9158256,0.8974817,0.8157898,0.8284832,0.8971207,0.8696259,0.7995165,0.8952773,0.9047813,0.8414435,0.8175447,0.8679450,0.8989988,0.8720991,0.9068836,0.8563984,0.8477403,0.9053078,0.8382680,0.9233961,0.8776305,0.8476039,0.8757274,0.8367223,0.8902053,0.8900747,0.8499712,0.9243095,0.9013406,0.8827414,0.9213959,0.9044908,0.8827921,0.9132662,0.8599509,0.8540704,0.8029002,0.9141690,0.8893087,0.8264495,0.9143421,0.8924707,0.8640181,0.8799240,0.8700806,0.9118301,0.9032465,0.9202174,0.8036189,0.7574337,0.8510264,0.8597115,0.9163138,0.8598701,0.8510799,0.8510454,0.8370419,0.9105266,0.8179767,0.8578510,0.9257905,0.7577408,0.8986502,0.9186972,0.9098938,0.8788151,0.8899354,0.8862525,0.9020034,0.8212283,0.8261627,0.8518356,0.8066691,0.9019051,0.9159018,0.8580289,0.8948625,0.8545307,0.8907971,0.8947830,0.8413420,0.8972629,0.8812642,0.8932056,0.9155125,0.8040277,0.9130156,0.8737745,0.8231803,0.9186167,0.8900818,0.8663966,0.8281140,0.9080825,0.8281417,0.7394252,0.8491504,0.8393717,0.8790260,0.8906212,0.8966380,0.8902008,0.8209971,0.8958807,0.8926814,0.8481155,0.8894868,0.8758781,0.9149792,0.8295139,0.8806846,0.9099017,0.9156061,0.9220730,0.9311170,0.8074391,0.8839042,0.8891606,0.9128618,0.8536483,0.8391932,0.8906619,0.9172650,0.8684195,0.8573831,0.7410772,0.9070807,0.8722107,0.9084290,0.8425597,0.8811592,0.7924909,0.9274315,0.8714107,0.8677776,0.8214337,0.8728230,0.8576269,0.9106559,0.8949196,0.8529433,0.8753853,0.8920366,0.8702634,0.8569098,0.8940922,0.9092524,0.8879360,0.7636228,0.9078817,0.8102215,0.8805447,0.9049338,0.8429736,0.8628108,0.8989667,0.8961777,0.8991613,0.8948489,0.9271717,0.8050737,0.8794477,0.8551988,0.8280641,0.8708555,0.9116639,0.9008359,0.8427056,0.8270838,0.9114738,0.9237391,0.8500532,0.9231245,0.9091995,0.8902626,0.9147270,0.8557590,0.9102765,0.9131631,0.9025394,0.8452528,0.9178764,0.8957807,0.8578016,0.8698949,0.9066356,0.7820718,0.8611070,0.8056034,0.8929845,0.8660980,0.8288083,0.8896818,0.8503533,0.8988618,0.8936967,0.8388595,0.8222149,0.8839929,0.7946578,0.7868661,0.8592207,0.9055545,0.8642253,0.9361421,0.9027797,0.9132896,0.7997852,0.8725229,0.9065639,0.8867951,0.8591830,0.9167022,0.8609007,0.7988869,0.8819517,0.9118174,0.8225448,0.8651576,0.7660422,0.8946755,0.8241314,0.7956331,0.8423326,0.9320513,0.8523898,0.9095059,0.9126258,0.9073986,0.9270702,0.8938166,0.8595549,0.8690139,0.8722658,0.8881545,0.9114580,0.8952353,0.8305193,0.8935713,0.8051354,0.8382164,0.8217092,0.8749182,0.8974011,0.8967125,0.8834960,0.9099826,0.8567050,0.8437638,0.9029428,0.8986677,0.9213678,0.9059749,0.9201968,0.8776219,0.8632839,0.8404179,0.9079095,0.8931324,0.8459628,0.8545079,0.8506446,0.9124826,0.8736277,0.8606547,0.8827172,0.9309771,0.8947999,0.9233190,0.9025371,0.8945466,0.9209991,0.9038263,0.8535564,0.9126302,0.8650498,0.8574425,0.9108368,0.8556345,0.8858349,0.8841828,0.8780697,0.9103469,0.8385181,0.8887703,0.8548835,0.8739894,0.8628747,0.8876577)
plot2 = ggplot(data = NULL, aes(x = dist2, y = ..density..)) + geom_histogram(bins = 100) + geom_vline(aes(xintercept = 0.8066626), color = "red") + ggtitle("Method 2", paste("p =", as.character(0.04740741))) + xlab("Sorensen Index")
grid.arrange(plot1, plot2, ncol=2)

## ---- echo=FALSE--------------------------------------------------------------
unlink("./Objects", recursive = TRUE)

