% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/posterior.predict.R
\name{posterior.predict}
\alias{posterior.predict}
\title{Prediction of the max-stable process at target positions.}
\usage{
posterior.predict(fit, targets, targets.covariates, sites.covariates,
  predict.type = "kriging")
}
\arguments{
\item{fit}{A named list.
Output from the \code{hkevp.fit} procedure.}

\item{targets}{A matrix of real values.
Coordinates of the sites where the marginal GEV distribution should be extrapolated. Each row corresponds to a target position and each column to a coordinate.}

\item{targets.covariates}{A matrix of real values.
Spatial covariates associated with each target. Each row corresponds to a target and each column to a spatial covariate. By default, only the intercept and the target coordinates are used. An error occurs if number of targets covariates and sites covariates do not match.}

\item{sites.covariates}{A matrix of real values.
Spatial covariates associated to the sites. Each row corresponds to a site and each column to a spatial covariate. By default, only the intercept and the target coordinates are used. An error occurs if number of targets covariates and sites covariates do not match.}

\item{predict.type}{Character string.
Specifies the type of prediction. Must be one of "\code{kriging}" (default) or "\code{climat}". See details.}
}
\value{
A three-dimensional array where:
\itemize{
\item Each row corresponds to a different realisation of the process (a block).
\item Each column corresponds to a target position.
\item Each slice corresponds to a MCMC step.
}
}
\description{
Returns the predictive posterior distribution of the block maxima process \eqn{Y(\cdot)} at a set of target sites \eqn{(s_1^*, ..., s_k^*)}, given the observations at \eqn{(s_1, ..., s_n)} and the output from \code{hkevp.fit}.

Two types of conditional prediction are available, as described in \cite{Shaby and Reich (2012)}. See details.
}
\details{
The spatial prediction of \eqn{Y_t(s^*)} for a target site \eqn{s^*} and a realisation \eqn{t} of the process is described in \cite{Shaby and Reich (2012)}. This method involves a three-step procedure:
\enumerate{
\item Computation of the residual dependence process \eqn{\theta(\cdot)} at the target positions.
\item Computation of the conditional GEV parameters \eqn{(\mu^*,\sigma^*,\xi^*)} at the target sites. See the definition of the HKEVP in \cite{Reich and Shaby (2012)}.
\item Generation of \eqn{Y_t(s^*)} from an independent GEV distribution with parameters \eqn{(\mu^*,\sigma^*,\xi^*)}.
}

As sketched in \cite{Shaby and Reich (2012)}, two types of prediction are possible: the kriging-type and the climatological-type. These two types differ when the residual dependence process \eqn{\theta} is computed (first step of the prediction):
\itemize{
\item The kriging-type takes the actual value of \eqn{A} in the MCMC algorithm to compute the residual dependence process. The prediction will be the distribution of the maximum recorded at the specified targets.
\item The climatological-type generates \eqn{A} by sampling from the positive stable distribution with characteristic exponent \eqn{\alpha}, where \eqn{\alpha} is the actual value of the MCMC step. The prediction in climatological-type will be the distribution of what could happen in the conditions of the HKEVP dependence structure.
}

Posterior distribution for each realisation \eqn{t} of the process and each target position \eqn{s^*} is represented with a sample where each element corresponds to a step of the MCMC procedure.
}
\examples{
# Simulation of HKEVP:
sites <- as.matrix(expand.grid(1:3,1:3))
knots <- sites
mu <- sites[,1]*10
sigma <- 3
xi <- .2
alpha <- .4
tau <- 1
ysim <- hkevp.rand(10, sites, knots, mu, sigma, xi, alpha, tau)

# HKEVP fit (omitting first site, used as target):
fit <- hkevp.fit(ysim, sites, knots, niter = 100, nburn = 50, quiet = FALSE)

# Extrapolation:
targets <- matrix(sites[1,], 1, 2)
ypred <- posterior.predict(fit, targets, predict.type = "kriging")

# Plot of the density and the true value for 4 first realizations:
# Could need more sites/iterations to be more precise
# par(mfrow = c(2, 2))
# plot(density(ypred[1,1,]), main = "Predictive density for year 1")
# abline(v = ysim[1,1], col = 2, lwd = 2)
# plot(density(ypred[2,1,]), main = "Predictive density for year 2")
# abline(v = ysim[2,1], col = 2, lwd = 2)
# plot(density(ypred[3,1,]), main = "Predictive density for year 3")
# abline(v = ysim[3,1], col = 2, lwd = 2)
# plot(density(ypred[4,1,]), main = "Predictive density for year 4")
# abline(v = ysim[4,1], col = 2, lwd = 2)




}
\author{
Quentin Sebille
}
\references{
Reich, B. J., & Shaby, B. A. (2012). A hierarchical max-stable spatial model for extreme precipitation. The annals of applied statistics, 6(4), 1430. <DOI:10.1214/12-AOAS591>

Shaby, B. A., & Reich, B. J. (2012). Bayesian spatial extreme value analysis to assess the changing risk of concurrent high temperatures across large portions of European cropland. Environmetrics, 23(8), 638-648. <DOI:10.1002/env.2178>
}

