% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/IMSE.R
\name{IMSE.search}
\alias{IMSE.search}
\title{IMSE minimization}
\usage{
IMSE.search(model, replicate = FALSE, Xcand = NULL,
  control = list(tol_dist = 1e-04, tol_diff = 0.001, multi.start = 5, maxit =
  100, maximin = TRUE, Xstart = NULL), Wijs = NULL, seed = NULL)
}
\arguments{
\item{model}{\code{homGP} or \code{hetGP} model}

\item{replicate}{if \code{TRUE}, search only on existing designs}

\item{Xcand}{optional set of of candidates for discrete search}

\item{control}{list in case \code{Xcand == NULL}, with elements \code{multi.start},
to perform a multi-start optimization is conducted based on \code{\link[stats]{optim}}, with \code{maxit} iterations each.
Also, \code{tol_dist} defines the minimum distance to an existing design for a new point to be added, otherwise the closest existing design is chosen.
In a similar fashion, \code{tol_dist} is the minimum relative change of IMSPE for adding a new design.}

\item{Wijs}{optional previously computed matrix of Wijs, see \code{\link[hetGP]{Wij}}}

\item{seed}{optional seed for the generation of designs}
}
\value{
list with \code{par}, \code{value} elements, and additional slot \code{new} (boolean if it is or not a new design) and \code{id} giving the index of the duplicated design.
}
\description{
Search for best reduction in IMSE
}
\examples{
###############################################################################
## Bi-variate example
###############################################################################

nvar <- 2 

set.seed(42)
ftest <- function(x, coef = 0.1) return(sin(2*pi*sum(x)) + rnorm(1, sd = coef))

n <- 25 # must be a square
xgrid0 <- seq(0.1, 0.9, length.out = sqrt(n))
designs <- as.matrix(expand.grid(xgrid0, xgrid0))
X <- designs[rep(1:n, sample(1:10, n, replace = TRUE)),]
Z <- apply(X, 1, ftest)

model <- mleHomGP(X, Z, lower = rep(0.1, nvar), upper = rep(1, nvar))

ngrid <- 51
xgrid <- seq(0,1, length.out = ngrid)
Xgrid <- as.matrix(expand.grid(xgrid, xgrid))

preds <- predict(x = Xgrid, object =  model)

## Initial plots
contour(x = xgrid,  y = xgrid, z = matrix(preds$mean, ngrid),
        main = "Predicted mean", nlevels = 20)
points(model$X0, col = 'blue', pch = 20)

IMSE_grid <- apply(Xgrid, 1, crit_IMSE, model = model)
filled.contour(x = xgrid, y = xgrid, matrix(IMSE_grid, ngrid),
               nlevels = 20, color.palette = terrain.colors, 
               main = "Initial IMSPE criterion landscape",
plot.axes = {axis(1); axis(2); points(model$X0, pch = 20)})

## Sequential IMSPE search
nsteps <- 1 # Increase for better results

for(i in 1:nsteps){
  res <- IMSE.search(model, control = list(multi.start = 100, maxit = 50))
  newX <- res$par
  newZ <- ftest(newX)
  model <- update(object = model, Xnew = newX, Znew = newZ)
}

## Final plots
contour(x = xgrid,  y = xgrid, z = matrix(preds$mean, ngrid),
        main = "Predicted mean", nlevels = 20)
points(model$X0, col = 'blue', pch = 20)

IMSE_grid <- apply(Xgrid, 1, crit_IMSE, model = model)
filled.contour(x = xgrid, y = xgrid, matrix(IMSE_grid, ngrid),
               nlevels = 20, color.palette = terrain.colors, 
               main = "Final IMSPE criterion landscape",
plot.axes = {axis(1); axis(2); points(model$X0, pch = 20)})

}
