% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/compute_comorbidity.R
\name{compute_comorbidity}
\alias{compute_comorbidity}
\title{Compute Elixhauser Comorbidity Index}
\usage{
compute_comorbidity(
  data,
  vars,
  icd_ver = c("ICD-10", "ICD-9-CM-3digits", "ICD-9-CM-5digits"),
  clnt_id,
  uid = NULL,
  sum_by = c("row", "clnt"),
  excl = NULL
)
}
\arguments{
\item{data}{Data.frames or remote tables (e.g., from \code{\link[dbplyr:tbl_sql]{dbplyr::tbl_sql()}})}

\item{vars}{An expression passing to \code{\link[dplyr:select]{dplyr::select()}}. It can be quoted/unquoted column names, or helper functions, such as \code{\link[dplyr:reexports]{dplyr::starts_with()}}.}

\item{icd_ver}{One of \code{c("ICD-10", "ICD-9-CM-3digits", "ICD-9-CM-5digits")}. Specify the ICD code version used in \code{data}. The ICD-10 and ICD-9-CM 5 digits version are from Quan et al. (2005). The ICD-9-CM 3 digits version is adopted from Manitoba Centre for Health Policy. It uses BOTH 3-digit and 5-digit codes in search. See their web page for cautions and limitations of the 3 digit version if your data only has 3-digit data (\url{http://mchp-appserv.cpe.umanitoba.ca/viewConcept.php?printer=Y&conceptID=1436#CAUTIONS}).}

\item{clnt_id}{Grouping variable (quoted/unquoted).}

\item{uid}{Variable name for a unique row identifier. It is necessary for SQL to produce consistent result based on sorting.}

\item{sum_by}{One of "row" or "clnt". The "row" option computes total score for each row (default), and the "clnt" option summarizes total score by \code{clnt_id}. Each disease categories will be counted only once in the calculation regardless of multiple records in a category.}

\item{excl}{A character vector of disease categories labels that should be excluded in the total score calculation. This is useful when some of the categories are the exposure/outcome of interest, and the goal is to measure comorbidity excluding these disease. See detail for a list of the categories and labels.}
}
\value{
A data.frame or remote table with binary indicators for each categories as columns.
}
\description{
This function computes unweighted Elixhauser Comorbidity Index for both data.frame and remote table input. The ICD codes used to identify the 31 disease categories is from Quan et al. (2005).
}
\details{
List of disease categories - labels (in quote):
\enumerate{
\item Congestive Heart Failure - "chf"
\item Cardiac Arrhythmia - "arrhy"
\item Valvular Disease - "vd"
\item Pulmonary Circulation Disorders - "pcd"
\item Peripheral Vascular Disorders - "pvd"
\item Hypertension Uncomplicated - "hptn_nc"
\item Hypertension complicated - "hptn_C"
\item Paralysis - "para"
\item Other Neurological Disorders - "Othnd"
\item Chronic Pulmonary Disease - "copd"
\item Diabetes Uncomplicated - "diab_nc"
\item Diabetes Complicated - "diab_c"
\item Hypothyroidism - "hptothy"
\item Renal Failure - "rf"
\item Liver Disease - "ld"
\item Peptic Ulcer Disease excluding bleeding - "pud_nb"
\item AIDS/HIV - "hiv"
\item Lymphoma - "lymp"
\item Metastatic Cancer - "mets"
\item Solid Tumor without Metastasis - "tumor"
\item Rheumatoid Arthritis/collagen - "rheum_a"
\item Coagulopathy - "coag"
\item Obesity - "obesity"
\item Weight Loss - "wl"
\item Fluid and Electrolyte Disorders - "fluid"
\item Blood Loss Anemia - "bla"
\item Deficiency Anemia - "da"
\item Alcohol Abuse - "alcohol"
\item Drug Abuse - "drug"
\item Psychoses - "psycho"
\item Depression - "dep"
}
}
\examples{
# make ICD-9 toy data
df <- data.frame(
  uid = 1:10, clnt_id = sample(1:3, 10, replace = TRUE),
  diagx_1 = c("193", "2780", "396", "4254", "4150", "401", "401", "0932", "5329", "2536"),
  diagx_2 = c(NA, NA, "72930", "V6542", "493", "405", "5880", "2409", "714", NA)
)

# compute Elixhauser Comorbidity Index by row
# uid is needed for by row calculation
# 3 categories were excluded in total_eci
compute_comorbidity(df,
  vars = starts_with("diagx"),
  icd_ver = "ICD-9-CM-5digits",
  clnt_id = clnt_id, uid = uid,
  excl = c("drug", "psycho", "dep")
)

# compute ECI by person
compute_comorbidity(df,
  vars = starts_with("diagx"),
  icd_ver = "ICD-9-CM-5digits",
  clnt_id = clnt_id,
  sum_by = "clnt"
)
}
\references{
Quan H, Sundararajan V, Halfon P, Fong A, Burnand B, Luthi JC, Saunders LD, Beck CA, Feasby TE, Ghali WA. Coding algorithms for defining comorbidities in ICD-9-CM and ICD-10 administrative data. Med Care 2005;43(11):1130-1139.
}
