hawkesbow
================
Felix Cheysson

<!-- README.md is generated from README.Rmd. Please edit that file -->

<!-- badges: start -->

<!-- badges: end -->

This package implements a spectral approach to the parametric estimation
of Hawkes processes from Binned Observations through Whittle likelihood
(HawkesBOW). It is based on the results of the article (Cheysson and
Lang 2020).

# Installation

You can install the released version of hawkesbow from
[GitHub](https://github.com/fcheysson/hawkesbow) with:

``` r
devtools::install_github("fcheysson/hawkesbow")
```

# The Hawkes process

Hawkes processes form a family of models for point processes for which
the occurrence of an event temporarily increases the probability of
future events occurring (Hawkes 1971). Formally, a Hawkes process
![N](https://latex.codecogs.com/png.latex?N "N") on ![\\mathbb
R](https://latex.codecogs.com/png.latex?%5Cmathbb%20R "\\mathbb R") can
be defined from its conditional intensity
![\\lambda(\\cdot)](https://latex.codecogs.com/png.latex?%5Clambda%28%5Ccdot%29
"\\lambda(\\cdot)"):  where the variables
![T\_i](https://latex.codecogs.com/png.latex?T_i "T_i") denote the
arrival times of the process, the *immigration intensity*
![\\eta](https://latex.codecogs.com/png.latex?%5Ceta "\\eta") is a
positive constant, and the *reproduction function* ![h: \\mathbb
R\_{\\ge 0} \\rightarrow \\mathbb
R\_{\\ge 0}](https://latex.codecogs.com/png.latex?h%3A%20%5Cmathbb%20R_%7B%5Cge%200%7D%20%5Crightarrow%20%5Cmathbb%20R_%7B%5Cge%200%7D
"h: \\mathbb R_{\\ge 0} \\rightarrow \\mathbb R_{\\ge 0}") is a
measurable function. The reproduction function can be further decomposed
as ![h = \\mu
h^\\ast](https://latex.codecogs.com/png.latex?h%20%3D%20%5Cmu%20h%5E%5Cast
"h = \\mu h^\\ast"), where ![\\mu = \\int\_\\mathbb R h(t) \\mathrm{d} t
\< 1](https://latex.codecogs.com/png.latex?%5Cmu%20%3D%20%5Cint_%5Cmathbb%20R%20h%28t%29%20%5Cmathrm%7Bd%7D%20t%20%3C%201
"\\mu = \\int_\\mathbb R h(t) \\mathrm{d} t \< 1") is called the  and
![h^\\ast](https://latex.codecogs.com/png.latex?h%5E%5Cast "h^\\ast") is
a true density function, ![\\int\_\\mathbb{R} h^\\ast(t) \\mathrm{d} t
= 1](https://latex.codecogs.com/png.latex?%5Cint_%5Cmathbb%7BR%7D%20h%5E%5Cast%28t%29%20%5Cmathrm%7Bd%7D%20t%20%3D%201
"\\int_\\mathbb{R} h^\\ast(t) \\mathrm{d} t = 1"), called the .

Alternatively, the Hawkes process can be constructed as a poissonian
cluster process (Hawkes and Oakes 1974). The process consists of a flow
of *immigrants*, the cluster centres, arriving according to a
homogeneous Poisson process of intensity
![\\eta](https://latex.codecogs.com/png.latex?%5Ceta "\\eta"). Then, an
immigrant arriving at time
![T\_i](https://latex.codecogs.com/png.latex?T_i "T_i") generates
*children* according to an inhomogeneous Poisson process of intensity
![h(\\cdot -
T\_i)](https://latex.codecogs.com/png.latex?h%28%5Ccdot%20-%20T_i%29
"h(\\cdot - T_i)"). These in turn independently generate children
according to the same process, and so on *ad infinitum*. The processes
consisting of an immigrant and all its descendants are therefore
branching processes, and are independent of each other. Finally, the
process resulting from the superposition of all these branching
processes is a Hawkes process of conditional intensity
![\\lambda(\\cdot)](https://latex.codecogs.com/png.latex?%5Clambda%28%5Ccdot%29
"\\lambda(\\cdot)") (see the following figure of a realisation of an
exponential Hawkes process with ![\\\\eta
= 1](https://latex.codecogs.com/png.latex?%5C%5Ceta%20%3D%201
"\\\\eta = 1"), ![h(t) =
e^{-2t}](https://latex.codecogs.com/png.latex?h%28t%29%20%3D%20e%5E%7B-2t%7D
"h(t) = e^{-2t}")).

![Realisation of an exponential Hawkes process](man/figures/hawkes.png)

This package also supports non-causal Hawkes processes, for which the
reproduction kernel
![h^\\ast](https://latex.codecogs.com/png.latex?h%5E%5Cast "h^\\ast")
may take non-negative values on ![\\mathbb
R\_{\\le 0}](https://latex.codecogs.com/png.latex?%5Cmathbb%20R_%7B%5Cle%200%7D
"\\mathbb R_{\\le 0}"). Such processes are not defined through their
conditional intensity functions, but through the poisson cluster
representation, where each individual can potentially generate
offsprings both in the future and in the past.

The count sequence of a Hawkes process is the time series generated by
the event counts of the process, that is the series obtained by counting
the events of the process on intervals of fixed length. Formally, the
count sequence with bin size
![\\Delta](https://latex.codecogs.com/png.latex?%5CDelta "\\Delta")
associated to the point process
![N](https://latex.codecogs.com/png.latex?N "N") is the sequence
![(X\_k)\_{k \\in \\mathbb
Z}](https://latex.codecogs.com/png.latex?%28X_k%29_%7Bk%20%5Cin%20%5Cmathbb%20Z%7D
"(X_k)_{k \\in \\mathbb Z}").

# Estimation procedure

This package fully supports the parametric estimation of Hawkes
processes from their count sequences via minimisation of the Whittle
likelihood, and partially supports the estimation from the arrival times
via maximisation of the likelihood.

#### Maximum likelihood estimation

The parameters of a Hawkes process with arrival times ![(T\_i)\_{1\\le
i\\le
n}](https://latex.codecogs.com/png.latex?%28T_i%29_%7B1%5Cle%20i%5Cle%20n%7D
"(T_i)_{1\\le i\\le n}") on the interval ![\[0,
T\]](https://latex.codecogs.com/png.latex?%5B0%2C%20T%5D "[0, T]") can
be estimated by maximising the following pseudo-likelihood   
![\\mathcal L\_n(\\theta) = \\left( \\prod\_{i=1}^n \\lambda(T\_i)
\\right) \\exp \\left( - \\int\_0^T \\lambda(s) \\mathrm ds
\\right).](https://latex.codecogs.com/png.latex?%5Cmathcal%20L_n%28%5Ctheta%29%20%3D%20%5Cleft%28%20%5Cprod_%7Bi%3D1%7D%5En%20%5Clambda%28T_i%29%20%5Cright%29%20%5Cexp%20%5Cleft%28%20-%20%5Cint_0%5ET%20%5Clambda%28s%29%20%5Cmathrm%20ds%20%5Cright%29.
"\\mathcal L_n(\\theta) = \\left( \\prod_{i=1}^n \\lambda(T_i) \\right) \\exp \\left( - \\int_0^T \\lambda(s) \\mathrm ds \\right).")  
Note that each step of the optimisation is usually of complexity
![\\mathcal
O(n^2)](https://latex.codecogs.com/png.latex?%5Cmathcal%20O%28n%5E2%29
"\\mathcal O(n^2)").

#### Minimisation of the Whittle likelihood

Alternatively, the parameters of a Hawkes process with count sequence
![(X\_k)\_{1\\le k\\le
N}](https://latex.codecogs.com/png.latex?%28X_k%29_%7B1%5Cle%20k%5Cle%20N%7D
"(X_k)_{1\\le k\\le N}") can be estimated by minimising the log-spectral
(Whittle) likelihood   
![\\mathcal L\_N^\\prime(\\theta) = \\frac{1}{4\\pi} \\int\_{-\\pi}^\\pi
\\left( \\log f\_\\theta(\\omega) +
\\frac{I\_N(\\omega)}{f\_\\theta(\\omega)} \\right) \\mathrm
d\\omega,](https://latex.codecogs.com/png.latex?%5Cmathcal%20L_N%5E%5Cprime%28%5Ctheta%29%20%3D%20%5Cfrac%7B1%7D%7B4%5Cpi%7D%20%5Cint_%7B-%5Cpi%7D%5E%5Cpi%20%5Cleft%28%20%5Clog%20f_%5Ctheta%28%5Comega%29%20%2B%20%5Cfrac%7BI_N%28%5Comega%29%7D%7Bf_%5Ctheta%28%5Comega%29%7D%20%5Cright%29%20%5Cmathrm%20d%5Comega%2C
"\\mathcal L_N^\\prime(\\theta) = \\frac{1}{4\\pi} \\int_{-\\pi}^\\pi \\left( \\log f_\\theta(\\omega) + \\frac{I_N(\\omega)}{f_\\theta(\\omega)} \\right) \\mathrm d\\omega,")  
where ![I\_N(\\omega) = (2\\pi N)^{-1} \\left| \\sum\_{k=1}^N X\_k
\\exp(-ik\\omega)
\\right|^2](https://latex.codecogs.com/png.latex?I_N%28%5Comega%29%20%3D%20%282%5Cpi%20N%29%5E%7B-1%7D%20%5Cleft%7C%20%5Csum_%7Bk%3D1%7D%5EN%20X_k%20%5Cexp%28-ik%5Comega%29%20%5Cright%7C%5E2
"I_N(\\omega) = (2\\pi N)^{-1} \\left| \\sum_{k=1}^N X_k \\exp(-ik\\omega) \\right|^2")
denotes the periodogram of
![(X\_k)](https://latex.codecogs.com/png.latex?%28X_k%29 "(X_k)") and
![f\_\\theta(\\cdot)](https://latex.codecogs.com/png.latex?f_%5Ctheta%28%5Ccdot%29
"f_\\theta(\\cdot)") its spectral density function. Note that each step
of the optimisation is of complexity ![\\mathcal
O(n)](https://latex.codecogs.com/png.latex?%5Cmathcal%20O%28n%29
"\\mathcal O(n)") and the periodogram can be calculated in complexity
![\\mathcal O(n \\log
n)](https://latex.codecogs.com/png.latex?%5Cmathcal%20O%28n%20%5Clog%20n%29
"\\mathcal O(n \\log n)") using a Fast Fourier Transform algorithm,
making this method usually faster than maximum likelihood estimation.

# Main usage

This package supports both Hawkes processes and their count sequences.

## Simulation of Hawkes processes

Hawkes processes can be simulated by the function `hawkes`:

``` r
hawkes(end, fun, repr, family, M=null, ...)
```

where `end` denotes the end of the interval
![\[0,T\]](https://latex.codecogs.com/png.latex?%5B0%2CT%5D "[0,T]").
This function uses the cluster representation:

  - First, the immigrants are drawn according to a (potentially
    inhomogeneous) Poisson process with intensity measure `fun`.

  - Second, the number of offsprings of an immigrant are drawn from a
    Poisson distribution with intensity `repr`.

  - Third, these offsprings are distributed according to the `family`
    distribution.

  - Then, further offsprings are generated according to the last two
    steps.

The argument `fun` can take a numeric value or be specified as a
function, in which case the argument `M` must be specified as an upper
bound on `fun` (to allow simulation of the immigration process by
thinning). The argument `family` can either be specified as a string
`name` corresponding to a distribution with random generation function
`rname` (for example `exp` for `rexp`), or directly as a random
generation function. The optional arguments `...` are passed to the
random generation function specified by `family`.

This returns a list of class `hawkes`, whose realisations are stored in
the member `p`. Other members of the list are mainly used for support
functions.

#### Examples

  - Simulate a Hawkes process with immigration intensity ![\\eta
    = 1](https://latex.codecogs.com/png.latex?%5Ceta%20%3D%201
    "\\eta = 1"), reproduction mean ![\\mu
    = 0.5](https://latex.codecogs.com/png.latex?%5Cmu%20%3D%200.5
    "\\mu = 0.5") and reproduction kernel ![h^\\ast(t)
    = 2e^{-2t} 1\_{\\{t
    \\ge 0\\}}](https://latex.codecogs.com/png.latex?h%5E%5Cast%28t%29%20%3D%202e%5E%7B-2t%7D%201_%7B%5C%7Bt%20%5Cge%200%5C%7D%7D
    "h^\\ast(t) = 2e^{-2t} 1_{\\{t \\ge 0\\}}") on
    ![\[0, 10\]](https://latex.codecogs.com/png.latex?%5B0%2C%2010%5D
    "[0, 10]"):

<!-- end list -->

``` r
x <- hawkes(10, fun = 1, repr = 0.5, family = "exp", rate = 2)
```

  - Simulate a Hawkes process with inhomogeneous immigration process
    with intensity ![\\eta(t) = 1 +
    \\sin(t)](https://latex.codecogs.com/png.latex?%5Ceta%28t%29%20%3D%201%20%2B%20%5Csin%28t%29
    "\\eta(t) = 1 + \\sin(t)"), reproduction mean ![\\mu
    = 0.25](https://latex.codecogs.com/png.latex?%5Cmu%20%3D%200.25
    "\\mu = 0.25") and
    ![\[0,1\]](https://latex.codecogs.com/png.latex?%5B0%2C1%5D
    "[0,1]")-triangular reproduction kernel ![h^\\ast(t) = (1 -
    t)1\_{\\{0 \\le t
    \\le 1\\}}](https://latex.codecogs.com/png.latex?h%5E%5Cast%28t%29%20%3D%20%281%20-%20t%291_%7B%5C%7B0%20%5Cle%20t%20%5Cle%201%5C%7D%7D
    "h^\\ast(t) = (1 - t)1_{\\{0 \\le t \\le 1\\}}"):

<!-- end list -->

``` r
x <- hawkes(10, fun=function(y) {1+sin(y)}, M=2, repr=0.25,
            family=function(n) {1 - sqrt(1 - runif(n))})
```

#### Plot function

  - Hawkes processes can be plotted with the function `plot.hawkes`:

<!-- end list -->

``` r
plot.hawkes(x, intensity = FALSE, precision = 1e3, fun = NULL, repr = NULL, family = NULL, 
            M = NULL, ...)
```

If `x` is of class `hawkes`, as for objects returned by the function
`hawkes`, arguments `fun` through `M` can be ignored. If `intensity` is
set to `FALSE`, this plots the genealogy of the simulated Hawkes process
(as in preceding figure, bottom panel). If it is set to `TRUE`, this
plots the conditional intensity of the process (as in preceding figure,
top panel).

## Estimation of Hawkes processes

Two functions implement the estimation of Hawkes processes: `mle` from
arrival times ![(T\_i)](https://latex.codecogs.com/png.latex?%28T_i%29
"(T_i)") and `whittle` from count sequences
![(X\_k)](https://latex.codecogs.com/png.latex?%28X_k%29 "(X_k)"). While
the optimisation procedure rely on existing functions (see below),
calculations of both the usual and Whittle likelihood functions are done
in C++ via Rcpp (Eddelbuettel and François 2011) and RcppArmadillo
(Eddelbuettel and Sanderson 2014).

#### By maximum likelihood

The maximum likelihood method is implemented by the function

``` r
mle(events, kern, end, init = NULL, opts = NULL, ...)
```

`events` holds the arrival times
![(T\_i)](https://latex.codecogs.com/png.latex?%28T_i%29 "(T_i)") in
ascending order, `kern` must be a string (partially) matching one of the
reproduction kernels (see below) and `end` denotes the endpoint
![T](https://latex.codecogs.com/png.latex?T "T") of observation of the
process.

The optimisation of the maximum likelihood function is done by the
function `nloptr` from the package nloptr (Johnson, n.d.) with algorithm
L-BFGS-B, where the derivatives of the likelihood are calculated
explicitly. By default, parameters are constrained to be positive, and
additionally ![\\mu](https://latex.codecogs.com/png.latex?%5Cmu "\\mu")
is constrained to be below 1. However, both the arguments `opts` and
`...` are passed on to `nloptr`, so the algorithm, the constraints, or
any other parameter of the optimisation procedure can be changed.

Example of use:

``` r
x = hawkes(100, fun = 1, repr = .5, family = "exp", rate = 1)
mle(x$p, "Exponential", x$end)
```

#### By minimisation of the Whittle likelihood

The Whittle likelihood method is implemented by the function

``` r
whittle(counts, kern, binsize = NULL, trunc = 5L, init = NULL, ...)
```

`counts` holds the count sequence
![(X\_k)](https://latex.codecogs.com/png.latex?%28X_k%29 "(X_k)"),
`kern` must be a string (partially) matching one of the reproduction
kernels (see below), `binsize` denotes the bin size
![\\Delta](https://latex.codecogs.com/png.latex?%5CDelta "\\Delta") and
`trunc` is the number of foldings due to aliasing taken into account.

The optimisation of the Whittle likelihood function is done by the
function `optim`, with algorithm L-BFGS-B where the derivatives of the
likelihood are approximated by finite differences. By default,
parameters are constrained to be positive, and additionally
![\\mu](https://latex.codecogs.com/png.latex?%5Cmu "\\mu") is
constrained to be below 1. However, the argument `...` is passed to
`optim`, so any optimisation parameter can be changed.

Example of use:

``` r
x = hawkes(1000, fun = 1, repr = .5, family = "exp", rate = 1)
y = discrete(x, binsize = 1)
whittle(y, "Exponential", 1)
```

Note that `discrete` is a useful function to create the count sequence
![(X\_k)](https://latex.codecogs.com/png.latex?%28X_k%29 "(X_k)")
associated with an object `x` of class `hawkes`.

# Reproduction kernels

We introduce the reproduction kernels that are currently implemented in
this package. Recall that the Fourier transform of a reproduction kernel
is given by   
![\\widetilde {h^\\ast}(\\omega) = \\int\_\\mathbb R \\exp(-i\\omega t)
h(t) \\mathrm
dt,](https://latex.codecogs.com/png.latex?%5Cwidetilde%20%7Bh%5E%5Cast%7D%28%5Comega%29%20%3D%20%5Cint_%5Cmathbb%20R%20%5Cexp%28-i%5Comega%20t%29%20h%28t%29%20%5Cmathrm%20dt%2C
"\\widetilde {h^\\ast}(\\omega) = \\int_\\mathbb R \\exp(-i\\omega t) h(t) \\mathrm dt,")  
and that it is a Hermitian function ![\\widetilde {h^\\ast}(-\\omega) =
\\overline{\\widetilde
{h^\\ast}(\\omega)}](https://latex.codecogs.com/png.latex?%5Cwidetilde%20%7Bh%5E%5Cast%7D%28-%5Comega%29%20%3D%20%5Coverline%7B%5Cwidetilde%20%7Bh%5E%5Cast%7D%28%5Comega%29%7D
"\\widetilde {h^\\ast}(-\\omega) = \\overline{\\widetilde {h^\\ast}(\\omega)}").

## The exponential kernel

This is the exponential density function with parameter ![\\beta
\> 0](https://latex.codecogs.com/png.latex?%5Cbeta%20%3E%200
"\\beta \> 0"):   
![h^\\ast(t) = \\beta e^{-\\beta t} 1\_{\\{t
\> 0\\}}.](https://latex.codecogs.com/png.latex?h%5E%5Cast%28t%29%20%3D%20%5Cbeta%20e%5E%7B-%5Cbeta%20t%7D%201_%7B%5C%7Bt%20%3E%200%5C%7D%7D.
"h^\\ast(t) = \\beta e^{-\\beta t} 1_{\\{t \> 0\\}}.")  
Its Fourier transform is   
![\\widetilde {h^\\ast}(\\omega) = \\beta \\frac{1}{\\beta + i\\omega} =
\\beta \\frac{\\beta - i \\omega}{\\beta^2 +
\\omega^2}.](https://latex.codecogs.com/png.latex?%5Cwidetilde%20%7Bh%5E%5Cast%7D%28%5Comega%29%20%3D%20%5Cbeta%20%5Cfrac%7B1%7D%7B%5Cbeta%20%2B%20i%5Comega%7D%20%3D%20%5Cbeta%20%5Cfrac%7B%5Cbeta%20-%20i%20%5Comega%7D%7B%5Cbeta%5E2%20%2B%20%5Comega%5E2%7D.
"\\widetilde {h^\\ast}(\\omega) = \\beta \\frac{1}{\\beta + i\\omega} = \\beta \\frac{\\beta - i \\omega}{\\beta^2 + \\omega^2}.")  

The exponential kernel can be specified with the string `Exponential`
and the parameter ![\\beta](https://latex.codecogs.com/png.latex?%5Cbeta
"\\beta") with the usual argument `rate`. Both maximum and Whittle
likelihood methods are fully implemented for exponential kernels.
Moreover, the likelihood function is implemented in complexity
![\\mathcal
O(n)](https://latex.codecogs.com/png.latex?%5Cmathcal%20O%28n%29
"\\mathcal O(n)"), using the relations in (Ozaki and Ogata 1979).

## The symmetric exponential kernel

This is a symmetrised version of the exponential density function with
parameter ![\\beta
\> 0](https://latex.codecogs.com/png.latex?%5Cbeta%20%3E%200
"\\beta \> 0"):   
![h^\\ast(t) = \\frac 1 2 \\beta e^{-\\beta
|t|}.](https://latex.codecogs.com/png.latex?h%5E%5Cast%28t%29%20%3D%20%5Cfrac%201%202%20%5Cbeta%20e%5E%7B-%5Cbeta%20%7Ct%7C%7D.
"h^\\ast(t) = \\frac 1 2 \\beta e^{-\\beta |t|}.")  
Its Fourier transform is   
![\\widetilde {h^\\ast}(\\omega) = \\frac{\\beta^2}{\\beta^2 +
\\omega^2}.](https://latex.codecogs.com/png.latex?%5Cwidetilde%20%7Bh%5E%5Cast%7D%28%5Comega%29%20%3D%20%5Cfrac%7B%5Cbeta%5E2%7D%7B%5Cbeta%5E2%20%2B%20%5Comega%5E2%7D.
"\\widetilde {h^\\ast}(\\omega) = \\frac{\\beta^2}{\\beta^2 + \\omega^2}.")  

The symmetric exponential kernel can be specified with the string
`SymmetricExponential` and the parameter
![\\beta](https://latex.codecogs.com/png.latex?%5Cbeta "\\beta") with
the argument `rate`. Only the Whittle likelihood method is implemented
for symmetric exponential kernels. Note that it is a non-causal kernel,
as ![h^\\ast(t)
\\ne 0](https://latex.codecogs.com/png.latex?h%5E%5Cast%28t%29%20%5Cne%200
"h^\\ast(t) \\ne 0") for ![t
\< 0](https://latex.codecogs.com/png.latex?t%20%3C%200 "t \< 0").

## The gaussian kernel

This is the gaussian density function with mean ![\\nu \\in \\mathbb
R](https://latex.codecogs.com/png.latex?%5Cnu%20%5Cin%20%5Cmathbb%20R
"\\nu \\in \\mathbb R") and variance ![\\sigma^2
\> 0](https://latex.codecogs.com/png.latex?%5Csigma%5E2%20%3E%200
"\\sigma^2 \> 0"):   
![h^\\ast(t) = \\frac{1}{\\sigma
\\sqrt{2\\pi}}\\exp\\left(-\\frac{(t-\\nu)^2}{2\\sigma^2}\\right).](https://latex.codecogs.com/png.latex?h%5E%5Cast%28t%29%20%3D%20%5Cfrac%7B1%7D%7B%5Csigma%20%5Csqrt%7B2%5Cpi%7D%7D%5Cexp%5Cleft%28-%5Cfrac%7B%28t-%5Cnu%29%5E2%7D%7B2%5Csigma%5E2%7D%5Cright%29.
"h^\\ast(t) = \\frac{1}{\\sigma \\sqrt{2\\pi}}\\exp\\left(-\\frac{(t-\\nu)^2}{2\\sigma^2}\\right).")  
Its Fourier transform is   
![\\widetilde {h^\\ast}(\\omega) =
\\exp\\left(-\\frac{\\sigma^2\\omega^2}{2}-i\\nu\\omega\\right).](https://latex.codecogs.com/png.latex?%5Cwidetilde%20%7Bh%5E%5Cast%7D%28%5Comega%29%20%3D%20%5Cexp%5Cleft%28-%5Cfrac%7B%5Csigma%5E2%5Comega%5E2%7D%7B2%7D-i%5Cnu%5Comega%5Cright%29.
"\\widetilde {h^\\ast}(\\omega) = \\exp\\left(-\\frac{\\sigma^2\\omega^2}{2}-i\\nu\\omega\\right).")  

The gaussian kernel can be specified with the string `Gaussian` and its
parameters ![\\nu](https://latex.codecogs.com/png.latex?%5Cnu "\\nu")
and ![\\sigma](https://latex.codecogs.com/png.latex?%5Csigma "\\sigma")
with the usual arguments `mean` and `sd` respectively. Only the Whittle
likelihood method is implemented for gaussian kernels. Note that it is a
non-causal kernel, as ![h^\\ast(t)
\\ne 0](https://latex.codecogs.com/png.latex?h%5E%5Cast%28t%29%20%5Cne%200
"h^\\ast(t) \\ne 0") for ![t
\< 0](https://latex.codecogs.com/png.latex?t%20%3C%200 "t \< 0").

## The power law kernel

This is a normalised and shifted power law function, with shape
![\\theta \> 0](https://latex.codecogs.com/png.latex?%5Ctheta%20%3E%200
"\\theta \> 0") and scale ![a
\> 0](https://latex.codecogs.com/png.latex?a%20%3E%200 "a \> 0"):   
![h^\\ast(t) = \\theta a^\\theta (t+a)^{-\\theta-1} 1\_{\\{\\theta \> 0
\\}}.](https://latex.codecogs.com/png.latex?h%5E%5Cast%28t%29%20%3D%20%5Ctheta%20a%5E%5Ctheta%20%28t%2Ba%29%5E%7B-%5Ctheta-1%7D%201_%7B%5C%7B%5Ctheta%20%3E%200%20%5C%7D%7D.
"h^\\ast(t) = \\theta a^\\theta (t+a)^{-\\theta-1} 1_{\\{\\theta \> 0 \\}}.")  
For positive ![\\omega](https://latex.codecogs.com/png.latex?%5Comega
"\\omega"), its Fourier transform is given by   
![\\widetilde{h\_\\theta^\\ast}(\\omega) = \\theta \\exp(i\\omega
a)E\_{\\theta + 1} (i\\omega
a),](https://latex.codecogs.com/png.latex?%5Cwidetilde%7Bh_%5Ctheta%5E%5Cast%7D%28%5Comega%29%20%3D%20%5Ctheta%20%5Cexp%28i%5Comega%20a%29E_%7B%5Ctheta%20%2B%201%7D%20%28i%5Comega%20a%29%2C
"\\widetilde{h_\\theta^\\ast}(\\omega) = \\theta \\exp(i\\omega a)E_{\\theta + 1} (i\\omega a),")  
where
![E\_{\\theta}(ix)](https://latex.codecogs.com/png.latex?E_%7B%5Ctheta%7D%28ix%29
"E_{\\theta}(ix)") denotes the integral   
![E\_{\\theta}(ix) = \\int\_1^\\infty t^{-\\theta} \\exp(-ixt) \\mathrm
dt.](https://latex.codecogs.com/png.latex?E_%7B%5Ctheta%7D%28ix%29%20%3D%20%5Cint_1%5E%5Cinfty%20t%5E%7B-%5Ctheta%7D%20%5Cexp%28-ixt%29%20%5Cmathrm%20dt.
"E_{\\theta}(ix) = \\int_1^\\infty t^{-\\theta} \\exp(-ixt) \\mathrm dt.")  
With successive integration by parts, this integral can be related to
![E\_{\\theta'}(ix)](https://latex.codecogs.com/png.latex?E_%7B%5Ctheta%27%7D%28ix%29
"E_{\\theta'}(ix)"), with ![0 \< \\theta'
\\le 1](https://latex.codecogs.com/png.latex?0%20%3C%20%5Ctheta%27%20%5Cle%201
"0 \< \\theta' \\le 1").

If ![\\theta'
= 1](https://latex.codecogs.com/png.latex?%5Ctheta%27%20%3D%201
"\\theta' = 1") or equivalently ![\\theta \\in \\mathbb
N\_{\\ne 0}](https://latex.codecogs.com/png.latex?%5Ctheta%20%5Cin%20%5Cmathbb%20N_%7B%5Cne%200%7D
"\\theta \\in \\mathbb N_{\\ne 0}"), the integral
![E\_1(ix)](https://latex.codecogs.com/png.latex?E_1%28ix%29 "E_1(ix)"),
called the exponential integral with imaginary argument, can be related
the trigonometric integrals and calculated using Padé approximants (Rowe
et al. 2015, Appendix B), accurate to better than
![10^{-16}](https://latex.codecogs.com/png.latex?10%5E%7B-16%7D
"10^{-16}"). The function `E1_imaginary` implements this approximation.

If ![\\theta'
\< 1](https://latex.codecogs.com/png.latex?%5Ctheta%27%20%3C%201
"\\theta' \< 1") or equivalently ![\\theta \\in \\mathbb R \\setminus
\\mathbb
N](https://latex.codecogs.com/png.latex?%5Ctheta%20%5Cin%20%5Cmathbb%20R%20%5Csetminus%20%5Cmathbb%20N
"\\theta \\in \\mathbb R \\setminus \\mathbb N"), the integral
![E\_\\theta(ix)](https://latex.codecogs.com/png.latex?E_%5Ctheta%28ix%29
"E_\\theta(ix)") can be related to the incomplete gamma function with
imaginary argument   
![\\Gamma\_i(x, \\alpha) = \\int\_x^\\infty t^{\\alpha-1} e^{-it}
\\mathrm{d}t,](https://latex.codecogs.com/png.latex?%5CGamma_i%28x%2C%20%5Calpha%29%20%3D%20%5Cint_x%5E%5Cinfty%20t%5E%7B%5Calpha-1%7D%20e%5E%7B-it%7D%20%5Cmathrm%7Bd%7Dt%2C
"\\Gamma_i(x, \\alpha) = \\int_x^\\infty t^{\\alpha-1} e^{-it} \\mathrm{d}t,")  
where ![0 \< \\alpha
\< 1](https://latex.codecogs.com/png.latex?0%20%3C%20%5Calpha%20%3C%201
"0 \< \\alpha \< 1"). We implemented Taylor approximations of this
integral, accurate to better than
![10^{-7}](https://latex.codecogs.com/png.latex?10%5E%7B-7%7D
"10^{-7}"), in the function `inc_gamma_imag`.

The power law kernel can be specified with the string `PowerLaw` and its
parameters ![\\theta](https://latex.codecogs.com/png.latex?%5Ctheta
"\\theta") and ![a](https://latex.codecogs.com/png.latex?a "a") with the
arguments `shape` and `scale` respectively. Both maximum and Whittle
likelihood methods are implemented for power law kernels.

## The Pareto kernels

This is the Pareto density function with shape ![\\theta
\> 0](https://latex.codecogs.com/png.latex?%5Ctheta%20%3E%200
"\\theta \> 0") and scale ![a
\> 0](https://latex.codecogs.com/png.latex?a%20%3E%200 "a \> 0"):   
![h\_\\theta^\\ast(t) = \\theta a^\\theta t^{-\\theta - 1} 1\_{\\{t \>
a\\}}.](https://latex.codecogs.com/png.latex?h_%5Ctheta%5E%5Cast%28t%29%20%3D%20%5Ctheta%20a%5E%5Ctheta%20t%5E%7B-%5Ctheta%20-%201%7D%201_%7B%5C%7Bt%20%3E%20a%5C%7D%7D.
"h_\\theta^\\ast(t) = \\theta a^\\theta t^{-\\theta - 1} 1_{\\{t \> a\\}}.")  
For positive ![\\omega](https://latex.codecogs.com/png.latex?%5Comega
"\\omega"), its Fourier transform is given by   
![\\widetilde{h\_\\theta^\\ast}(\\omega) = \\theta E\_{\\theta + 1}
(i\\omega
a),](https://latex.codecogs.com/png.latex?%5Cwidetilde%7Bh_%5Ctheta%5E%5Cast%7D%28%5Comega%29%20%3D%20%5Ctheta%20E_%7B%5Ctheta%20%2B%201%7D%20%28i%5Comega%20a%29%2C
"\\widetilde{h_\\theta^\\ast}(\\omega) = \\theta E_{\\theta + 1} (i\\omega a),")  
see above for the definition of
![E\_\\theta(ix)](https://latex.codecogs.com/png.latex?E_%5Ctheta%28ix%29
"E_\\theta(ix)").

Only Pareto kernels with fixed ![\\theta
= 1](https://latex.codecogs.com/png.latex?%5Ctheta%20%3D%201
"\\theta = 1"), ![2](https://latex.codecogs.com/png.latex?2 "2"), and
![3](https://latex.codecogs.com/png.latex?3 "3") have been implemented
and can specified with the strings `Pareto1`, `Pareto2` and `Pareto3`
respectively, with parameter ![a](https://latex.codecogs.com/png.latex?a
"a") specified with the argument `scale`. Only the Whittle method is
available for Pareto kernels.

# To be implemented

  - Improve this vignette: it is currently too sparse and functions of
    the package could need some better description.

  - Add some real datasets to the package: real life case-studies with
    good datasets help understand the functionalities of a package.

  - Variance and confidence intervals for the estimation with function
    `whittle`: note that currently, the variance-covariance matrix
    returned by the optimisation method in function `whittle` is not
    accurate, as it ignores the dependence within the count sequence
    ![(X\_k)](https://latex.codecogs.com/png.latex?%28X_k%29 "(X_k)").

  - Diagnostics for the estimated model: spectral density based
    goodness-of-fit tests are to be implemented for the estimated Hawkes
    processes, based on the work of (Paparoditis 2000).

  - Custom built-kernels: allow the user to input reproduction kernels
    that are not already implemented.

<!-- Bibliographie -->

# References

<div id="refs" class="references">

<div id="ref-Cheysson2020">

Cheysson, Felix, and Gabriel Lang. 2020. “Strong mixing condition for
Hawkes processes and application to Whittle estimation from count data.”
*arXiv*, March. <https://arxiv.org/abs/2003.04314>.

</div>

<div id="ref-Eddelbuettel2011">

Eddelbuettel, Dirk, and Romain François. 2011. “Rcpp : Seamless R and
C++ Integration.” *J. Stat. Softw.* 40 (8): 1–18.
<https://doi.org/10.18637/jss.v040.i08>.

</div>

<div id="ref-RcppArmadillo">

Eddelbuettel, Dirk, and Conrad Sanderson. 2014. “RcppArmadillo:
Accelerating R with High-Performance C++ Linear Algebra.” *Computational
Statistics and Data Analysis* 71: 1054–63.
<http://dx.doi.org/10.1016/j.csda.2013.02.005>.

</div>

<div id="ref-Hawkes1971a">

Hawkes, Alan G. 1971. “Spectra of Some Self-Exciting and Mutually
Exciting Point Processes.” *Biometrika* 58 (1): 83–90.
<https://doi.org/10.2307/2334319>.

</div>

<div id="ref-Hawkes1974">

Hawkes, Alan G., and David Oakes. 1974. “A cluster process
representation of a self-exciting process.” *J. Appl. Probab.* 11 (03):
493–503. <https://doi.org/10.2307/3212693>.

</div>

<div id="ref-Johnson">

Johnson, Steven G. n.d. “The NLopt nonlinear-optimization package.”

</div>

<div id="ref-Ozaki1979">

Ozaki, T., and Yosihiko Ogata. 1979. “Maximum likelihood estimation of
Hawkes’ self-exciting point processes.” *Ann. Inst. Stat. Math.* 31 (1):
145–55. <https://doi.org/10.1007/BF02480272>.

</div>

<div id="ref-Paparoditis2000">

Paparoditis, Efstathios. 2000. “Spectral density based goodness-of-fit
tests for time series models.” *Scand. J. Stat.* 27 (1): 143–76.
<https://doi.org/10.1111/1467-9469.00184>.

</div>

<div id="ref-Rowe2015">

Rowe, B. T. P., M. Jarvis, R. Mandelbaum, G. M. Bernstein, J. Bosch, M.
Simet, J. E. Meyers, et al. 2015. “GalSim: The modular galaxy image
simulation toolkit.” *Astron. Comput.* 10: 121–50.
<https://doi.org/10.1016/j.ascom.2015.02.002>.

</div>

</div>
