% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/format_data.R
\name{fmt_date}
\alias{fmt_date}
\title{Format values as dates}
\usage{
fmt_date(
  data,
  columns = everything(),
  rows = everything(),
  date_style = "iso",
  pattern = "{x}",
  locale = NULL
)
}
\arguments{
\item{data}{A table object that is created using the \code{\link[=gt]{gt()}} function.}

\item{columns}{The columns to format. Can either be a series of column names
provided in \code{\link[=c]{c()}}, a vector of column indices, or a helper function
focused on selections. The select helper functions are: \code{\link[=starts_with]{starts_with()}},
\code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}}, \code{\link[=matches]{matches()}}, \code{\link[=one_of]{one_of()}}, \code{\link[=num_range]{num_range()}}, and
\code{\link[=everything]{everything()}}.}

\item{rows}{Optional rows to format. Providing \code{\link[=everything]{everything()}} (the
default) results in all rows in \code{columns} being formatted. Alternatively,
we can supply a vector of row captions within \code{\link[=c]{c()}}, a vector of row
indices, or a helper function focused on selections. The select helper
functions are: \code{\link[=starts_with]{starts_with()}}, \code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}}, \code{\link[=matches]{matches()}},
\code{\link[=one_of]{one_of()}}, \code{\link[=num_range]{num_range()}}, and \code{\link[=everything]{everything()}}. We can also use expressions
to filter down to the rows we need (e.g.,
\verb{[colname_1] > 100 & [colname_2] < 50}).}

\item{date_style}{The date style to use. By default this is \code{"iso"} which
corresponds to ISO 8601 date formatting. The other date styles can be
viewed using \code{\link[=info_date_style]{info_date_style()}}.}

\item{pattern}{A formatting pattern that allows for decoration of the
formatted value. The value itself is represented by \code{{x}} and all other
characters are taken to be string literals.}

\item{locale}{An optional locale identifier that can be used for formatting
the value according the locale's rules. Examples include \code{"en"} for English
(United States) and \code{"fr"} for French (France). The use of a locale ID will
override any locale-specific values provided. We can use the
\code{\link[=info_locales]{info_locales()}} function as a useful reference for all of the locales that
are supported.}
}
\value{
An object of class \code{gt_tbl}.
}
\description{
Format input values to time values using one of 41 preset date styles. Input
can be in the form of \code{POSIXt} (i.e., datetimes), the \code{Date} type, or
\code{character} (must be in the ISO 8601 form of \verb{YYYY-MM-DD HH:MM:SS} or
\code{YYYY-MM-DD}).
}
\section{Compatibility of formatting function with data values}{


The \code{fmt_date()} formatting function is compatible with body cells that are
of the \code{"Date"}, \code{"POSIXt"} or \code{"character"} types. Any other types of body
cells are ignored during formatting. This is to say that cells of
incompatible data types may be targeted, but there will be no attempt to
format them.
}

\section{Targeting cells with \code{columns} and \code{rows}}{


Targeting of values is done through \code{columns} and additionally by \code{rows} (if
nothing is provided for \code{rows} then entire columns are selected). The
\code{columns} argument allows us to target a subset of cells contained in the
resolved columns. We say resolved because aside from declaring column names
in \code{c()} (with bare column names or names in quotes) we can use
\strong{tidyselect}-style expressions. This can be as basic as supplying a select
helper like \code{starts_with()}, or, providing a more complex incantation like

\code{where(~ is.numeric(.x) && max(.x, na.rm = TRUE) > 1E6)}

which targets numeric columns that have a maximum value greater than
1,000,000 (excluding any \code{NA}s from consideration).

By default all columns and rows are selected (with the \code{everything()}
defaults). Cell values that are incompatible with a given formatting function
will be skipped over, like \code{character} values and numeric \verb{fmt_*()}
functions. So it's safe to select all columns with a particular formatting
function (only those values that can be formatted will be formatted), but,
you may not want that. One strategy is to format the bulk of cell values with
one formatting function and then constrain the columns for later passes with
other types of formatting (the last formatting done to a cell is what you get
in the final output).

Once the columns are targeted, we may also target the \code{rows} within those
columns. This can be done in a variety of ways. If a stub is present, then we
potentially have row identifiers. Those can be used much like column names in
the \code{columns}-targeting scenario. We can use simpler \strong{tidyselect}-style
expressions (the select helpers should work well here) and we can use quoted
row identifiers in \code{c()}. It's also possible to use row indices (e.g.,
\code{c(3, 5, 6)}) though these index values must correspond to the row numbers of
the input data (the indices won't necessarily match those of rearranged rows
if row groups are present). One more type of expression is possible, an
expression that takes column values (can involve any of the available columns
in the table) and returns a logical vector. This is nice if you want to base
formatting on values in the column or another column, or, you'd like to use a
more complex predicate expression.
}

\section{Formatting with the \code{date_style} argument}{


We need to supply a preset date style to the \code{date_style} argument. The date
styles are numerous and can handle localization to any supported locale. A
large segment of date styles are termed flexible date formats and this means
that their output will adapt to any \code{locale} provided. That feature makes the
flexible date formats a better option for locales other than \code{"en"} (the
default locale).

The following table provides a listing of all date styles and their output
values (corresponding to an input date of \code{2000-02-29}).\tabular{llll}{
    \tab Date Style \tab Output \tab Notes \cr
   1 \tab \code{"iso"} \tab \code{"2000-02-29"} \tab ISO 8601 \cr
   2 \tab \code{"wday_month_day_year"} \tab \code{"Tuesday, February 29, 2000"} \tab  \cr
   3 \tab \code{"wd_m_day_year"} \tab \code{"Tue, Feb 29, 2000"} \tab  \cr
   4 \tab \code{"wday_day_month_year"} \tab \code{"Tuesday 29 February 2000"} \tab  \cr
   5 \tab \code{"month_day_year"} \tab \code{"February 29, 2000"} \tab  \cr
   6 \tab \code{"m_day_year"} \tab \code{"Feb 29, 2000"} \tab  \cr
   7 \tab \code{"day_m_year"} \tab \code{"29 Feb 2000"} \tab  \cr
   8 \tab \code{"day_month_year"} \tab \code{"29 February 2000"} \tab  \cr
   9 \tab \code{"day_month"} \tab \code{"29 February"} \tab  \cr
   10 \tab \code{"day_m"} \tab \code{"29 Feb"} \tab  \cr
   11 \tab \code{"year"} \tab \code{"2000"} \tab  \cr
   12 \tab \code{"month"} \tab \code{"February"} \tab  \cr
   13 \tab \code{"day"} \tab \code{"29"} \tab  \cr
   14 \tab \code{"year.mn.day"} \tab \code{"2000/02/29"} \tab  \cr
   15 \tab \code{"y.mn.day"} \tab \code{"00/02/29"} \tab  \cr
   16 \tab \code{"year_week"} \tab \code{"2000-W09"} \tab  \cr
   17 \tab \code{"year_quarter"} \tab \code{"2000-Q1"} \tab  \cr
   18 \tab \code{"yMd"} \tab \code{"2/29/2000"} \tab flexible \cr
   19 \tab \code{"yMEd"} \tab \code{"Tue, 2/29/2000"} \tab flexible \cr
   20 \tab \code{"yMMM"} \tab \code{"Feb 2000"} \tab flexible \cr
   21 \tab \code{"yMMMM"} \tab \code{"February 2000"} \tab flexible \cr
   22 \tab \code{"yMMMd"} \tab \code{"Feb 29, 2000"} \tab flexible \cr
   23 \tab \code{"yMMMEd"} \tab \code{"Tue, Feb 29, 2000"} \tab flexible \cr
   24 \tab \code{"GyMd"} \tab \code{"2/29/2000 A"} \tab flexible \cr
   25 \tab \code{"GyMMMd"} \tab \code{"Feb 29, 2000 AD"} \tab flexible \cr
   26 \tab \code{"GyMMMEd"} \tab \code{"Tue, Feb 29, 2000 AD"} \tab flexible \cr
   27 \tab \code{"yM"} \tab \code{"2/2000"} \tab flexible \cr
   28 \tab \code{"Md"} \tab \code{"2/29"} \tab flexible \cr
   29 \tab \code{"MEd"} \tab \code{"Tue, 2/29"} \tab flexible \cr
   30 \tab \code{"MMMd"} \tab \code{"Feb 29"} \tab flexible \cr
   31 \tab \code{"MMMEd"} \tab \code{"Tue, Feb 29"} \tab flexible \cr
   32 \tab \code{"MMMMd"} \tab \code{"February 29"} \tab flexible \cr
   33 \tab \code{"GyMMM"} \tab \code{"Feb 2000 AD"} \tab flexible \cr
   34 \tab \code{"yQQQ"} \tab \code{"Q1 2000"} \tab flexible \cr
   35 \tab \code{"yQQQQ"} \tab \code{"1st quarter 2000"} \tab flexible \cr
   36 \tab \code{"Gy"} \tab \code{"2000 AD"} \tab flexible \cr
   37 \tab \code{"y"} \tab \code{"2000"} \tab flexible \cr
   38 \tab \code{"M"} \tab \code{"2"} \tab flexible \cr
   39 \tab \code{"MMM"} \tab \code{"Feb"} \tab flexible \cr
   40 \tab \code{"d"} \tab \code{"29"} \tab flexible \cr
   41 \tab \code{"Ed"} \tab \code{"29 Tue"} \tab flexible \cr
}


We can use the \code{\link[=info_date_style]{info_date_style()}} function within the console to view a
similar table of date styles with example output.
}

\section{Adapting output to a specific \code{locale}}{


This formatting function can adapt outputs according to a provided \code{locale}
value. Examples include \code{"en"} for English (United States) and \code{"fr"} for
French (France). Note that a \code{locale} value provided here will override any
global locale setting performed in \code{\link[=gt]{gt()}}'s own \code{locale} argument (it is
settable there as a value received by all other functions that have a
\code{locale} argument). As a useful reference on which locales are supported, we
can use the \code{\link[=info_locales]{info_locales()}} function to view an info table.
}

\section{Examples}{


Use \code{\link{exibble}} to create a \strong{gt} table. Keep only the \code{date} and \code{time}
columns. Format the \code{date} column to have dates formatted with the
\code{"month_day_year"} date style.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{exibble |>
  dplyr::select(date, time) |>
  gt() |>
  fmt_date(
    columns = date,
    date_style = "month_day_year"
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_fmt_date_1.png" alt="This image of a table was generated from the first code example in the `fmt_date()` help file." style="width:100\%;">
}}

Use \code{\link{exibble}} to create a \strong{gt} table. Keep only the \code{date} and \code{time}
columns. Format the \code{date} column to have mixed date formats (dates after
April will be different than the others because of the expressions used
in the \code{rows} argument).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{exibble |>
  dplyr::select(date, time) |>
  gt() |>
  fmt_date(
    columns = date,
    rows = as.Date(date) > as.Date("2015-04-01"),
    date_style = "m_day_year"
  ) |>
  fmt_date(
    columns = date,
    rows = as.Date(date) <= as.Date("2015-04-01"),
    date_style = "day_m_year"
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_fmt_date_2.png" alt="This image of a table was generated from the second code example in the `fmt_date()` help file." style="width:100\%;">
}}

Use \code{\link{exibble}} to create another \strong{gt} table, this time only with the
\code{date} column. Format the \code{date} column to use the \code{"yMMMEd"} date style
(which is one of the 'flexible' styles). Also, set the locale to \code{"nl"} to
get the dates in Dutch.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{exibble |>
  dplyr::select(date) |>
  gt() |>
  fmt_date(
    columns = date,
    date_style = "yMMMEd",
    locale = "nl"
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_fmt_date_3.png" alt="This image of a table was generated from the third code example in the `fmt_date()` help file." style="width:100\%;">
}}
}

\section{Function ID}{

3-13
}

\section{Function Introduced}{

\code{v0.2.0.5} (March 31, 2020)
}

\seealso{
The vector-formatting version of this function: \code{\link[=vec_fmt_date]{vec_fmt_date()}}.

Other data formatting functions: 
\code{\link{data_color}()},
\code{\link{fmt_auto}()},
\code{\link{fmt_bins}()},
\code{\link{fmt_bytes}()},
\code{\link{fmt_currency}()},
\code{\link{fmt_datetime}()},
\code{\link{fmt_duration}()},
\code{\link{fmt_engineering}()},
\code{\link{fmt_flag}()},
\code{\link{fmt_fraction}()},
\code{\link{fmt_image}()},
\code{\link{fmt_index}()},
\code{\link{fmt_integer}()},
\code{\link{fmt_markdown}()},
\code{\link{fmt_number}()},
\code{\link{fmt_partsper}()},
\code{\link{fmt_passthrough}()},
\code{\link{fmt_percent}()},
\code{\link{fmt_roman}()},
\code{\link{fmt_scientific}()},
\code{\link{fmt_spelled_num}()},
\code{\link{fmt_time}()},
\code{\link{fmt_url}()},
\code{\link{fmt}()},
\code{\link{sub_large_vals}()},
\code{\link{sub_missing}()},
\code{\link{sub_small_vals}()},
\code{\link{sub_values}()},
\code{\link{sub_zero}()}
}
\concept{data formatting functions}
