% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hilbert.R
\name{hilbert}
\alias{hilbert}
\title{Hilbert transform}
\usage{
hilbert(x, n = ifelse(is.vector(x), length(x), nrow(x)))
}
\arguments{
\item{x}{Input array, specified as a vector or a matrix. In case of a matrix,
the Hilbert transform of all columns is computed.}

\item{n}{use an n-point FFT to compute the Hilbert transform. The input data
is zero-padded or truncated to length n, as appropriate.}
}
\value{
Analytic signal, of length \code{n}, returned as a complex vector or
  matrix, the real part of which contains the original signal, and the
  imaginary part of which contains the Hilbert transform of \code{x}.
}
\description{
Computes the extension of a real valued signal to an analytic signal.
}
\details{
The function returns returns a complex helical sequence, sometimes called the
analytic signal, from a real data sequence. The analytic signal has a real
part, which is the original data, and an imaginary part, which contains the
Hilbert transform. The imaginary part is a version of the original real
sequence with a 90 degrees phase shift. Sines are therefore transformed to
cosines, and conversely, cosines are transformed to sines. The
Hilbert-transformed series has the same amplitude and frequency content as
the original sequence. The transform includes phase information that depends
on the phase of the original.
}
\examples{
## notice that the imaginary signal is phase-shifted 90 degrees
t <- seq(0, 10, length = 256)
z <- hilbert(sin(2 * pi * 0.5 * t))
plot(t, Re(z), type = "l", col="blue")
lines (t, Im(z), col = "red")
legend('topright', lty = 1, legend = c("Real", "Imag"),
       col = c("blue", "red"))

## the magnitude of the hilbert transform eliminates the carrier
t <- seq(0, 10, length = 1024)
x <- 5 * cos(0.2 * t) * sin(100 * t)
plot(t, x, type = "l", col = "green")
lines (t, abs(hilbert(x)), col = "blue")
legend('topright', lty = 1, legend = c("x", "|hilbert(x)|"),
        col = c("green", "blue"))

}
\references{
\url{https://en.wikipedia.org/wiki/Hilbert_transform},
  \url{https://en.wikipedia.org/wiki/Analytic_signal}
}
\author{
Paul Kienzle, \email{pkienzle@users.sf.net},\cr
 Peter L. Soendergaard.\cr
 Conversion to R by Geert van Boxtel, \email{gjmvanboxtel@gmail.com}
}
