% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/causal_survival_forest.R
\name{causal_survival_forest}
\alias{causal_survival_forest}
\title{Causal survival forest}
\usage{
causal_survival_forest(
  X,
  Y,
  W,
  D,
  W.hat = NULL,
  target = c("RMST", "survival.probability"),
  horizon = NULL,
  failure.times = NULL,
  num.trees = 2000,
  sample.weights = NULL,
  clusters = NULL,
  equalize.cluster.weights = FALSE,
  sample.fraction = 0.5,
  mtry = min(ceiling(sqrt(ncol(X)) + 20), ncol(X)),
  min.node.size = 5,
  honesty = TRUE,
  honesty.fraction = 0.5,
  honesty.prune.leaves = TRUE,
  alpha = 0.05,
  imbalance.penalty = 0,
  stabilize.splits = TRUE,
  ci.group.size = 2,
  tune.parameters = "none",
  compute.oob.predictions = TRUE,
  num.threads = NULL,
  seed = runif(1, 0, .Machine$integer.max)
)
}
\arguments{
\item{X}{The covariates.}

\item{Y}{The event time (must be non-negative).}

\item{W}{The treatment assignment (must be a binary vector with no NAs).}

\item{D}{The event type (0: censored, 1: failure).}

\item{W.hat}{Estimates of the treatment propensities E[W | X = x]. If W.hat = NULL,
these are estimated using a separate regression forest. Default is NULL.}

\item{target}{The target estimand. Choices are Restricted Mean Survival Time ("RMST") which estimates 1)
E[min(T(1), horizon) - min(T(0), horizon) | X = x], or "survival.probability" which estimates 2)
P(T(1) > horizon | X = x) - P(T(0) > horizon | X = x). Default is "RMST".}

\item{horizon}{A scalar that defines the estimand (required). If target is "RMST" then this defines
the maximum follow-up time. If target is "survival.probability", then this defines the time point
for the absolute risk difference estimate.}

\item{failure.times}{A vector of event times to fit the survival curves at. If NULL, then all the unique
event times are used. This speeds up forest estimation by constraining the event grid. Observed event
times are rounded down to the last sorted occurance less than or equal to the specified failure time.
The time points should be in increasing order.
Default is NULL.}

\item{num.trees}{Number of trees grown in the forest. Note: Getting accurate
confidence intervals generally requires more trees than
getting accurate predictions. Default is 2000.}

\item{sample.weights}{Weights given to each sample in estimation.
If NULL, each observation receives the same weight.
Note: To avoid introducing confounding, weights should be
independent of the potential outcomes given X. Sample weights
are not used in survival spliting. Default is NULL.}

\item{clusters}{Vector of integers or factors specifying which cluster each observation corresponds to.
Default is NULL (ignored).}

\item{equalize.cluster.weights}{If FALSE, each unit is given the same weight (so that bigger
clusters get more weight). If TRUE, each cluster is given equal weight in the forest. In this case,
during training, each tree uses the same number of observations from each drawn cluster: If the
smallest cluster has K units, then when we sample a cluster during training, we only give a random
K elements of the cluster to the tree-growing procedure. When estimating average treatment effects,
each observation is given weight 1/cluster size, so that the total weight of each cluster is the
same. Note that, if this argument is FALSE, sample weights may also be directly adjusted via the
sample.weights argument. If this argument is TRUE, sample.weights must be set to NULL. Default is
FALSE.}

\item{sample.fraction}{Fraction of the data used to build each tree.
Note: If honesty = TRUE, these subsamples will
further be cut by a factor of honesty.fraction. Default is 0.5.}

\item{mtry}{Number of variables tried for each split. Default is
\eqn{\sqrt p + 20} where p is the number of variables.}

\item{min.node.size}{A target for the minimum number of observations in each tree leaf. Note that nodes
with size smaller than min.node.size can occur, as in the original randomForest package.
Default is 5.}

\item{honesty}{Whether to use honest splitting (i.e., sub-sample splitting). Default is TRUE.
For a detailed description of honesty, honesty.fraction, honesty.prune.leaves, and recommendations for
parameter tuning, see the grf algorithm reference.}

\item{honesty.fraction}{The fraction of data that will be used for determining splits if honesty = TRUE. Corresponds
to set J1 in the notation of the paper. Default is 0.5 (i.e. half of the data is used for
determining splits).}

\item{honesty.prune.leaves}{If TRUE, prunes the estimation sample tree such that no leaves
are empty. If FALSE, keep the same tree as determined in the splits sample (if an empty leave is encountered, that
tree is skipped and does not contribute to the estimate). Setting this to FALSE may improve performance on
small/marginally powered data, but requires more trees (note: tuning does not adjust the number of trees).
Only applies if honesty is enabled. Default is TRUE.}

\item{alpha}{A tuning parameter that controls the maximum imbalance of a split. This parameter plays the same
role as in causal forest and survival forest, where for the latter the number of failures in
each child has to be at least one or `alpha` times the number of samples in the parent node. Default is 0.05.
(On data with very low event rate the default value may be too high for the forest to split
and lowering it may be beneficial).}

\item{imbalance.penalty}{A tuning parameter that controls how harshly imbalanced splits are penalized. Default is 0.}

\item{stabilize.splits}{Whether or not the treatment and censoring status should be taken into account when
determining the imbalance of a split. The requirement for valid split candidates is the same as in causal_forest
with the additional constraint that num.failures(child) >= num.samples(parent) * alpha. Default is TRUE.}

\item{ci.group.size}{The forest will grow ci.group.size trees on each subsample.
In order to provide confidence intervals, ci.group.size must
be at least 2. Default is 2.}

\item{tune.parameters}{(Currently only applies to the regression forest used in W.hat estimation)
A vector of parameter names to tune.
If "all": all tunable parameters are tuned by cross-validation. The following parameters are
tunable: ("sample.fraction", "mtry", "min.node.size", "honesty.fraction",
 "honesty.prune.leaves", "alpha", "imbalance.penalty"). If honesty is FALSE the honesty.* parameters are not tuned.
Default is "none" (no parameters are tuned).}

\item{compute.oob.predictions}{Whether OOB predictions on training set should be precomputed. Default is TRUE.}

\item{num.threads}{Number of threads used in training. By default, the number of threads is set
to the maximum hardware concurrency.}

\item{seed}{The seed of the C++ random number generator.}
}
\value{
A trained causal_survival_forest forest object.
}
\description{
Trains a causal survival forest that can be used to estimate
conditional treatment effects tau(X) with right-censored outcomes.
We estimate either 1)
tau(X) = E[min(T(1), horizon) - min(T(0), horizon) | X = x],
where T(1) and T(0) are potental outcomes corresponding to the two possible treatment states
and `horizon` is the maximum follow-up time, or 2)
tau(X) = P(T(1) > horizon | X = x) - P(T(0) > horizon | X = x), for a chosen time point `horizon`.
}
\details{
The causal survival forest paper defines the survival function in the 2nd estimand with weak inequality.
It is defined using strict inequality in the R package (note that P(T >= h) = P(T > h - epsilon)).
}
\examples{
\donttest{
# Train a causal survival forest targeting a Restricted Mean Survival Time (RMST)
# with maxium follow-up time set to `horizon`.
n <- 2000
p <- 5
X <- matrix(runif(n * p), n, p)
W <- rbinom(n, 1, 0.5)
horizon <- 1
failure.time <- pmin(rexp(n) * X[, 1] + W, horizon)
censor.time <- 2 * runif(n)
# Discretizing continuous events decreases runtime.
Y <- round(pmin(failure.time, censor.time), 2)
D <- as.integer(failure.time <= censor.time)
cs.forest <- causal_survival_forest(X, Y, W, D, horizon = horizon)

# Predict using the forest.
X.test <- matrix(0.5, 10, p)
X.test[, 1] <- seq(0, 1, length.out = 10)
cs.pred <- predict(cs.forest, X.test)

# Predict on out-of-bag training samples.
cs.pred <- predict(cs.forest)

# Compute a doubly robust estimate of the average treatment effect.
average_treatment_effect(cs.forest)

# Compute the best linear projection on the first covariate.
best_linear_projection(cs.forest, X[, 1])

# Train a causal survival forest targeting an absolute risk difference
# at the median timepoint `horizon`.
cs.forest.prob <- causal_survival_forest(X, Y, W, D, target = "survival.probability", horizon = 0.5)
}

}
\references{
Cui, Yifan, Michael R. Kosorok, Erik Sverdrup, Stefan Wager, and Ruoqing Zhu.
 "Estimating Heterogeneous Treatment Effects with Right-Censored Data via Causal Survival Forests."
 arXiv preprint arXiv:2001.09887, 2020.
}
