% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/geneticAlgorithm.R
\name{GAfit}
\alias{GAfit}
\title{Genetic algorithm for preliminary estimation of a GMVAR, StMVAR, or G-StMVAR model}
\usage{
GAfit(
  data,
  p,
  M,
  model = c("GMVAR", "StMVAR", "G-StMVAR"),
  conditional = TRUE,
  parametrization = c("intercept", "mean"),
  constraints = NULL,
  same_means = NULL,
  structural_pars = NULL,
  ngen = 200,
  popsize,
  smart_mu = min(100, ceiling(0.5 * ngen)),
  initpop = NULL,
  mu_scale,
  mu_scale2,
  omega_scale,
  W_scale,
  lambda_scale,
  ar_scale = 0.2,
  upper_ar_scale = 1,
  ar_scale2 = 1,
  regime_force_scale = 1,
  red_criteria = c(0.05, 0.01),
  pre_smart_mu_prob = 0,
  to_return = c("alt_ind", "best_ind"),
  minval,
  seed = NULL
)
}
\arguments{
\item{data}{a matrix or class \code{'ts'} object with \code{d>1} columns. Each column is taken to represent
a univariate time series. \code{NA} values are not supported.}

\item{p}{a positive integer specifying the autoregressive order of the model.}

\item{M}{\describe{
  \item{For \strong{GMVAR} and \strong{StMVAR} models:}{a positive integer specifying the number of mixture components.}
  \item{For \strong{G-StMVAR} models:}{a size (2x1) integer vector specifying the number of \emph{GMVAR type} components \code{M1} in the
   first element and \emph{StMVAR type} components \code{M2} in the second element. The total number of mixture components is \code{M=M1+M2}.}
}}

\item{model}{is "GMVAR", "StMVAR", or "G-StMVAR" model considered? In the G-StMVAR model, the first \code{M1} components
are GMVAR type and the rest \code{M2} components are StMVAR type.}

\item{conditional}{a logical argument specifying whether the conditional or exact log-likelihood function}

\item{parametrization}{\code{"intercept"} or \code{"mean"} determining whether the model is parametrized with intercept
parameters \eqn{\phi_{m,0}} or regime means \eqn{\mu_{m}}, m=1,...,M.}

\item{constraints}{a size \eqn{(Mpd^2 x q)} constraint matrix \strong{\eqn{C}} specifying general linear constraints
to the autoregressive parameters. We consider constraints of form
(\strong{\eqn{\phi}}\eqn{_{1}}\eqn{,...,}\strong{\eqn{\phi}}\eqn{_{M}) = }\strong{\eqn{C \psi}},
where \strong{\eqn{\phi}}\eqn{_{m}}\eqn{ = (vec(A_{m,1}),...,vec(A_{m,p}) (pd^2 x 1), m=1,...,M},
contains the coefficient matrices and \strong{\eqn{\psi}} \eqn{(q x 1)} contains the related parameters.
For example, to restrict the AR-parameters to be the same for all regimes, set \strong{\eqn{C}}=
[\code{I:...:I}]\strong{'} \eqn{(Mpd^2 x pd^2)} where \code{I = diag(p*d^2)}.
Ignore (or set to \code{NULL}) if linear constraints should \strong{not} be employed.}

\item{same_means}{Restrict the mean parameters of some regimes to be the same? Provide a list of numeric vectors
such that each numeric vector contains the regimes that should share the common mean parameters. For instance, if
\code{M=3}, the argument \code{list(1, 2:3)} restricts the mean parameters of the second and third regime to be
the same but the first regime has freely estimated (unconditional) mean. Ignore or set to \code{NULL} if mean parameters
should not be restricted to be the same among any regimes. \strong{This constraint is available only for mean parametrized models;
that is, when \code{parametrization="mean"}.}}

\item{structural_pars}{If \code{NULL} a reduced form model is considered. For structural model, should be a list containing
the following elements:
\itemize{
  \item \code{W} - a \eqn{(dxd)} matrix with its entries imposing constraints on \eqn{W}: \code{NA} indicating that the element is
    unconstrained, a positive value indicating strict positive sign constraint, a negative value indicating strict
    negative sign constraint, and zero indicating that the element is constrained to zero.
  \item \code{C_lambda} - a \eqn{(d(M-1) x r)} constraint matrix that satisfies (\strong{\eqn{\lambda}}\eqn{_{2}}\eqn{,...,}
    \strong{\eqn{\lambda}}\eqn{_{M}) =} \strong{\eqn{C_{\lambda} \gamma}} where \strong{\eqn{\gamma}} is the new \eqn{(r x 1)}
    parameter subject to which the model is estimated (similarly to AR parameter constraints). The entries of \code{C_lambda}
    must be either \strong{positive} or \strong{zero}. Ignore (or set to \code{NULL}) if the eigenvalues \eqn{\lambda_{mi}}
    should not be constrained.
}
See Virolainen (2020) for the conditions required to identify the shocks and for the B-matrix as well (it is \eqn{W} times
a time-varying diagonal matrix with positive diagonal entries).}

\item{ngen}{a positive integer specifying the number of generations to be ran through in
the genetic algorithm.}

\item{popsize}{a positive even integer specifying the population size in the genetic algorithm.
Default is \code{10*n_params}.}

\item{smart_mu}{a positive integer specifying the generation after which the random mutations
in the genetic algorithm are "smart". This means that mutating individuals will mostly mutate fairly
close (or partially close) to the best fitting individual (which has the least regimes with time varying
mixing weights practically at zero) so far.}

\item{initpop}{a list of parameter vectors from which the initial population of the genetic algorithm
  will be generated from. The parameter vectors should be...
  \describe{
    \item{\strong{For unconstrained models:}}{
      Should be size \eqn{((M(pd^2+d+d(d+1)/2+2)-M1-1)x1)} and have the form
      \strong{\eqn{\theta}}\eqn{ = }(\strong{\eqn{\upsilon}}\eqn{_{1}},
      ...,\strong{\eqn{\upsilon}}\eqn{_{M}}, \eqn{\alpha_{1},...,\alpha_{M-1},}\strong{\eqn{\nu}}\eqn{)}, where
      \itemize{
        \item \strong{\eqn{\upsilon}}\eqn{_{m}} \eqn{ = (\phi_{m,0},}\strong{\eqn{\phi}}\eqn{_{m}}\eqn{,\sigma_{m})}
        \item \strong{\eqn{\phi}}\eqn{_{m}}\eqn{ = (vec(A_{m,1}),...,vec(A_{m,p})}
        \item and \eqn{\sigma_{m} = vech(\Omega_{m})}, m=1,...,M,
        \item \strong{\eqn{\nu}}\eqn{=(\nu_{M1+1},...,\nu_{M})}
        \item \eqn{M1} is the number of GMVAR type regimes.
      }
    }
    \item{\strong{For constrained models:}}{
      Should be size \eqn{((M(d+d(d+1)/2+2)+q-M1-1)x1)} and have the form
      \strong{\eqn{\theta}}\eqn{ = (\phi_{1,0},...,\phi_{M,0},}\strong{\eqn{\psi},}
      \eqn{\sigma_{1},...,\sigma_{M},\alpha_{1},...,\alpha_{M-1},}\strong{\eqn{\nu}}), where
      \itemize{
        \item \strong{\eqn{\psi}} \eqn{(qx1)} satisfies (\strong{\eqn{\phi}}\eqn{_{1}}\eqn{,...,}
        \strong{\eqn{\phi}}\eqn{_{M}) =} \strong{\eqn{C \psi}} where \strong{\eqn{C}} is a \eqn{(Mpd^2xq)}
        constraint matrix.
      }
    }
    \item{\strong{For same_means models:}}{
      Should have the form
      \strong{\eqn{\theta}}\eqn{ = (}\strong{\eqn{\mu},}\strong{\eqn{\psi},}
      \eqn{\sigma_{1},...,\sigma_{M},\alpha_{1},...,\alpha_{M-1},}\strong{\eqn{\nu}}\eqn{)}, where
      \itemize{
        \item \strong{\eqn{\mu}}\eqn{= (\mu_{1},...,\mu_{g})} where
          \eqn{\mu_{i}} is the mean parameter for group \eqn{i} and
          \eqn{g} is the number of groups.
        \item If AR constraints are employed, \strong{\eqn{\psi}} is as for constrained
          models, and if AR constraints are not employed, \strong{\eqn{\psi}}\eqn{ = }
          (\strong{\eqn{\phi}}\eqn{_{1}}\eqn{,...,}\strong{\eqn{\phi}}\eqn{_{M})}.
      }
    }
    \item{\strong{For structural models:}}{
      Should have the form
      \strong{\eqn{\theta}}\eqn{ = (\phi_{1,0},...,\phi_{M,0},}\strong{\eqn{\phi}}\eqn{_{1},...,}\strong{\eqn{\phi}}\eqn{_{M},
      vec(W),}\strong{\eqn{\lambda}}\eqn{_{2},...,}\strong{\eqn{\lambda}}\eqn{_{M},\alpha_{1},...,\alpha_{M-1},}\strong{\eqn{\nu}}\eqn{)}, where
      \itemize{
        \item\strong{\eqn{\lambda}}\eqn{_{m}=(\lambda_{m1},...,\lambda_{md})} contains the eigenvalues of the \eqn{m}th mixture component.
      }
      \describe{
        \item{\strong{If AR parameters are constrained: }}{Replace \strong{\eqn{\phi}}\eqn{_{1}}\eqn{,...,}
        \strong{\eqn{\phi}}\eqn{_{M}} with \strong{\eqn{\psi}} \eqn{(qx1)} that satisfies (\strong{\eqn{\phi}}\eqn{_{1}}\eqn{,...,}
        \strong{\eqn{\phi}}\eqn{_{M}) =} \strong{\eqn{C \psi}}, as above.}
        \item{\strong{If same_means: }}{Replace \eqn{(\phi_{1,0},...,\phi_{M,0})} with \eqn{(\mu_{1},...,\mu_{g})},
          as above.}
        \item{\strong{If \eqn{W} is constrained:}}{Remove the zeros from \eqn{vec(W)} and make sure the other entries satisfy
         the sign constraints.}
        \item{\strong{If \eqn{\lambda_{mi}} are constrained:}}{Replace \strong{\eqn{\lambda}}\eqn{_{2},...,}\strong{\eqn{\lambda}}\eqn{_{M}}
         with \strong{\eqn{\gamma}} \eqn{(rx1)} that satisfies (\strong{\eqn{\lambda}}\eqn{_{2}}\eqn{,...,}
        \strong{\eqn{\lambda}}\eqn{_{M}) =} \strong{\eqn{C_{\lambda} \gamma}} where \eqn{C_{\lambda}} is a \eqn{(d(M-1) x r)}
         constraint matrix.}
      }
    }
  }
  Above, \eqn{\phi_{m,0}} is the intercept parameter, \eqn{A_{m,i}} denotes the \eqn{i}th coefficient matrix of the \eqn{m}th
  mixture component, \eqn{\Omega_{m}} denotes the error term covariance matrix of the \eqn{m}:th mixture component, and
  \eqn{\alpha_{m}} is the mixing weight parameter. The \eqn{W} and \eqn{\lambda_{mi}} are structural parameters replacing the
  error term covariance matrices (see Virolainen, 2020). If \eqn{M=1}, \eqn{\alpha_{m}} and \eqn{\lambda_{mi}} are dropped.
  If \code{parametrization=="mean"}, just replace each \eqn{\phi_{m,0}} with regimewise mean \eqn{\mu_{m}}.
  \eqn{vec()} is vectorization operator that stacks columns of a given matrix into a vector. \eqn{vech()} stacks columns
  of a given matrix from the principal diagonal downwards (including elements on the diagonal) into a vector.

  In the \strong{GMVAR model}, \eqn{M1=M} and \strong{\eqn{\nu}} is dropped from the parameter vector. In the \strong{StMVAR} model, \eqn{M1=0}.
  In the \strong{G-StMVAR} model, the first \code{M1} regimes are \emph{GMVAR type} and the rest \code{M2} regimes are \emph{StMVAR type}.
  In \strong{StMVAR} and \strong{G-StMVAR} models, the degrees of freedom parameters in \strong{\eqn{\nu}} should be strictly larger than two.

  The notation is similar to the cited literature.}

\item{mu_scale}{a size \eqn{(dx1)} vector defining \strong{means} of the normal distributions from which each
mean parameter \eqn{\mu_{m}} is drawn from in random mutations. Default is \code{colMeans(data)}. Note that
mean-parametrization is always used for optimization in \code{GAfit} - even when \code{parametrization=="intercept"}.
However, input (in \code{initpop}) and output (return value) parameter vectors can be intercept-parametrized.}

\item{mu_scale2}{a size \eqn{(dx1)} strictly positive vector defining \strong{standard deviations} of the normal
distributions from which each mean parameter \eqn{\mu_{m}} is drawn from in random mutations.
Default is \code{2*sd(data[,i]), i=1,..,d}.}

\item{omega_scale}{a size \eqn{(dx1)} strictly positive vector specifying the scale and variability of the
random covariance matrices in random mutations. The covariance matrices are drawn from (scaled) Wishart
distribution. Expected values of the random covariance matrices are \code{diag(omega_scale)}. Standard
deviations of the diagonal elements are \code{sqrt(2/d)*omega_scale[i]}
and for non-diagonal elements they are \code{sqrt(1/d*omega_scale[i]*omega_scale[j])}.
Note that for \code{d>4} this scale may need to be chosen carefully. Default in \code{GAfit} is
\code{var(stats::ar(data[,i], order.max=10)$resid, na.rm=TRUE), i=1,...,d}. This argument is ignored if
structural model is considered.}

\item{W_scale}{a size \eqn{(dx1)} strictly positive vector partly specifying the scale and variability of the
random covariance matrices in random mutations. The elements of the matrix \eqn{W} are drawn independently
from such normal distributions that the expectation of the main \strong{diagonal} elements of the first
regime's error term covariance matrix \eqn{\Omega_1 = WW'} is \code{W_scale}. The distribution of \eqn{\Omega_1}
will be in some sense like a Wishart distribution but with the columns (elements) of \eqn{W} obeying the given
constraints. The constraints are accounted for by setting the element to be always zero if it is subject to a zero
constraint and for sign constraints the absolute value or negative the absolute value are taken, and then the
variances of the elements of \eqn{W} are adjusted accordingly. This argument is ignored if reduced form model
is considered.}

\item{lambda_scale}{a length \eqn{M - 1} vector specifying the \strong{standard deviation} of the mean zero normal
  distribution from which the eigenvalue \eqn{\lambda_{mi}} parameters are drawn from in random mutations.
  As the eigenvalues should always be positive, the absolute value is taken. The elements of \code{lambda_scale}
  should be strictly positive real numbers with the \eqn{m-1}th element giving the degrees of freedom for the \eqn{m}th
  regime. The expected value of the main \strong{diagonal} elements \eqn{ij} of the \eqn{m}th \eqn{(m>1)} error term covariance
  matrix will be \code{W_scale[i]*(d - n_i)^(-1)*sum(lambdas*ind_fun)} where the \eqn{(d x 1)} vector \code{lambdas} is
  drawn from the absolute value of the t-distribution, \code{n_i} is the number of zero constraints in the \eqn{i}th
  row of \eqn{W} and \code{ind_fun} is an indicator function that takes the value one iff the \eqn{ij}th element of
  \eqn{W} is not constrained to zero. Basically, larger lambdas (or smaller degrees of freedom) imply larger variance.

  If the lambda parameters are \strong{constrained} with the \eqn{(d(M - 1) x r)} constraint matrix \eqn{C_lambda},
  then provide a length \eqn{r} vector specifying the standard deviation of the (absolute value of the) mean zero
  normal distribution each of the \eqn{\gamma} parameters are drawn from (the \eqn{\gamma} is a \eqn{(r x 1)} vector).
  The expected value of the main diagonal elements of the covariance matrices then depend on the constraints.

  This argument is ignored if \eqn{M==1} or a reduced form model is considered. Default is \code{rep(3, times=M-1)}
  if lambdas are not constrained and \code{rep(3, times=r)} if lambdas are constrained.

  As with omega_scale and W_scale, this argument should be adjusted carefully if specified by hand. \strong{NOTE}
  that if lambdas are constrained in some other way than restricting some of them to be identical, this parameter
  should be adjusted accordingly in order to the estimation succeed!}

\item{ar_scale}{a positive real number adjusting how large AR parameter values are typically proposed in construction
of the initial population: larger value implies larger coefficients (in absolute value). After construction of the
initial population, a new scale is drawn from \code{(0, 0.)} uniform distribution in each iteration.}

\item{upper_ar_scale}{the upper bound for \code{ar_scale} parameter (see above) in the random mutations. Setting
this too high might lead to failure in proposing new parameters that are well enough inside the parameter space,
and especially with large \code{p} one might want to try smaller upper bound (e.g., 0.5).}

\item{ar_scale2}{a positive real number adjusting how large AR parameter values are typically proposed in some
random mutations (if AR constraints are employed, in all random mutations): larger value implies \strong{smaller} coefficients
(in absolute value). \strong{Values larger than 1 can be used if the AR coefficients are expected to be very small.
If set smaller than 1, be careful as it might lead to failure in the creation of stationary parameter candidates}}

\item{regime_force_scale}{a non-negative real number specifying how much should natural selection favor individuals
with less regimes that have almost all mixing weights (practically) at zero. Set to zero for no favoring or large
number for heavy favoring. Without any favoring the genetic algorithm gets more often stuck in an area of the
parameter space where some regimes are wasted, but with too much favouring the best genes might never mix into
the population and the algorithm might converge poorly. Default is \code{1} and it gives \eqn{2x} larger surviving
probability weights for individuals with no wasted regimes compared to individuals with one wasted regime.
Number \code{2} would give \eqn{3x} larger probability weights etc.}

\item{red_criteria}{a length 2 numeric vector specifying the criteria that is used to determine whether a regime is
redundant (or "wasted") or not.
Any regime \code{m} which satisfies \code{sum(mixingWeights[,m] > red_criteria[1]) < red_criteria[2]*n_obs} will
be considered "redundant". One should be careful when adjusting this argument (set \code{c(0, 0)} to fully disable
the 'redundant regime' features from the algorithm).}

\item{pre_smart_mu_prob}{A number in \eqn{[0,1]} giving a probability of a "smart mutation" occuring randomly in each
iteration before the iteration given by the argument \code{smart_mu}.}

\item{to_return}{should the genetic algorithm return the best fitting individual which has "positive enough" mixing
weights for as many regimes as possible (\code{"alt_ind"}) or the individual which has the highest log-likelihood
in general (\code{"best_ind"}) but might have more wasted regimes?}

\item{minval}{a real number defining the minimum value of the log-likelihood function that will be considered.
Values smaller than this will be treated as they were \code{minval} and the corresponding individuals will
never survive. The default is \code{-(10^(ceiling(log10(n_obs)) + d) - 1)}.}

\item{seed}{a single value, interpreted as an integer, or NULL, that sets seed for the random number generator in the beginning of
the function call. If calling \code{GAfit} from \code{fitGSMVAR}, use the argument \code{seeds} instead of passing the argument \code{seed}.}
}
\value{
Returns the estimated parameter vector which has the form described in \code{initpop}.
}
\description{
\code{GAfit} estimates the specified GMVAR, StMVAR, or G-StMVAR model using a genetic algorithm.
  It's designed to find starting values for gradient based methods.
}
\details{
The core of the genetic algorithm is mostly based on the description by \emph{Dorsey and Mayer (1995)}.
 It utilizes a slightly modified version of the individually adaptive crossover and mutation rates described
 by \emph{Patnaik and Srinivas (1994)} and employs (50\%) fitness inheritance discussed by
 \emph{Smith, Dike and Stegmann (1995)}.

 By "redundant" or "wasted" regimes we mean regimes that have the time varying mixing weights practically at
 zero for almost all t. A model including redundant regimes would have about the same log-likelihood value without
 the redundant regimes and there is no purpose to have redundant regimes in a model.
}
\examples{
\donttest{
 # Preliminary estimation of a G-StMVAR(1, 1, 1) model with 50 generations.
 GA_estimates <- GAfit(gdpdef, p=1, M=c(1, 1), model="G-StMVAR",
                       ngen=50, seed=1)
 GA_estimates
}
}
\references{
\itemize{
   \item Ansley C.F., Kohn R. 1986. A note on reparameterizing a vector autoregressive
         moving average model to enforce stationarity. \emph{Journal of statistical computation
         and simulation}, \strong{24}:2,  99-106.
   \item Dorsey R. E. and Mayer W. J. 1995. Genetic algorithms for estimation problems with multiple optima,
         nondifferentiability, and other irregular features. \emph{Journal of Business & Economic Statistics},
        \strong{13}, 53-66.
   \item Kalliovirta L., Meitz M. and Saikkonen P. 2016. Gaussian mixture vector autoregression.
         \emph{Journal of Econometrics}, \strong{192}, 485-498.
   \item Patnaik L.M. and Srinivas M. 1994. Adaptive Probabilities of Crossover and Mutation in Genetic Algorithms.
         \emph{Transactions on Systems, Man and Cybernetics} \strong{24}, 656-667.
   \item Smith R.E., Dike B.A., Stegmann S.A. 1995. Fitness inheritance in genetic algorithms.
         \emph{Proceedings of the 1995 ACM Symposium on Applied Computing}, 345-350.
   \item Virolainen S. 2020. Structural Gaussian mixture vector autoregressive model. Unpublished working
     paper, available as arXiv:2007.04713.
   \item Virolainen S. 2021. Gaussian and Student's t mixture vector autoregressive model. Unpublished working
     paper, available as arXiv:2109.13648.
 }
}
