% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MAINest.R
\name{fitGMVAR}
\alias{fitGMVAR}
\title{Two-phase maximum likelihood estimation of a GMVAR model}
\usage{
fitGMVAR(data, p, M, conditional = TRUE,
  parametrization = c("intercept", "mean"), constraints = NULL,
  ncalls = round(10 + 9 * log(M)), ncores = min(2, ncalls,
  parallel::detectCores()), maxit = 300, seeds = NULL,
  print_res = TRUE, ...)
}
\arguments{
\item{data}{a matrix or class \code{'ts'} object with \code{d>1} columns. Each column is taken to represent
a single time series. \code{NA} values are not supported.}

\item{p}{a positive integer specifying the autoregressive order of the model.}

\item{M}{a positive integer specifying the number of mixture components.}

\item{conditional}{a logical argument specifying whether the conditional or exact log-likelihood function
should be used.}

\item{parametrization}{\code{"mean"} or \code{"intercept"} determining whether the model is parametrized with regime means \eqn{\mu_{m}} or
intercept parameters \eqn{\phi_{m,0}}, m=1,...,M.}

\item{constraints}{a size \eqn{(Mpd^2 x q)} constraint matrix \strong{\eqn{C}} specifying general linear constraints
to the autoregressive parameters. We consider constraints of form
(\strong{\eqn{\phi}}\eqn{_{1}}\eqn{,...,}\strong{\eqn{\phi}}\eqn{_{M}) = }\strong{\eqn{C \psi}},
where \strong{\eqn{\phi}}\eqn{_{m}}\eqn{ = (vec(A_{m,1}),...,vec(A_{m,p}) (pd^2 x 1), m=1,...,M}
contains the coefficient matrices and \strong{\eqn{\psi}} \eqn{(q x 1)} contains the constrained parameters.
For example, to restrict the AR-parameters to be the same for all regimes, set \strong{\eqn{C}}=
[\code{I:...:I}]\strong{'} \eqn{(Mpd^2 x pd^2)} where \code{I = diag(p*d^2)}.
Ignore (or set to \code{NULL}) if linear constraints should \strong{not} be employed.}

\item{ncalls}{the number of estimation rounds that should be performed.}

\item{ncores}{the number CPU cores to be used in parallel computing.}

\item{maxit}{the maximum number of iterations in the variable metric algorithm.}

\item{seeds}{a length \code{ncalls} vector containing the random number generator seed for each call to the genetic algorithm,
or \code{NULL} for not initializing the seed. Exists for creating reproducible results.}

\item{print_res}{should summaries of estimation results be printed?}

\item{...}{additional settings passed to the function \code{GAfit} employing the genetic algorithm.}
}
\value{
Returns an object of class \code{'gmvar'} defining the estimated GMVAR model. Multivariate quantile residuals
  (Kalliovirta and Saikkonen 2010) are also computed and included in the returned object. In addition, the returned
  object contains the estimates and log-likelihood values from all the estimation rounds performed.
  The estimated parameter vector can be obtained at \code{gmvar$params} (and corresponding approximate standard errors
  at \code{gmvar$std_errors}) and it is...
  \describe{
    \item{\strong{Unconstrained models:}}{
      a size \eqn{((M(pd^2+d+d(d+1)/2+1)-1)x1)} vector that has form
      \strong{\eqn{\theta}}\eqn{ = }(\strong{\eqn{\upsilon}}\eqn{_{1}},
      ...,\strong{\eqn{\upsilon}}\eqn{_{M}}, \eqn{\alpha_{1},...,\alpha_{M-1}}), where:
      \itemize{
        \item \strong{\eqn{\upsilon}}\eqn{_{m}} \eqn{ = (\phi_{m,0},}\strong{\eqn{\phi}}\eqn{_{m}}\eqn{,\sigma_{m})}
        \item \strong{\eqn{\phi}}\eqn{_{m}}\eqn{ = (vec(A_{m,1}),...,vec(A_{m,p})}
        \item and \eqn{\sigma_{m} = vech(\Omega_{m})}, m=1,...,M.
      }
    }
    \item{\strong{Constrained models:}}{
      a size \eqn{((M(d+d(d+1)/2+1)+q-1)x1)} vector that has form
      \strong{\eqn{\theta}}\eqn{ = (\phi_{1,0},...,\phi_{M,0},}\strong{\eqn{\psi}}
      \eqn{,\sigma_{1},...,\sigma_{M},\alpha_{1},...,\alpha_{M-1})}, where:
      \itemize{
        \item \strong{\eqn{\psi}} \eqn{(qx1)} satisfies (\strong{\eqn{\phi}}\eqn{_{1}}\eqn{,...,}
        \strong{\eqn{\phi}}\eqn{_{M}) =} \strong{\eqn{C \psi}}. Here \strong{\eqn{C}} is \eqn{(Mpd^2xq)}
        constraint matrix.
      }
    }
  }
  Above, \eqn{\phi_{m,0}} is the intercept parameter, \eqn{A_{m,i}} denotes the \eqn{i}:th coefficient matrix of
  the \eqn{m}:th mixture component, \eqn{\Omega_{m}} denotes the error term covariance matrix of the \eqn{m}:th
  mixture component, and \eqn{\alpha_{m}} is the mixing weight parameter.
  If \code{parametrization=="mean"}, just replace each \eqn{\phi_{m,0}} with regimewise mean \eqn{\mu_{m}}.
  \eqn{vec()} is vectorization operator that stacks columns of a given matrix into a vector. \eqn{vech()} stacks
  columns of a given matrix from the principal diagonal downwards (including elements on the diagonal) into a vector.
  The notation is in line with the cited article by \emph{Kalliovirta, Meitz and Saikkonen (2016)} which introduces
  the GMVAR model.

  Remark that the first autocovariance/correlation matrix in \code{$uncond_moments} is for the lag zero,
  the second one for the lag one, etc.
}
\description{
\code{fitGMVAR} estimates a GMVAR model in two phases: in the first phase it uses a genetic algorithm
  to find starting values for a gradient based variable metric algorithm, which it then uses to finalize the
  estimation in the second phase. Parallel computing is utilized to perform multiple rounds of estimations in parallel.
}
\details{
Because of complexity and multimodality of the log-likelihood function, it's \strong{not certain} that the estimation
 algorithms will end up in the global maximum point. It's expected that most of the estimation rounds will end up in
 some local maximum point instead. Therefore a number of estimation rounds is required for reliable results. Because
 of the nature of the model, the estimation may fail especially in the cases where the number of mixture components
 is chosen too large.

 The estimation process is computationally heavy and it might take considerably long time for large models with
 large number of observations. If the iteration limit \code{maxit} in the variable metric algorithm is reached,
 one can continue the estimation by iterating more with the function \code{iterate_more}. Alternatively, one may
 use the found estimates as starting values for the genetic algorithm and and employ another round of estimation.

 The code of the genetic algorithm is mostly based on the description by \emph{Dorsey and Mayer (1995)} but it
 includes some extra features that were found useful for this particular estimation problem. For instance,
 the genetic algorithm uses a slightly modified version of the individually adaptive crossover and mutation
 rates descriped by \emph{Patnaik and Srinivas (1994)} and employs (50\%) fitness inheritance discussed
 by \emph{Smith, Dike and Stegmann (1995)}.

 The gradient based variable metric algorithm used in the second phase is implemented with function \code{optim}
 from the package \code{stats}.
}
\section{S3 methods}{

  The following S3 methods are supported for class \code{'gmvar'}: \code{logLik}, \code{residuals}, \code{print}, \code{summary},
   \code{predict} and \code{plot}.
}

\examples{
\donttest{
## These are long running examples that use parallel computing!

# These examples use the data 'eurusd' which comes with the
# package, but in a scaled form (similar to Kalliovirta et al. 2016).
data(eurusd, package="gmvarkit")
data <- cbind(10*eurusd[,1], 100*eurusd[,2])
colnames(data) <- colnames(eurusd)

# GMVAR(1,2) model: 10 estimation rounds with seeds set
# for reproducibility
fit12 <- fitGMVAR(data, p=1, M=2, ncalls=10, seeds=1:10)
fit12
plot(fit12)
summary(fit12)

# GMVAR(2,2) model with mean parametrization
fit22 <- fitGMVAR(data, p=2, M=2, parametrization="mean",
                  ncalls=16, seeds=11:26)
fit22

# GMVAR(2,2) model with autoregressive parameters restricted
# to be the same for both regimes
C_mat <- rbind(diag(2*2^2), diag(2*2^2))
fit22c <- fitGMVAR(data, p=2, M=2, constraints=C_mat)
fit22c

# GMVAR(2,2) model with autoregressive parameters restricted
# to be the same for both regimes and non-diagonl elements
# the coefficient matrices constrained to zero. Estimation
# with only 10 estimation rounds.
tmp <- matrix(c(1, rep(0, 10), 1, rep(0, 8), 1, rep(0, 10), 1),
 nrow=2*2^2, byrow=FALSE)
C_mat2 <- rbind(tmp, tmp)
fit22c2 <- fitGMVAR(data, p=2, M=2, constraints=C_mat2)
fit22c2
}
}
\references{
\itemize{
   \item Dorsey R. E. and Mayer W. J. 1995. Genetic algorithms for estimation problems with multiple optima,
         nondifferentiability, and other irregular features. \emph{Journal of Business & Economic Statistics},
        \strong{13}, 53-66.
   \item Kalliovirta L., Meitz M. and Saikkonen P. 2016. Gaussian mixture vector autoregression.
         \emph{Journal of Econometrics}, \strong{192}, 485-498.
   \item Kalliovirta L. and Saikkonen P. 2010. Reliable Residuals for Multivariate Nonlinear
         Time Series Models. \emph{Unpublished Revision of HECER Discussion Paper No. 247}.
   \item Patnaik L.M. and Srinivas M. 1994. Adaptive Probabilities of Crossover and Mutation in Genetic Algorithms.
         \emph{Transactions on Systems, Man and Cybernetics} \strong{24}, 656-667.
   \item Smith R.E., Dike B.A., Stegmann S.A. 1995. Fitness inheritance in genetic algorithms.
         \emph{Proceedings of the 1995 ACM Symposium on Applied Computing}, 345-350.
 }
}
\seealso{
\code{\link{GMVAR}}, \code{\link{iterate_more}}, \code{\link{predict.gmvar}}, \code{\link{profile_logliks}},
  \code{\link{simulateGMVAR}}, \code{\link{quantile_residual_tests}}, \code{\link{print_std_errors}},
  \code{\link{swap_parametrization}}, \code{\link{get_gradient}}
}
