% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/stat-fit-residuals.R
\name{stat_fit_residuals}
\alias{stat_fit_residuals}
\title{Residuals from a model fit}
\usage{
stat_fit_residuals(
  mapping = NULL,
  data = NULL,
  geom = "point",
  method = "lm",
  method.args = list(),
  formula = NULL,
  resid.type = NULL,
  position = "identity",
  na.rm = FALSE,
  orientation = NA,
  show.legend = FALSE,
  inherit.aes = TRUE,
  ...
)
}
\arguments{
\item{mapping}{The aesthetic mapping, usually constructed with
\code{\link[ggplot2]{aes}} or \code{\link[ggplot2]{aes_}}. Only needs
to be set at the layer level if you are overriding the plot defaults.}

\item{data}{A layer specific dataset - only needed if you want to override
the plot defaults.}

\item{geom}{The geometric object to use display the data}

\item{method}{function or character If character, "lm", "rlm", and "rq"
are implemented. If a function, it must support parameters \code{formula}
and \code{data}.}

\item{method.args}{named list with additional arguments.}

\item{formula}{a "formula" object. Using aesthetic names instead of
original variable names.}

\item{resid.type}{character passed to \code{residuals()} as argument for
\code{type}.}

\item{position}{The position adjustment to use for overlapping points on this
layer}

\item{na.rm}{a logical indicating whether NA values should be stripped
before the computation proceeds.}

\item{orientation}{character Either "x" or "y" controlling the default for
\code{formula}.}

\item{show.legend}{logical. Should this layer be included in the legends?
\code{NA}, the default, includes if any aesthetics are mapped. \code{FALSE}
never includes, and \code{TRUE} always includes.}

\item{inherit.aes}{If \code{FALSE}, overrides the default aesthetics, rather
than combining with them. This is most useful for helper functions that
define both data and aesthetics and should not inherit behaviour from the
default plot specification, e.g. \code{\link[ggplot2]{borders}}.}

\item{...}{other arguments passed on to \code{\link[ggplot2]{layer}}. This
can include aesthetics whose values you want to set, not map. See
\code{\link[ggplot2]{layer}} for more details.}
}
\description{
\code{stat_fit_residuals} fits a linear model and returns
   residuals ready to be plotted as points.
}
\details{
This stat can be used to automatically plot residuals as points in a
  plot. At the moment it supports only linear models fitted with function
  \code{lm()}. This stat only generates the residuals.

  A ggplot statistic receives as data a data frame that is not the one passed
  as argument by the user, but instead a data frame with the variables mapped
  to aesthetics. In other words, it respects the grammar of graphics and
  consequently within the model \code{formula} names of
  aesthetics like $x$ and $y$ should be used intead of the original variable
  names, while data is automatically passed the data frame. This helps ensure
  that the model is fitted to the same data as plotted in other layers.
}
\note{
Parameter \code{orientation} is redundant as it only affects the default
  for \code{formula} but is included for consistency with
  \code{ggplot2}.
}
\section{Computed variables}{
 Data frame with same \code{nrow} as \code{data}
  as subset for each group containing five numeric variables. \describe{
  \item{x}{x coordinates of observations} \item{y.resid}{residuals from
  fitted values} \item{y.resid.abs}{absolute residuals from the fit}}.

  By default \code{stat(y.resid)} is mapped to the \code{y} aesthetic.
}

\examples{
# generate artificial data
set.seed(4321)
x <- 1:100
y <- (x + x^2 + x^3) + rnorm(length(x), mean = 0, sd = mean(x^3) / 4)
my.data <- data.frame(x, y)

# plot residuals from linear model
ggplot(my.data, aes(x, y)) +
  geom_hline(yintercept = 0, linetype = "dashed") +
  stat_fit_residuals(formula = y ~ x)

# plot residuals from linear model with y as explanatory variable
ggplot(my.data, aes(x, y)) +
  geom_vline(xintercept = 0, linetype = "dashed") +
  stat_fit_residuals(formula = x ~ y) +
  coord_flip()

# give a name to a formula
my.formula <- y ~ poly(x, 3, raw = TRUE)

# plot residuals from linear model
ggplot(my.data, aes(x, y)) +
  geom_hline(yintercept = 0, linetype = "dashed") +
  stat_fit_residuals(formula = my.formula) +
  coord_flip()

ggplot(my.data, aes(x, y)) +
  geom_hline(yintercept = 0, linetype = "dashed") +
  stat_fit_residuals(formula = my.formula, resid.type = "response")

# plot residuals from robust regression
ggplot(my.data, aes(x, y)) +
  geom_hline(yintercept = 0, linetype = "dashed") +
  stat_fit_residuals(formula = my.formula, method = "rlm")

# plot residuals with weights indicated by colour
my.data.outlier <- my.data
my.data.outlier[6, "y"] <- my.data.outlier[6, "y"] * 10
ggplot(my.data.outlier, aes(x, y)) +
  stat_fit_residuals(formula = my.formula, method = "rlm",
                      mapping = aes(colour = after_stat(weights)),
                      show.legend = TRUE) +
  scale_color_gradient(low = "red", high = "blue", limits = c(0, 1),
                       guide = "colourbar")

# plot weighted residuals with weights indicated by colour
ggplot(my.data.outlier) +
  stat_fit_residuals(formula = my.formula, method = "rlm",
                     mapping = aes(x = x,
                                   y = stage(start = y, after_stat = y * weights),
                                   colour = after_stat(weights)),
                     show.legend = TRUE) +
  scale_color_gradient(low = "red", high = "blue", limits = c(0, 1),
                       guide = "colourbar")

# plot residuals from quantile regression (median)
ggplot(my.data, aes(x, y)) +
  geom_hline(yintercept = 0, linetype = "dashed") +
  stat_fit_residuals(formula = my.formula, method = "rq")

# plot residuals from quantile regression (upper quartile)
ggplot(my.data, aes(x, y)) +
  geom_hline(yintercept = 0, linetype = "dashed") +
  stat_fit_residuals(formula = my.formula, method = "rq",
  method.args = list(tau = 0.75))

# inspecting the returned data
if (requireNamespace("gginnards", quietly = TRUE)) {
  library(gginnards)

  ggplot(my.data, aes(x, y)) +
   stat_fit_residuals(formula = my.formula, resid.type = "working",
                      geom = "debug")

  ggplot(my.data, aes(x, y)) +
    stat_fit_residuals(formula = my.formula, method = "rlm",
                       geom = "debug")
}

}
\seealso{
Other ggplot statistics for model fits: 
\code{\link{stat_fit_augment}()},
\code{\link{stat_fit_deviations}()},
\code{\link{stat_fit_glance}()},
\code{\link{stat_fit_tb}()},
\code{\link{stat_fit_tidy}()}
}
\concept{ggplot statistics for model fits}
