% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/stan_sar.R
\name{stan_sar}
\alias{stan_sar}
\title{Simultaneous autoregressive (SAR) models}
\source{
Cliff, A D and Ord, J K (1981). \emph{Spatial Processes: Models and Applications}. Pion.

Cressie, Noel (2015 (1993)). \emph{Statistics for Spatial Data}. Wiley Classics, Revised Edition.

Cressie, Noel and Wikle, Christopher (2011). \emph{Statistics for Spatio-Temporal Data}. Wiley.

Donegan, Connor (2021). Building spatial conditional autoregressive (CAR) models in the Stan programming language. \emph{OSF Preprints}. \doi{10.31219/osf.io/3ey65}.

LeSage, James (2014). What Regional Scientists Need to Know about Spatial Econometrics. \emph{The Review of Regional Science} 44: 13-32 (2014 Southern Regional Science Association Fellows Address).

LeSage, James, & Pace, Robert Kelley (2009). \emph{Introduction to Spatial Econometrics}. Chapman and Hall/CRC.
}
\usage{
stan_sar(
  formula,
  slx,
  re,
  data,
  C,
  sar_parts = prep_sar_data(C),
  family = auto_gaussian(),
  prior = NULL,
  ME = NULL,
  centerx = FALSE,
  prior_only = FALSE,
  censor_point,
  chains = 4,
  iter = 2000,
  refresh = 500,
  keep_all = FALSE,
  pars = NULL,
  slim = FALSE,
  drop = NULL,
  control = NULL,
  ...
)
}
\arguments{
\item{formula}{A model formula, following the R \code{\link[stats]{formula}} syntax. Binomial models can be specified by setting the left hand side of the equation to a data frame of successes and failures, as in \code{cbind(successes, failures) ~ x}.}

\item{slx}{Formula to specify any spatially-lagged covariates. As in, \code{~ x1 + x2} (the intercept term will be removed internally). When setting priors for \code{beta}, remember to include priors for any SLX terms.}

\item{re}{To include a varying intercept (or "random effects") term, \code{alpha_re}, specify the grouping variable here using formula syntax, as in \code{~ ID}. Then, \code{alpha_re} is a vector of parameters added to the linear predictor of the model, and:

\if{html}{\out{<div class="sourceCode">}}\preformatted{alpha_re ~ N(0, alpha_tau)
alpha_tau ~ Student_t(d.f., location, scale).
}\if{html}{\out{</div>}}

With the SAR model, any \code{alpha_re} term should be at a \emph{different} level or scale than the observations; that is, at a different scale than the autocorrelation structure of the SAR model itself.}

\item{data}{A \code{data.frame} or an object coercible to a data frame by \code{as.data.frame} containing the model data.}

\item{C}{Spatial weights matrix (conventionally referred to as \eqn{W} in the SAR model). Typically, this will be created using \code{geostan::shape2mat(shape, style = "W")}. This will be passed internally to \code{\link[geostan]{prep_sar_data}}, and will also be used to calculate residual spatial autocorrelation as well as any user specified \code{slx} terms; it will automatically be row-standardized before calculating \code{slx} terms. See \code{\link[geostan]{shape2mat}}.}

\item{sar_parts}{Optional. If not provided, then \code{\link[geostan]{prep_sar_data}} will be used automatically to create \code{sar_parts} using the user-provided spatial weights matrix.}

\item{family}{The likelihood function for the outcome variable. Current options are \code{auto_gaussian()}, \code{binomial(link = "logit")}, and \code{poisson(link = "log")}; if \code{family = gaussian()} is provided, it will automatically be converted to \code{auto_gaussian()}.}

\item{prior}{A named list of parameters for prior distributions (see \code{\link[geostan]{priors}}):
\describe{

\item{intercept}{The intercept is assigned a Gaussian prior distribution (see \code{\link[geostan]{normal}}}.

\item{beta}{Regression coefficients are assigned Gaussian prior distributions. Variables must follow their order of appearance in the model \code{formula}. Note that if you also use \code{slx} terms (spatially lagged covariates), and you use custom priors for \code{beta}, then you have to provide priors for the slx terms. Since slx terms are \emph{prepended} to the design matrix, the prior for the slx term will be listed first.
}

\item{sar_scale}{Scale parameter for the SAR model, \code{sar_scale}. The scale is assigned a Student's t prior model (constrained to be positive).}

\item{sar_rho}{The spatial autocorrelation parameter in the SAR model, \code{rho}, is assigned a uniform prior distribution. By default, the prior will be uniform over all permissible values as determined by the eigenvalues of the spatial weights matrix. The range of permissible values for \code{rho} is printed to the console by \code{\link[geostan]{prep_sar_data}}.}

\item{tau}{The scale parameter for any varying intercepts (a.k.a exchangeable random effects, or partial pooling) terms. This scale parameter, \code{tau}, is assigned a Student's t prior (constrained to be positive).}

}}

\item{ME}{To model observational uncertainty (i.e. measurement or sampling error) in any or all of the covariates, provide a list of data as constructed by the \code{\link[geostan]{prep_me_data}} function.}

\item{centerx}{To center predictors on their mean values, use \code{centerx = TRUE}. If the ME argument is used, the modeled covariate (i.e., latent variable), rather than the raw observations, will be centered. When using the ME argument, this is the recommended method for centering the covariates.}

\item{prior_only}{Logical value; if \code{TRUE}, draw samples only from the prior distributions of parameters.}

\item{censor_point}{Integer value indicating the maximum censored value; this argument is for modeling censored (suppressed) outcome data, typically disease case counts or deaths.}

\item{chains}{Number of MCMC chains to use.}

\item{iter}{Number of samples per chain.}

\item{refresh}{Stan will print the progress of the sampler every \code{refresh} number of samples. Set \code{refresh=0} to silence this.}

\item{keep_all}{If \code{keep_all = TRUE} then samples for all parameters in the Stan model will be kept; this is necessary if you want to do model comparison with Bayes factors and the \code{bridgesampling} package.}

\item{pars}{Optional; specify any additional parameters you'd like stored from the Stan model.}

\item{slim}{If \code{slim = TRUE}, then the Stan model will not collect the most memory-intensive parameters (including n-length vectors of fitted values, log-likelihoods, and ME-modeled covariate values). This will disable many convenience functions that are otherwise available for fitted \code{geostan} models, such as the extraction of residuals, fitted values, and spatial trends, WAIC, and spatial diagnostics, and ME diagnostics; many quantities of interest, such as fitted values and spatial trends, can still be calculated manually using given parameter estimates. The "slim" option is designed for data-intensive routines, such as regression with raster data, Monte Carlo studies, and measurement error models. For more control over which parameters are kept or dropped, use the \code{drop} argument instead of \code{slim}.}

\item{drop}{Provide a vector of character strings to specify the names of any parameters that you do not want MCMC samples for. Dropping parameters in this way can improve sampling speed and reduce memory usage. The following parameter vectors can potentially be dropped from SAR models:
\describe{
\item{fitted}{The N-length vector of fitted values}
\item{log_lik}{The N-length vector of pointwise log-likelihoods, which is used to calculate WAIC.}
\item{alpha_re}{Vector of 'random effects'/varying intercepts.}
\item{log_lambda_mu}{Linear predictor inside the SAR model (for Poisson and binomial models)}
\item{x_true}{N-length vector of 'latent'/modeled covariate values created for measurement error (ME) models.}
}
Using \code{drop = c('fitted', 'log_lik', 'alpha_re', 'x_true')} is equivalent to \code{slim = TRUE}. Note that if \code{slim = TRUE}, then \code{drop} will be ignored---so only use one or the other.}

\item{control}{A named list of parameters to control the sampler's behavior. See \code{\link[rstan]{stan}} for details.}

\item{...}{Other arguments passed to \code{\link[rstan]{sampling}}.}
}
\value{
An object of class class \code{geostan_fit} (a list) containing:
\describe{
\item{summary}{Summaries of the main parameters of interest; a data frame.}
\item{diagnostic}{Widely Applicable Information Criteria (WAIC) with a measure of effective number of parameters (\code{eff_pars}) and mean log pointwise predictive density (\code{lpd}), and mean residual spatial autocorrelation as measured by the Moran coefficient.}
\item{stanfit}{an object of class \code{stanfit} returned by \code{rstan::stan}}
\item{data}{a data frame containing the model data}
\item{family}{the user-provided or default \code{family} argument used to fit the model}
\item{formula}{The model formula provided by the user (not including CAR component)}
\item{slx}{The \code{slx} formula}
\item{re}{A list containing \code{re}, the varying intercepts (\code{re}) formula if provided, and
\code{Data} a data frame with columns \code{id}, the grouping variable, and \code{idx}, the index values assigned to each group.}
\item{priors}{Prior specifications.}

\item{x_center}{If covariates are centered internally (\code{centerx = TRUE}), then \code{x_center} is a numeric vector of the values on which covariates were centered.}

\item{spatial}{A data frame with the name of the spatial component parameter (either "phi" or, for auto Gaussian models, "trend") and method ("SAR")}
\item{ME}{A list indicating if the object contains an ME model; if so, the user-provided ME list is also stored here.}
\item{C}{Spatial weights matrix (in sparse matrix format).}
}
}
\description{
Fit data to an spatial Gaussian SAR (spatial error) model, or model a vector of spatially-autocorrelated parameters using a SAR prior model.
}
\details{
Discussions of SAR models may be found in Cliff and Ord (1981), Cressie (2015, Ch. 6), LeSage and Pace (2009), and LeSage (2014). The Stan implementation draws from Donegan (2021).

The general scheme of the SAR model for numeric vector \eqn{y} is
\deqn{y = \mu + ( I - \rho W)^{-1} \epsilon}
\deqn{\epsilon \sim Gauss(0, \sigma^2 I)}
where \eqn{W} is the spatial weights matrix, \eqn{I} is the n-by-n identity matrix, and \eqn{\rho} is a spatial autocorrelation parameter. In words, the errors of the regression equation are spatially autocorrelated.

Re-arranging terms, the model can also be written as follows:
\deqn{y = \mu + \rho W (y - \mu)  + \epsilon}
which perhaps shows more intuitively the implicit spatial trend component, \eqn{\rho W (y - \mu)}.

Most often, this model is applied directly to observations (referred to below as the auto-Gaussian model). The SAR model can also be applied to a vector of parameters inside a hierarchical model. The latter enables spatial autocorrelation to be modeled when the observations are discrete counts (e.g., disease incidence data).

A note on terminology: the spatial statistics literature conceptualizes the simultaneously-specified spatial autoregressive model (SAR) in relation to the conditionally-specified spatial autoregressive model (CAR) (see \link[geostan]{stan_car}) (see Cliff and Ord 1981). The spatial econometrics literature, by contrast, refers to the simultaneously-specified spatial autoregressive (SAR) model as the spatial error model (SEM), and they contrast the SEM with the spatial lag model (which contains a spatially-lagged dependent variable on the right-hand-side of the regression equation) (see LeSage 2014).
\subsection{Auto-Gaussian}{

When \code{family = auto_gaussian()}, the SAR model is specified as follows:

\deqn{y \sim Gauss(\mu, \Sigma)}
\deqn{\Sigma = \sigma^2 (I - \rho W)^{-1}(I - \rho W')^{-1}}
where \eqn{\mu} is the mean vector (with intercept, covariates, etc.), \eqn{W} is a spatial weights matrix (usually row-standardized), and \eqn{\sigma} is a scale parameter.

The SAR model contains an implicit spatial trend (i.e., spatial autocorrelation) component \eqn{\phi} which is calculated as follows:
\deqn{
\phi = \rho W (y - \mu)
}

This term can be extracted from a fitted auto-Gaussian model using the \link[geostan]{spatial} method.

When applied to a fitted auto-Gaussian model, the \link[geostan]{residuals.geostan_fit} method returns 'de-trended' residuals \eqn{R} by default. That is,
\deqn{
R = y - \mu - \rho W (y - \mu).
}
To obtain "raw" residuals (\eqn{y - \mu}), use \code{residuals(fit, detrend = FALSE)}. Similarly, the fitted values obtained from the \link[geostan]{fitted.geostan_fit} will include the spatial trend term by default.
}

\subsection{Poisson}{

For \code{family = poisson()}, the model is specified as:

\deqn{y \sim Poisson(e^{O + \lambda})}
\deqn{\lambda \sim Gauss(\mu, \Sigma)}
\deqn{\Sigma = \sigma^2 (I - \rho W)^{-1}(I - \rho W')^{-1}.}

If the raw outcome consists of a rate \eqn{\frac{y}{p}} with observed counts \eqn{y} and denominator \eqn{p} (often this will be the size of the population at risk), then the offset term \eqn{O=log(p)} is the log of the denominator.

This is often written (equivalently) as:

\deqn{y \sim Poisson(e^{O + \mu + \phi})}
\deqn{ \phi \sim Gauss(0, \Sigma) }
\deqn{ \Sigma = \sigma^2 (I - \rho W)^{-1}(I - \rho W')^{-1}.}

For Poisson models, the \link[geostan]{spatial} method returns the parameter vector \eqn{\phi}.

In the Poisson SAR model, \eqn{\phi} contains a latent spatial trend as well as additional variation around it. If you would like to extract the latent/implicit spatial trend from \eqn{\phi}, you can do so by calculating:
\deqn{
 \rho W \phi.
}
}

\subsection{Binomial}{

For \code{family = binomial()}, the model is specified as:

\deqn{y \sim Binomial(N, \lambda) }
\deqn{logit(\lambda) \sim Gauss(\mu, \Sigma) }
\deqn{\Sigma = \sigma^2 (I - \rho W)^{-1}(I - \rho W')^{-1}.}

where outcome data \eqn{y} are counts, \eqn{N} is the number of trials, and \eqn{\lambda} is the rate of 'success'. Note that the model formula should be structured as: \code{cbind(sucesses, failures) ~ 1} (for an intercept-only model), such that \code{trials = successes + failures}.

For fitted Binomial models, the \code{\link[geostan]{spatial}} method will return the parameter vector \code{phi}, equivalent to:

\deqn{\phi = logit(\lambda) - \mu.}

As is also the case for the Poisson model, \eqn{\phi} contains a latent spatial trend as well as additional variation around it. If you would like to extract the latent/implicit spatial trend from \eqn{\phi}, you can do so by calculating:
\deqn{
\rho W \phi.
}
}

\subsection{Additional functionality}{

The SAR models can also incorporate spatially-lagged covariates, measurement/sampling error in covariates (particularly when using small area survey estimates as covariates), missing outcome data, and censored outcomes (such as arise when a disease surveillance system suppresses data for privacy reasons). For details on these options, please see the Details section in the documentation for \link[geostan]{stan_glm}.
}
}
\examples{
# model mortality risk
data(georgia)
W <- shape2mat(georgia, style = "W")

fit <- stan_sar(log(rate.male) ~ 1,
                C = W,
                data = georgia,
                chains = 1, # for ex. speed only
                iter = 700 
                )

rstan::stan_rhat(fit$stanfit)
rstan::stan_mcse(fit$stanfit)
print(fit)
plot(fit)
sp_diag(fit, georgia)

\donttest{
 # a more appropriate model for count data:
fit2 <- stan_sar(deaths.male ~ offset(log(pop.at.risk.male)),
                C = W,
                data = georgia,
                family = poisson(),
                chains = 1, # for ex. speed only
                iter = 700 
                 )
sp_diag(fit2, georgia)
}
}
\author{
Connor Donegan, \email{connor.donegan@gmail.com}
}
