% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/galah_bbox.R, R/galah_geolocate.R,
%   R/galah_polygon.R
\name{galah_bbox}
\alias{galah_bbox}
\alias{galah_geolocate}
\alias{galah_polygon}
\alias{st_crop.data_request}
\title{Narrow a query to within a specified area}
\usage{
galah_bbox(...)

galah_geolocate(..., type = c("polygon", "bbox"))

galah_polygon(...)

\method{st_crop}{data_request}(x, y, ...)
}
\arguments{
\item{...}{a single \code{sf} object, WKT string or shapefile. Bounding boxes can
be supplied as a \code{tibble}/\code{data.frame} or a \code{bbox}}

\item{type}{\code{string}: one of \code{c("polygon", "bbox")}. Defaults to
\code{"polygon"}. If \code{type = "polygon"}, a multipolygon will be built via
\code{\link[=galah_polygon]{galah_polygon()}}. If \code{type = "bbox"}, a multipolygon will be built via
\code{\link[=galah_bbox]{galah_bbox()}}. The multipolygon is used to narrow a query to the ALA.}

\item{x}{An object of class \code{data_request}, created using \code{\link[=galah_call]{galah_call()}}}

\item{y}{A valid Well-Known Text string (wkt), a \code{POLYGON} or a \code{MULTIPOLYGON}}
}
\value{
length-1 string (class \code{character}) containing a multipolygon WKT
string representing the area provided.
}
\description{
Restrict results to those from a specified area using \code{galah_geolocate()}.
Areas can be specified as either polygons or bounding boxes, depending on
\code{type}. Alternatively, users can call the underlying functions directly via
\code{galah_bbox()} or \code{galah_polygon()}. Finally, it is possible to use \code{sf}
syntax by calling \code{st_crop()}, which is synonymous with \code{galah_polygon()}.

If calling \code{galah_geolocate()}, the default \code{type} is \code{"polygon"}, which
narrows queries to within an area supplied as a \code{POLYGON}. Polygons must be
specified as either an \code{sf} object, a 'well-known text' (WKT) string, or a
shapefile. Shapefiles must be simple to be accepted by the ALA.
Alternatively, set \code{type = "bbox"} to narrow queries to within a bounding
box. Bounding boxes can be extracted from a supplied \code{sf} object or
a shapefile. A bounding box can also be supplied as a \code{bbox} object
(via \code{sf::st_bbox()}) or a \code{tibble}/\code{data.frame}.
}
\details{
If \code{type = "polygon"}, WKT strings longer than 10000 characters and
\code{sf} objects with more than 500 vertices will not be
accepted by the ALA. Some polygons  may need to be simplified. If
\code{type = "bbox"}, sf objects and shapefiles will be converted to a bounding
box to query the ALA.
}
\examples{
\dontrun{
# Search for records within a polygon using a shapefile
location <- sf::st_read("path/to/shapefile.shp")
galah_call() |>
  galah_identify("vulpes") |>
  galah_geolocate(location) |>
  atlas_counts()
  
# Search for records within the bounding box of a shapefile
location <- sf::st_read("path/to/shapefile.shp")
galah_call() |>
  galah_identify("vulpes") |>
  galah_geolocate(location, type = "bbox") |>
  atlas_counts()

# Search for records within a polygon using an `sf` object
location <- "POLYGON((142.3 -29.0,142.7 -29.1,142.7 -29.4,142.3 -29.0))" |>
 sf::st_as_sfc()
galah_call() |>
  galah_identify("reptilia") |>
  galah_polygon(location) |>
  atlas_counts()
  
# Alternatively, we can use `st_crop()` as a synonym for `galah_polygon()`. 
# Hence the above example can be rewritten as:
request_data() |>
  identify("reptilia") |>
  st_crop(location) |>
  count() |>
  collect()
   
# Search for records using a Well-known Text string (WKT)
wkt <- "POLYGON((142.3 -29.0,142.7 -29.1,142.7 -29.4,142.3 -29.0))"
galah_call() |>
  galah_identify("vulpes") |>
  galah_geolocate(wkt) |>
  atlas_counts()

# Search for records within the bounding box extracted from an `sf` object
location <- "POLYGON((142.3 -29.0,142.7 -29.1,142.7 -29.4,142.3 -29.0))" |>
  sf::st_as_sfc()
galah_call() |>
  galah_identify("vulpes") |>
  galah_geolocate(location, type = "bbox") |>
  atlas_counts()

# Search for records using a bounding box of coordinates
b_box <- sf::st_bbox(c(xmin = 143, xmax = 148, ymin = -29, ymax = -28), 
                     crs = sf::st_crs("WGS84"))
galah_call() |>
  galah_identify("reptilia") |>
  galah_geolocate(b_box, type = "bbox") |>
  atlas_counts()

# Search for records using a bounding box in a `tibble` or `data.frame`
b_box <- tibble::tibble(xmin = 148, ymin = -29, xmax = 143, ymax = -21)
galah_call() |>
  galah_identify("vulpes") |>
  galah_geolocate(b_box, type = "bbox") |>
  atlas_counts()
}
}
