#' Generate a citation for occurrence data
#'
#' If a `data.frame` was generated using [atlas_occurrences()],
#' and the `mint_doi` argument was set to `TRUE`, the DOI associated
#' with that dataset is appended to the resulting `data.frame` as an
#' attribute. This function simply formats that DOI as a citation that can be 
#' included in a scientific publication. Please also consider citing this 
#' package, using the information in `citation("galah")`.
#' @param data data.frame: occurrence data generated by 
#' [atlas_occurrences()]
#' @return A string containing the citation for that dataset.
#' @export

atlas_citation <- function(data) {
  if (is.na(attributes(data)$doi)) {
    search_url <- attributes(data)$search_url
    if (is.na(search_url)) {
      bullets <- c(
        "This dataset does not have a DOI or associated search url.",
        i = "Please download again and then use `ala_citation`."
      )
      abort(bullets, call = caller_env())
    }
    return(paste0("ALA occurrence download accessed from R with galah",
           " (https://github.com/AtlasOfLivingAustralia/galah/) on ",
           Sys.Date(), ". Search url: ", search_url)
    )
  }
  paste0("ALA occurrence download ", attributes(data)$doi,
         ". Accessed from R with ", galah_version_string(),
         " (https://github.com/AtlasOfLivingAustralia/galah/) on ",
         Sys.Date(), ".")
}
