# Create a new project
dummypackage <- tempfile("inflate.tests")
dir.create(dummypackage)

# {fusen} steps
fill_description(pkg = dummypackage, fields = list(Title = "Dummy Package"))
dev_file <- suppressMessages(add_flat_template(pkg = dummypackage, overwrite = TRUE, open = FALSE))
flat_file <- dev_file[grepl("flat_", dev_file)]

usethis::with_project(dummypackage, {
  # More complicated example for tests
  testfile <- "tests-templates/dev-template-tests.Rmd"
  file.copy(
    system.file(testfile, package = "fusen"),
    flat_file,
    overwrite = TRUE
  )
  suppressMessages(
    inflate(pkg = dummypackage, flat_file = flat_file,
            vignette_name = "Get started", check = FALSE,
            open_vignette = FALSE)
  )

  test_that("inflate() worked correctly", {
    # R files
    my_median_file <- file.path(dummypackage, "R", "my_median.R")
    expect_true(file.exists(my_median_file))
    my_other_median_file <- file.path(dummypackage, "R", "my_other_median.R")
    expect_true(file.exists(my_other_median_file))
    my_third_median_file <- file.path(dummypackage, "R", "my_third_median.R")
    expect_true(file.exists(my_third_median_file))
    my_sixth_median_file <- file.path(dummypackage, "R", "my-sixth-median_function.R")
    expect_true(file.exists(my_sixth_median_file))
    myuppercasefunctionfile <- file.path(dummypackage, "R", "myuppercasefunction.R")
    expect_true(file.exists(myuppercasefunctionfile))

    # Found with chunk named `fun`
    my_noroxfunctionfile <- file.path(dummypackage, "R", "my_norox.R")
    expect_true(file.exists(my_noroxfunctionfile))
    # Found with chunk named `fun-norox2`
    my_norox2functionfile <- file.path(dummypackage, "R", "my_norox2.R")
    expect_true(file.exists(my_norox2functionfile))
    # Found with chunk named `fun_space`
    my_spacefunctionfile <- file.path(dummypackage, "R", "my_space.R")
    expect_true(file.exists(my_spacefunctionfile))
    my_space2functionfile <- file.path(dummypackage, "R", "my_space2.R")
    expect_true(file.exists(my_space2functionfile))

    # examples in R files
    my_median_lines <- readLines(my_median_file)
    expect_equal(my_median_lines[1], "# Generated by fusen: do not edit by hand")
    expect_true(all(my_median_lines[12:14] == c(
      "#' @examples", "#' my_median(2:20)", "#' my_median(1:12)"
    )))
    my_other_median_lines <- readLines(my_other_median_file)
    expect_equal(my_other_median_lines[1], "# Generated by fusen: do not edit by hand")
    expect_true(all(my_other_median_lines[12:15] == c(
      "#' @examples", "#' my_other_median(1:12)",
      "#' my_other_median(8:20)", "#' my_other_median(20:50)"
    )))
    my_third_median_lines <- readLines(my_third_median_file)
    # _no example
    expect_true(all(!grepl("#' @examples", my_third_median_lines)))
    # dot in name
    my_sixth_median_lines <- readLines(my_sixth_median_file)
    expect_true(all(my_sixth_median_lines[11:13] == c(
      "#' @examples", "#' my.sixth.median_function(1:12)", "#' my.sixth.median_function(8:20)"
    )))
    # _no roxygen at all
    my_norox_lines <- readLines(my_noroxfunctionfile)
    expect_true(all(my_norox_lines == c(
      "# Generated by fusen: do not edit by hand", "",
      "#' @noRd", "my_norox <- function(x) {", "  x + 10", "}"
    )))

    # _no roxygen but examples
    my_norox2_lines <- readLines(my_norox2functionfile)
    expect_equal(my_norox2_lines, c(
      "# Generated by fusen: do not edit by hand", "",
      "#' @noRd", "#' @examples",
      "#' \\dontrun{", "#' # comment",
      "#' my_norox2(10)", "#' }",
      "#' ",
      "#' \\dontrun{",
      "#' # comment",
      "#' my_norox2(12)", "#' }",
      "my_norox2 <- function(x) {", "  x + 10", "}"
    ))
    # _extra empty line and examples
    my_space_lines <- readLines(my_spacefunctionfile)
    expect_true(all(my_space_lines[8:12] == c(
      "#' @examples", "#' my_space(10)", "#' @export", "",  "my_space <- function(x) {"
    )))
    # _extra empty line and noRd
    my_space2_lines <- readLines(my_space2functionfile)
    expect_true(all(my_space2_lines[8:10] == c(
      "#' @noRd", "",  "my_space2 <- function(x) {"
    )))

    # vignette
    the_vignette <- file.path(dummypackage, "vignettes", "get-started.Rmd")
    expect_true(file.exists(the_vignette))
    vignette_lines <- readLines(the_vignette)
    # Do not edit by hand after yaml
    pos.dash <- grep("---", vignette_lines)
    donotedit <-grep(
      "<!-- This vignette is generated by fusen: do not edit by hand -->",
      vignette_lines)
    expect_equal(length(donotedit), 1)
    expect_true(pos.dash[2] < donotedit)
    # No dev chunks in the vignette
    expect_false(any(grepl("```{r dev}", vignette_lines, fixed = TRUE)))
    expect_false(any(grepl("```{r development-1", vignette_lines, fixed = TRUE)))

    # tests
    test_file <- file.path(dummypackage, "tests", "testthat", "test-my_median.R")
    expect_true(file.exists(test_file))
    test_lines <- readLines(test_file)
    expect_equal(
      test_lines,
      c(
        "# Generated by fusen: do not edit by hand",
        "",
        "test_that(\"my_median works properly and show error if needed\", {",
        "  expect_error(my_median(\"text\"))",
        "})")
    )
    expect_true(file.exists(
      file.path(dummypackage, "tests", "testthat", "test-my_other_median.R")
    ))
    expect_true(file.exists(
      file.path(dummypackage, "tests", "testthat", "test-my-sixth-median_function.R")
    ))
    expect_true(file.exists(
      file.path(dummypackage, "tests", "testthat", "test-myuppercasefunction.R")
    ))
    # no test
    expect_false(file.exists(
      file.path(dummypackage, "tests", "testthat", "test-my_norox.R")
    ))
    expect_false(file.exists(
      file.path(dummypackage, "tests", "testthat", "test-my_space2.R")
    ))
    # Namespace
    expect_true(file.exists(file.path(dummypackage, "NAMESPACE")))
  })
})


# Test package no check errors ----
usethis::with_project(dummypackage, {
  test_that("inflate() output error", {
    # Do not check inside check if on CRAN
    skip_on_os(os = c("windows", "solaris"))

    # If this check is run inside a not "--as-cran" check, then it wont work as expected
    check_out <- rcmdcheck::rcmdcheck(dummypackage, quiet = TRUE,
                                      args = c("--no-manual"))

    # No errors
    expect_true(length(check_out[["errors"]]) == 0)
    # 1 warning = License
    # expect_true(length(check_out[["warnings"]]) == 1)
    expect_true(grepl("license", check_out[["warnings"]][1]))
    # No Notes or only one if CRAN
    expect_true(length(check_out[["notes"]]) <= 1)
    if (length(check_out[["notes"]]) == 1) {
      # if tested as cran
      # 1 note on CRAN for new submission
      print(check_out[["notes"]])
      expect_true(grepl("New submission", check_out[["notes"]][1]))
    } else {
      expect_true(length(check_out[["notes"]]) == 0)
    }
  })
  # Clean R, tests and vignettes
  unlink(file.path(dummypackage, "R"), recursive = TRUE)
  unlink(file.path(dummypackage, "vignettes"), recursive = TRUE)
  unlink(file.path(dummypackage, "tests"), recursive = TRUE)
})

# Test no problem with special character in YAML ----
if (packageVersion("parsermd") > "0.1.2") {
  dummypackage.special <- tempfile("dummypackage.special")
  dir.create(dummypackage.special)

  # {fusen} steps
  fill_description(pkg = dummypackage.special, fields = list(Title = "Dummy Package"))
  dev_file <- add_flat_template(pkg = dummypackage.special, overwrite = TRUE, open = FALSE)
  flat_file <- dev_file[grepl("flat_", dev_file)]

  usethis::with_project(dummypackage.special, {

    testfile <- "tests-templates/dev-template-tests-special-char.Rmd"
    file.copy(
      system.file(testfile, package = "fusen"),
      flat_file,
      overwrite = TRUE
    )

    suppressMessages(
      inflate(pkg = dummypackage.special, flat_file = flat_file,
              vignette_name = "Get started", check = FALSE,
              open_vignette = FALSE)
    )

    test_that("inflate with special yaml worked correctly", {
      # R files
      my_median_file <- file.path(dummypackage.special, "R", "my_median.R")
      expect_true(file.exists(my_median_file))
    })
  })
}

# Test no attachment and no check when asked ----
unlink(file.path(dummypackage, "DESCRIPTION"), recursive = TRUE)
fill_description(pkg = dummypackage, fields = list(Title = "Dummy Package"))
dev_file <- add_flat_template(pkg = dummypackage, overwrite = TRUE, open = FALSE)
flat_file <- dev_file[grepl("flat_", dev_file)]

usethis::with_project(dummypackage, {
  suppressMessages(
    inflate(pkg = dummypackage, flat_file = flat_file, vignette_name = "Get started",
            check = FALSE, document = FALSE, open_vignette = FALSE)
  )
  desc_lines <- readLines(file.path(dummypackage, "DESCRIPTION"))

  test_that("no attachment run", {
    expect_false("Imports:" %in% desc_lines)
  })

  # Clean R, tests and vignettes
  unlink(file.path(dummypackage, "R"), recursive = TRUE)
  unlink(file.path(dummypackage, "vignettes"), recursive = TRUE)
  unlink(file.path(dummypackage, "tests"), recursive = TRUE)
})


# Tests no errors - no example, no tests ----
usethis::with_project(dummypackage, {
  file.copy(
    system.file("tests-templates/dev-template-no-example-no-tests.Rmd", package = "fusen"),
    flat_file,
    overwrite = TRUE
  )
  suppressMessages(
    inflate(pkg = dummypackage, flat_file = flat_file,
            vignette_name = "Get started", check = FALSE,
            open_vignette = FALSE)
  )

  test_that("inflate() output no error", {
    expect_true(file.exists(file.path(dummypackage, "vignettes", "get-started.Rmd")))
    expect_true(file.exists(file.path(dummypackage, "R", "my_median.R")))
    expect_true(!file.exists(file.path(dummypackage, "tests", "testthat", "test-my_median.R")))
  })
  # Clean R, tests and vignettes
  unlink(file.path(dummypackage, "R"), recursive = TRUE)
  unlink(file.path(dummypackage, "vignettes"), recursive = TRUE)
  unlink(file.path(dummypackage, "tests"), recursive = TRUE)
})

# Tests no errors - empty ----
usethis::with_project(dummypackage, {
  file.copy(
    system.file("tests-templates/dev-template-test-parse-nothing.Rmd", package = "fusen"),
    flat_file,
    overwrite = TRUE
  )
  test_that("inflate() output message", {
    suppressMessages(
      expect_message(
        inflate(pkg = dummypackage, flat_file = flat_file,
                vignette_name = "Get started", check = FALSE,
                open_vignette = FALSE)
      )
    )
  })
  # Clean R, tests and vignettes
  unlink(file.path(dummypackage, "R"), recursive = TRUE)
  unlink(file.path(dummypackage, "vignettes"), recursive = TRUE)
  unlink(file.path(dummypackage, "tests"), recursive = TRUE)
})

# Tests errors - vignette already exists ----
usethis::with_project(dummypackage, {
  suppressMessages(
    inflate(pkg = dummypackage, flat_file = flat_file,
            vignette_name = "Get started",
            check = FALSE, overwrite = "yes",
            open_vignette = FALSE)
  )

  test_that("inflate() output error when second time (not interactive)", {
    expect_error(
      suppressMessages(
        inflate(pkg = dummypackage, flat_file = flat_file,
                vignette_name = "Get started",
                check = FALSE,
                open_vignette = FALSE))
    )
    expect_error(
      suppressMessages(
        inflate(pkg = dummypackage, flat_file = flat_file,
                vignette_name = "Get started",
                check = FALSE, overwrite = 'no',
                open_vignette = FALSE))
    )
  })

  # No error with overwrite = 'yes'
  suppressMessages(
    inflate(pkg = dummypackage, flat_file = flat_file, vignette_name = "Get started",
            check = FALSE, overwrite = "yes", open_vignette = FALSE)
  )

  test_that("inflate() output no error", {
    expect_true(file.exists(file.path(dummypackage, "vignettes", "get-started.Rmd")))
  })

  # Clean R, tests and vignettes
  unlink(file.path(dummypackage, "R"), recursive = TRUE)
  unlink(file.path(dummypackage, "vignettes"), recursive = TRUE)
  unlink(file.path(dummypackage, "tests"), recursive = TRUE)
})

# Tests errors - duplicate functions ----
usethis::with_project(dummypackage, {
  file.copy(
    system.file("tests-templates/dev-template-stop-duplicate-fun.Rmd", package = "fusen"),
    flat_file,
    overwrite = TRUE
  )
  test_that("inflate() output error duplicate functions", {
    expect_error(
      suppressMessages(
        inflate(pkg = dummypackage, flat_file = flat_file,
                vignette_name = "Get started", check = FALSE,
                open_vignette = FALSE))
    )
  })
  # Clean R, tests and vignettes
  unlink(file.path(dummypackage, "R"), recursive = TRUE)
  unlink(file.path(dummypackage, "vignettes"), recursive = TRUE)
  unlink(file.path(dummypackage, "tests"), recursive = TRUE)

  # Tests errors - duplicate chunk names
  file.copy(
    system.file("tests-templates/dev-template-stop-duplicate-label.Rmd", package = "fusen"),
    flat_file,
    overwrite = TRUE
  )
  test_that("inflate() output error duplicate label names for vignette", {
    expect_error(
      suppressMessages(
        inflate(pkg = dummypackage, flat_file = flat_file,
                vignette_name = "Get started", check = FALSE,
                open_vignette = FALSE)
      )
    )
  })
  # Clean R, tests and vignettes
  unlink(file.path(dummypackage, "R"), recursive = TRUE)
  unlink(file.path(dummypackage, "vignettes"), recursive = TRUE)
  unlink(file.path(dummypackage, "tests"), recursive = TRUE)
})

# Test no errors - inflate with .Rproj and no .here ----
usethis::with_project(dummypackage, {
  file.remove(file.path(dummypackage, ".here"))
  file.remove(file.path(dummypackage, ".Rbuildignore"))
  cat("", file = file.path(dummypackage, 'dummy.Rproj'))

  # Add
  # {fusen} steps
  dev_file <- add_flat_template(pkg = dummypackage, overwrite = TRUE, open = FALSE)
  suppressMessages(
    inflate(pkg = dummypackage, flat_file = flat_file,
            vignette_name = "Get started", check = FALSE,
            open_vignette = FALSE)
  )

  test_that("add_flat_template inflates with .Rproj and no .here", {
    expect_true(file.exists(flat_file))
    expect_false(file.exists(file.path(dummypackage, ".here")))

    rbuildignore_file <- file.path(dummypackage, ".Rbuildignore")
    expect_true(file.exists(rbuildignore_file))
    rbuildignore_lines <- readLines(rbuildignore_file)
    expect_true(any(grepl("dev", rbuildignore_lines, fixed = TRUE)))
    expect_false(any(grepl("[.]here", rbuildignore_lines)))

    # R files
    my_median_file <- file.path(dummypackage, "R", "my_median.R")
    expect_true(file.exists(my_median_file))
    # vignette
    expect_true(file.exists(file.path(dummypackage, "vignettes", "get-started.Rmd")))
    # tests
    expect_true(file.exists(
      file.path(dummypackage, "tests", "testthat", "test-my_median.R")
    ))
  })

  # Clean R, tests and vignettes
  unlink(file.path(dummypackage, "R"), recursive = TRUE)
  unlink(file.path(dummypackage, "vignettes"), recursive = TRUE)
  unlink(file.path(dummypackage, "tests"), recursive = TRUE)
})
# Delete dummy package
unlink(dummypackage, recursive = TRUE)

# Test no errors - clean vignette_name for vignette ----
# Create a new project
dummypackage <- tempfile("clean.vignette")
dir.create(dummypackage)

# {fusen} steps
fill_description(pkg = dummypackage, fields = list(Title = "Dummy Package"))
dev_file <- suppressMessages(add_flat_template(pkg = dummypackage, overwrite = TRUE, open = FALSE))
flat_file <- dev_file[grepl("flat_", dev_file)]

usethis::with_project(dummypackage, {
  suppressMessages(
    inflate(pkg = dummypackage, flat_file = flat_file,
            vignette_name = "# y  _ p n@ \u00E9 ! 1", check = FALSE,
            open_vignette = FALSE)
  )
  # Vignette name is also cleaned by {usethis} for special characters
  vignette_path <- file.path(dummypackage, "vignettes", "y-p-n---1.Rmd")

  test_that("vignette is created with clean vignette_name", {
    expect_true(file.exists(vignette_path))
    # usethis::use_vignette writes in UTF-8
    vig_lines <- readLines(vignette_path, encoding = "UTF-8")
    expect_true(sum(grepl(enc2utf8("# y  _ p n@ \u00E9 ! 1"), vig_lines, fixed = TRUE)) == 2)
    expect_equal(vig_lines[2], enc2utf8('title: "# y  _ p n@ \u00E9 ! 1"'))
    expect_true(sum(grepl("y-p-n---1", vig_lines, fixed = TRUE)) == 0)
  })

  # Clean R, tests and vignettes
  unlink(file.path(dummypackage, "R"), recursive = TRUE)
  unlink(file.path(dummypackage, "vignettes"), recursive = TRUE)
  unlink(file.path(dummypackage, "tests"), recursive = TRUE)
})
# Delete dummy package
unlink(dummypackage, recursive = TRUE)


# Test stop when no DESCRIPTION file ----
dummypackage <- tempfile("descpackage")
dir.create(dummypackage)
dev_file <- add_flat_template(pkg = dummypackage, overwrite = TRUE, open = FALSE)
flat_file <- dev_file[grepl("flat_", dev_file)]

usethis::with_project(dummypackage, {
  test_that("stop when no DESCRIPTION file", {
    expect_error(
      suppressMessages(
        inflate(pkg = dummypackage, flat_file = flat_file,
                check = FALSE, open_vignette = FALSE)
      ),
      "DESCRIPTION file")
  })
})

# Delete dummy package
unlink(dummypackage, recursive = TRUE)
if (exists("dummypackage.special")) {
  unlink(dummypackage.special, recursive = TRUE)
}

# Deal with noRd, examples and dontrun ----
#' stop() if @noRd but there is an example...
#' Or suggests \dontrun{}, but need to be taken into account in vignette

dummypackage <- tempfile("nordpackage")
dir.create(dummypackage)
dev_file <- add_flat_template(pkg = dummypackage, overwrite = TRUE, open = FALSE)
flat_file <- dev_file[grepl("flat_", dev_file)]
fill_description(pkg = dummypackage, fields = list(Title = "Dummy Package"))

usethis::with_project(dummypackage, {
  file.copy(
    system.file("tests-templates/dev-template-nord-but-example.Rmd", package = "fusen"),
    flat_file,
    overwrite = TRUE
  )

  test_that("Deal with noRd but examples", {
    # No error
    expect_error(
      suppressMessages(
        inflate(pkg = dummypackage, flat_file = flat_file,
                check = FALSE, open_vignette = FALSE)
      ),
      regexp = NA)
    # Check error
    # skip_on_os(os = c("windows", "solaris"))
    skip_on_cran()

    # Could not find function "my_norox2"
    expect_error(rcmdcheck::rcmdcheck(dummypackage, quiet = TRUE,
                                      args = c("--no-manual")))
  })
})

# Delete dummy package
unlink(dummypackage, recursive = TRUE)

# Test checks all templates with inflate dots (...) ----
alltemp <- tempfile("all.templates.inflate")
dir.create(alltemp)

for (pkgname in c("full", "teaching", "minimal")) {
  # No "additional" with create_fusen
  # {fusen} steps
  path_foosen <- file.path(alltemp, pkgname)
  dev_file <- create_fusen(path_foosen, template = pkgname, open = FALSE)
  flat_file <- dev_file[grepl("flat_", dev_file)]

  usethis::with_project(path_foosen, {
    # Do not check inside check if on CRAN
    skip_on_os(os = c("windows", "solaris"))

    fill_description(pkg = path_foosen, fields = list(Title = "Dummy Package"))
    usethis::use_gpl_license()

    test_that(paste("Check returns OK for template", pkgname), {
      # extract the 'inflate' line in the flat file
      # Add pkg, check, quiet, args, overwrite
      # And inflate
      flat_lines <- readLines(flat_file)
      w.start <- grep("fusen::inflate\\(flat_file", flat_lines)
      w.end <- grep("\\)", flat_lines)
      w.end <- w.end[w.end >= w.start][1]
      inflate_lines <- flat_lines[w.start:w.end]

      if (!interactive()) {
        print(" ==== Not interactive ====")
        # Modify with extra values
        extra_params <- glue(
          'fusen::inflate(pkg = "{path_foosen}",
      check = FALSE, quiet = TRUE, args = c("--no-manual"),
      overwrite = TRUE, open_vignette = FALSE, '
        )
        to_inflate <- gsub("fusen::inflate\\(", extra_params, inflate_lines)

        # No redirection of stdout/stderr when non-interactive
        expect_error(
          suppressMessages(
            eval(parse(text = to_inflate))
          ),
          regexp = NA)

        # Run rcmdcheck
        # Do not check inside check if on CRAN
        skip_on_os(os = c("windows", "solaris"))


        # If this check is run inside a not "--as-cran" check, then it wont work as expected
        check_out <- rcmdcheck::rcmdcheck(path_foosen, quiet = TRUE,
                             args = c("--no-manual"))

        # No errors
        expect_true(length(check_out[["errors"]]) == 0)
        expect_true(length(check_out[["warnings"]]) <= 1)
        if (length(check_out[["warnings"]]) == 1) {
          expect_true(grepl("there is no package called", check_out[["warnings"]]))
        }
        #  ‘MASS’
        # print(" -- warnings --")
        # print(check_out[["warnings"]])
        skip_on_cran()
        expect_true(length(check_out[["notes"]]) == 0)
      } else {
        print(" ==== Interactive ====")
        # Modify with extra values
        extra_params <- glue(
          'fusen::inflate(pkg = "{path_foosen}",
      check = TRUE, quiet = TRUE, args = c("--no-manual"),
      overwrite = TRUE, open_vignette = FALSE, '
        )
        to_inflate <- gsub("fusen::inflate\\(", extra_params, inflate_lines)

        # No redirection of stdout/stderr when non-interactive
        expect_error(
          suppressMessages(
            eval(parse(text = to_inflate))
          ),
          regexp = NA)
      }

      skip_if_not(interactive())
      # Needs MASS, lattice, Matrix installed
      # quiet and checkdir
      checkdir <- file.path(alltemp, paste0("checkout", pkgname))
      extra_params <- glue(
        'fusen::inflate(pkg = "{path_foosen}",
      check = TRUE, check_dir = "{checkdir}",
      quiet = TRUE, args = c("--no-manual"),
      overwrite = TRUE, open_vignette = FALSE, '
      )
      to_inflate <- gsub("fusen::inflate\\(", extra_params, inflate_lines)

      expect_error(
        suppressMessages(
          eval(parse(text = to_inflate))
        ),
        regexp = NA)

      # Should not be any errors with templates
      check_lines <- readLines(file.path(checkdir, paste0(basename(path_foosen), ".Rcheck"), "00check.log"))
      expect_equal(check_lines[length(check_lines)], "Status: OK")
      unlink(checkdir, recursive = TRUE)
    })
    # })
  })

} # end of template loop
# Delete dummy package
unlink(alltemp, recursive = TRUE)

# Tests empty chunks ----
dummypackage <- tempfile("empty.chunks")
dir.create(dummypackage)

# {fusen} steps
fill_description(pkg = dummypackage, fields = list(Title = "Dummy Package"))
dev_file <- suppressMessages(add_flat_template(pkg = dummypackage, overwrite = TRUE, open = FALSE))
flat_file <- dev_file[grepl("flat_", dev_file)]

usethis::with_project(dummypackage, {
  file.copy(
    system.file("tests-templates/dev-template-empty-not-function.Rmd", package = "fusen"),
    flat_file,
    overwrite = TRUE
  )
  usethis::use_gpl_license()
  # Add cars data
  usethis::use_data(cars)

  test_that("inflate() output no error", {
    expect_error(
      suppressMessages(
        inflate(pkg = dummypackage, flat_file = flat_file,
                vignette_name = "Get started", check = FALSE,
                open_vignette = FALSE)),
      regexp = NA
    )

    # R files with chunk content - Name after title as function name is NA
    expect_equal(list.files(file.path(dummypackage, "R")),
                 c("my-data-doc.R", "my-pkg-doc.R", "onload.R"))
    pkgdoc <- file.path(dummypackage, "R", "my-pkg-doc.R")
    expect_true(file.exists(pkgdoc))
    pkgdoc_lines <- readLines(pkgdoc)
    expect_equal(length(pkgdoc_lines), 10)
    expect_equal(pkgdoc_lines[4], "\"_PACKAGE\"")

    datadoc <- file.path(dummypackage, "R", "my-data-doc.R")
    expect_true(file.exists(datadoc))
    datadoc_lines <- readLines(datadoc)
    expect_equal(length(datadoc_lines), 13)
    expect_equal(datadoc_lines[13], "\"cars\"")

    myonloadfile <- file.path(dummypackage, "R", "onload.R")
    expect_true(file.exists(myonloadfile))
    # No dot in name
    myonload_lines <- readLines(myonloadfile)
    expect_true(all(myonload_lines[6:8] == c(
      ".onLoad <- function(libname, pkgname) {",
      "        the_message()",
      "}"
    )))

    checkdir <- tempfile("dircheck")
    # Disable checking for future file timestamps
    if (!interactive()) {
      expect_error(
        suppressMessages(
          inflate(pkg = dummypackage, flat_file = flat_file,
                  vignette_name = "Get started", check = FALSE,
                  quiet = TRUE,
                  overwrite = TRUE, open_vignette = FALSE)),
        regexp = NA
      )

      # Do not check inside check if on CRAN
      skip_on_os(os = c("windows", "solaris"))

      # If this check is run inside a not "--as-cran" check, then it wont work as expected
      check_out <- rcmdcheck::rcmdcheck(dummypackage, quiet = TRUE,
                                        args = c("--no-manual"),
                                        check_dir = checkdir)

      # No errors
      expect_true(length(check_out[["errors"]]) == 0)
      expect_true(length(check_out[["warnings"]]) <= 1)
      if (length(check_out[["warnings"]]) == 1) {
        expect_true(grepl("there is no package called", check_out[["warnings"]]))
      }
      #  ‘MASS’
      # print(" -- warnings --")
      # print(check_out[["warnings"]])
      skip_on_cran()
      expect_true(length(check_out[["notes"]]) == 0)
    } else {
      expect_error(
        suppressMessages(
          inflate(pkg = dummypackage, flat_file = flat_file,
                  vignette_name = "Get started", check = TRUE,
                  check_dir = checkdir, quiet = TRUE,
                  overwrite = TRUE, open_vignette = FALSE)),
        regexp = NA
      )

      # Should not be any errors with templates in interactive
      check_lines <- readLines(file.path(checkdir, paste0(basename(dummypackage), ".Rcheck"), "00check.log"))
      expect_equal(check_lines[length(check_lines)], "Status: OK")
      unlink(checkdir, recursive = TRUE)
    }


  })
})
unlink(dummypackage, recursive = TRUE)

# Tests r6 chunks ----
dummypackage <- tempfile("r6class")
dir.create(dummypackage)

# {fusen} steps
fill_description(pkg = dummypackage, fields = list(Title = "Dummy Package"))
dev_file <- suppressMessages(add_flat_template(pkg = dummypackage, overwrite = TRUE, open = FALSE))
flat_file <- dev_file[grepl("flat_", dev_file)]

usethis::with_project(dummypackage, {
  file.copy(
    system.file("tests-templates/dev-template-r6class.Rmd", package = "fusen"),
    flat_file,
    overwrite = TRUE
  )
  usethis::use_gpl_license()
  # Add cars data
  usethis::use_data(cars)

  test_that("inflate() output no error with R6", {
    expect_error(
      suppressMessages(
        inflate(pkg = dummypackage, flat_file = flat_file,
                vignette_name = "Get started", check = FALSE,
                open_vignette = FALSE,
                # To avoid having {R6} in suggests
                document = FALSE)),
      regexp = NA
    )

    r6doc <- file.path(dummypackage, "R", "simple.R")
    expect_true(file.exists(r6doc))
    r6doc_lines <- readLines(r6doc)
    expect_equal(length(r6doc_lines), 9)
    expect_equal(r6doc_lines[4], "Simple <- R6::R6Class(\"Simple\",")

    r6doc <- file.path(dummypackage, "R", "simple2.R")
    expect_true(file.exists(r6doc))
    r6doc_lines <- readLines(r6doc)
    expect_equal(length(r6doc_lines), 9)
    expect_equal(r6doc_lines[4], "Simple2 <- R6Class(\"Simple2\",")
  })
})
unlink(dummypackage, recursive = TRUE)


# Test create_vignette_head ----

# Create a new project
dummypackage <- tempfile("vignette.head")
dir.create(dummypackage)

# {fusen} steps
fill_description(pkg = dummypackage, fields = list(Title = "Dummy Package"))

test_that("create_vignette_head works", {
  usethis::with_project(dummypackage, {
    head <- create_vignette_head(pkg = dummypackage, vignette_name = "My Super Vignette")
    expect_true(grepl('title: "My Super Vignette"', head))
    expect_true(grepl('  %\\VignetteIndexEntry{My Super Vignette}', head, fixed = TRUE))
    expect_true(grepl(paste0('library(', basename(dummypackage) ,')'), head, fixed = TRUE))
  })
})
unlink(dummypackage, recursive = TRUE)

# Depreaction of name and rmd ----
# Create a new project
dummypackage <- tempfile("inflate.deprecated")
dir.create(dummypackage)

# {fusen} steps
fill_description(pkg = dummypackage, fields = list(Title = "Dummy Package"))
dev_file <- suppressMessages(add_flat_template(pkg = dummypackage, overwrite = TRUE, open = FALSE))
flat_file <- dev_file[grepl("flat_", dev_file)]

test_that("rmd and name are deprecated works", {
  usethis::with_project(dummypackage, {
    expect_warning(
      inflate(pkg = ".",
              # flat_file = flat_file,
              rmd = flat_file,
              vignette_name = "Get started",
              check = FALSE, document = TRUE,
              overwrite = TRUE, open_vignette = FALSE),
      regexp = 'The `rmd` argument'
    )
    expect_warning(
      inflate(pkg = ".",
              flat_file = flat_file,
              # vignette_name = "Get started",
              name = "Get started",
              check = FALSE, document = TRUE,
              overwrite = TRUE, open_vignette = FALSE),
      regexp = 'The `name` argument'
    )
  })
})
unlink(dummypackage, recursive = TRUE)

# Test No vignette ----
dummypackage <- tempfile("inflate.no.vignette")
dir.create(dummypackage)

# {fusen} steps
fill_description(pkg = dummypackage, fields = list(Title = "Dummy Package"))
dev_file <- suppressMessages(add_flat_template(pkg = dummypackage, overwrite = TRUE, open = FALSE))
flat_file <- dev_file[grepl("flat_", dev_file)]

usethis::with_project(dummypackage, {

  test_that("inflate() worked correctly", {
    expect_message(
      inflate(pkg = dummypackage, flat_file = flat_file,
              vignette_name = NA, check = FALSE,
              open_vignette = FALSE),
      regexp = "no vignette created"
    )
    expect_equal(length(list.files(file.path(dummypackage, "vignettes"))), 0)

    expect_message(
      inflate(pkg = dummypackage, flat_file = flat_file,
              vignette_name = NULL, check = FALSE,
              open_vignette = FALSE),
      regexp = "no vignette created"
    )
    expect_equal(length(list.files(file.path(dummypackage, "vignettes"))), 0)

    expect_message(
      inflate(pkg = dummypackage, flat_file = flat_file,
              vignette_name = "", check = FALSE,
              open_vignette = FALSE),
      regexp = "no vignette created"
    )
    expect_equal(length(list.files(file.path(dummypackage, "vignettes"))), 0)

    expect_error(
      suppressMessages(
        inflate(pkg = dummypackage, flat_file = flat_file,
                vignette_name = "It works then", check = FALSE,
                open_vignette = FALSE)
      ),
      regexp = NA
    )
    expect_equal(list.files(file.path(dummypackage, "vignettes")), "it-works-then.Rmd")
  })
})
unlink(dummypackage, recursive = TRUE)

# Two funs same file - dev-template-two-fun-same-title.Rmd ----

# Create a new project
dummypackage <- tempfile("inflate.twofuns")
dir.create(dummypackage)

# {fusen} steps
fill_description(pkg = dummypackage, fields = list(Title = "Dummy Package"))
dev_file <- suppressMessages(add_flat_template(pkg = dummypackage, overwrite = TRUE, open = FALSE))
flat_file <- dev_file[grepl("flat_", dev_file)]

usethis::with_project(dummypackage, {
  # More complicated example for tests
  testfile <- "tests-templates/dev-template-two-fun-same-title.Rmd"
  file.copy(
    system.file(testfile, package = "fusen"),
    flat_file,
    overwrite = TRUE
  )
  suppressMessages(
    inflate(pkg = dummypackage, flat_file = flat_file,
            vignette_name = "Get started", check = FALSE,
            open_vignette = FALSE)
  )

  test_that("inflate() worked correctly", {
    # -- R files --
    my_median_file <- file.path(dummypackage, "R", "my_median.R")
    expect_true(file.exists(my_median_file))
    my_median2_file <- file.path(dummypackage, "R", "my_median2.R")
    expect_false(file.exists(my_median2_file))
    # chunk name
    my_chunk1_file <- file.path(dummypackage, "R", "fun_chunk1.R")
    expect_true(file.exists(my_chunk1_file))
    my_chunk1fun_file <- file.path(dummypackage, "R", "my_fun_chunk1.R")
    expect_false(file.exists(my_chunk1fun_file))
    my_chunk2fun_file <- file.path(dummypackage, "R", "my_fun_chunk2.R")
    expect_false(file.exists(my_chunk2fun_file))
    # same rdname
    my_rdname1_file <- file.path(dummypackage, "R", "same_rdname.R")
    expect_true(file.exists(my_rdname1_file))
    my_rdname1fun_file <- file.path(dummypackage, "R", "my_fun_rdname1.R")
    expect_false(file.exists(my_rdname1fun_file))
    my_rdname2fun_file <- file.path(dummypackage, "R", "my_fun_rdname2.R")
    expect_false(file.exists(my_rdname2fun_file))
    # same @filename
    my_filename1_file <- file.path(dummypackage, "R", "same_filename.R")
    expect_true(file.exists(my_filename1_file))
    my_filename1fun_file <- file.path(dummypackage, "R", "my_fun_filename1.R")
    expect_false(file.exists(my_filename1fun_file))
    my_filename2fun_file <- file.path(dummypackage, "R", "my_fun_filename2.R")
    expect_false(file.exists(my_filename2fun_file))
    # Same title
    r_lines <- readLines(my_median_file)
    expect_true(any(grepl("my_median <- function", r_lines)))
    expect_true(any(grepl("my_median2 <- function", r_lines)))
    # example at the right place
    expect_equal(r_lines[12:14],
                 c("#' @examples", "#' my_median(2:20)" , "#' my_median(1:12)")
    )
    expect_equal(r_lines[29:31],
                 c("#' @examples", "#' my_median2(2:20)" , "#' my_median2(1:12)")
    )
    # Same rdname
    r_lines <- readLines(my_rdname1_file)
    expect_true(any(grepl("my_fun_rdname1 <- function", r_lines)))
    expect_true(any(grepl("my_fun_rdname2 <- function", r_lines)))
    expect_equal(r_lines[13:15],
                 c("#' @examples" , "#' my_fun_rdname1(2:20)", "#' my_fun_rdname1(1:12)"))
    expect_equal(r_lines[21:25],
                 c("#' @rdname same_rdname" ,
                   "#' @importFrom stats median", "#' @export",
                   "#' @examples", "#' my_fun_rdname2(1:12)" ))
    # Same chunk name
    r_lines <- readLines(my_chunk1_file)
    expect_true(any(grepl("my_fun_chunk1 <- function", r_lines)))
    expect_true(any(grepl("my_fun_chunk2 <- function", r_lines)))
    # Same @filename
    r_lines <- readLines(my_filename1_file)
    expect_true(any(grepl("my_fun_filename1 <- function", r_lines)))
    expect_true(any(grepl("my_fun_filename2 <- function", r_lines)))
    # @filename cleaned in R file
    expect_false(any(grepl("@filename", r_lines)))

    # -- doc files --
    my_median_file <- file.path(dummypackage, "man", "my_median.Rd")
    expect_true(file.exists(my_median_file))
    my_median2_file <- file.path(dummypackage, "man", "my_median2.Rd")
    expect_true(file.exists(my_median2_file))
    # chunk name
    my_median_file <- file.path(dummypackage, "man", "my_fun_chunk1.Rd")
    expect_true(file.exists(my_median_file))
    my_median2_file <- file.path(dummypackage, "man", "my_fun_chunk2.Rd")
    expect_true(file.exists(my_median2_file))
    # rdname
    my_median_file <- file.path(dummypackage, "man", "my_fun_rdname1.Rd")
    expect_false(file.exists(my_median_file))
    my_median2_file <- file.path(dummypackage, "man", "my_fun_rdname2.Rd")
    expect_false(file.exists(my_median2_file))
    my_median3_file <- file.path(dummypackage, "man", "same_rdname.Rd")
    expect_true(file.exists(my_median3_file))
    # filename
    my_median_file <- file.path(dummypackage, "man", "my_fun_filename1.Rd")
    expect_true(file.exists(my_median_file))
    my_median2_file <- file.path(dummypackage, "man", "my_fun_filename2.Rd")
    expect_true(file.exists(my_median2_file))

    # -- test files --
    my_median_file <- file.path(dummypackage, "tests", "testthat", "test-my_median.R")
    expect_true(file.exists(my_median_file))
    my_median2_file <- file.path(dummypackage, "tests", "testthat", "test-my_median2.R")
    expect_false(file.exists(my_median2_file))
    # chunk name
    my_chunk1_file <- file.path(dummypackage, "tests", "testthat", "test-fun_chunk1.R")
    expect_true(file.exists(my_chunk1_file))
    my_chunk1fun_file <- file.path(dummypackage, "tests", "testthat", "test-my_fun_chunk1.R")
    expect_false(file.exists(my_chunk1fun_file))
    my_chunk2fun_file <- file.path(dummypackage, "tests", "testthat", "test-my_fun_chunk2.R")
    expect_false(file.exists(my_chunk2fun_file))
    # same rdname
    my_rdname1_file <- file.path(dummypackage, "tests", "testthat", "test-same_rdname.R")
    expect_true(file.exists(my_rdname1_file))
    my_rdname1fun_file <- file.path(dummypackage, "tests", "testthat", "test-my_fun_rdname1.R")
    expect_false(file.exists(my_rdname1fun_file))
    my_rdname2fun_file <- file.path(dummypackage, "tests", "testthat", "test-my_fun_rdname2.R")
    expect_false(file.exists(my_rdname2fun_file))
    # same @filename
    my_filename1_file <- file.path(dummypackage, "tests", "testthat", "test-same_filename.R")
    expect_true(file.exists(my_filename1_file))
    my_filename1fun_file <- file.path(dummypackage, "tests", "testthat", "test-my_fun_filename1.R")
    expect_false(file.exists(my_filename1fun_file))
    my_filename2fun_file <- file.path(dummypackage, "tests", "testthat", "test-my_fun_filename2.R")
    expect_false(file.exists(my_filename2fun_file))

    tests_lines <- readLines(my_median_file)
    expect_true(any(grepl("my_median", tests_lines)))
    expect_true(any(grepl("my_median2", tests_lines)))
    # Same rdname
    r_lines <- readLines(my_rdname1_file)
    expect_true(any(grepl("my_fun_rdname1", r_lines)))
    expect_true(any(grepl("my_fun_rdname2", r_lines)))
    # Same chunk name
    r_lines <- readLines(my_chunk1_file)
    expect_true(any(grepl("my_fun_chunk1", r_lines)))
    expect_true(any(grepl("my_fun_chunk2", r_lines)))
    # Same @filename
    r_lines <- readLines(my_filename1_file)
    expect_true(any(grepl("my_fun_filename1", r_lines)))
    expect_true(any(grepl("my_fun_filename2", r_lines)))
  })
})
unlink(dummypackage, recursive = TRUE)
