% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sync.cluster.R
\name{sync.cluster}
\alias{sync.cluster}
\title{Time Series Clustering based on Trend Synchronism}
\usage{
sync.cluster(formula, rate = 1, alpha = 0.05, ...)
}
\arguments{
\item{formula}{an object of class "\code{\link[stats]{formula}}", 
specifying the type of common trend 
for clustering the time series in a \eqn{T} by \eqn{N} matrix of time series 
(time series in columns) which is passed to \code{\link{sync.test}}. 
Variable \eqn{t} should be used to specify the form 
of the trend, where \eqn{t} is specified within the function automatically as a 
regular sequence of length \eqn{T} on the interval (0,1]. See `Examples'.}

\item{rate}{rate of removal of time series. Default is 1 (i.e., if hypothesis 
of synchronism is rejected one time series is removed at a time to re-test the 
remaining time series). Integer values above 1 are treated as number of time 
series to be removed. Values from 0 to 1 are treated as a fraction of 
time series to be removed.}

\item{alpha}{significance level for testing hypothesis of a common trend 
(using \code{\link{sync.test}}) of the parametric form specified in \code{formula}.}

\item{...}{arguments to be passed to \code{\link{sync.test}}, for example, 
number of bootstrap replications (\code{B}).}
}
\value{
A list with the elements:
\item{cluster}{an integer vector indicating the cluster to which each time series is 
allocated. A label \code{'0'} is assigned to time series which do not have a common trend 
with other time series (that is, all time series labeled with \code{'0'} are separate 
one-element clusters).}
\item{elements}{a list with names of the time series in each cluster.}

The further elements combine results of \code{\link{sync.test}} for each cluster with
at least two elements (that is, single-element clusters labeled with 
\code{'0'} are excluded):
\item{estimate}{a list with common parametric trend estimates obtained by 
\code{\link{sync.test}} for each cluster. 
The length of this list is \code{max(cluster)}.}
\item{pval}{a list of \eqn{p}-values of \code{\link{sync.test}} for each cluster.
The length of this list is \code{max(cluster)}.}
\item{statistic}{a list with values of \code{\link{sync.test}} test statistic for 
each cluster. The length of this list is \code{max(cluster)}.}
\item{ar_order}{a list of AR filter orders used in \code{\link{sync.test}} for each 
time series. The results are grouped by cluster in the list of length \code{max(cluster)}.}
\item{window_used}{a list of local windows used in \code{\link{sync.test}} for each 
time series. The results are grouped by cluster in the list of length \code{max(cluster)}.}
\item{all_considered_windows}{a list of all windows considered in 
\code{\link{sync.test}} and corresponding test results, for each cluster. 
The length of this list is \code{max(cluster)}.}
\item{WAVK_obs}{a list of WAVK test statistics obtained in \code{\link{sync.test}} 
for each time series. 
The results are grouped by cluster in the list of length \code{max(cluster)}.}
}
\description{
Cluster time series with a common parametric trend using the 
\code{\link{sync.test}} function 
\insertCite{Lyubchich_Gel_2016_synchronism,Ghahari_etal_2017_MBDCE}{funtimes}.
}
\details{
The \code{sync.cluster} function recursively clusters time series having 
a pre-specified common parametric trend until there are no time series left. 
Starting with the given \eqn{N} time series, the \code{\link{sync.test}} function 
is used to test for a common trend. If null hypothesis of common trend is not 
rejected by \code{\link{sync.test}}, the time series are grouped together 
(i.e., assigned to a cluster). Otherwise, the time series with the largest 
contribution to the test statistics are temporarily removed (the number of time 
series to remove depends on the \code{rate} of removal) and \code{\link{sync.test}} 
is applied again. The contribution to the test statistic is assessed by the
WAVK test statistic calculated for each time series.
}
\examples{
\dontrun{
## Simulate 4 autoregressive time series, 
## 3 having a linear trend and 1 without a trend:
set.seed(123)
T = 100 #length of time series
N = 4 #number of time series
X = sapply(1:N, function(x) arima.sim(n = T, 
           list(order = c(1, 0, 0), ar = c(0.6))))
X[,1] <- 5 * (1:T)/T + X[,1]
plot.ts(X)

# Finding clusters with common linear trends:
LinTrend <- sync.cluster(X ~ t) 
  
## Sample Output:
##[1] "Cluster labels:"
##[1] 0 1 1 1
##[1] "Number of single-element clusters (labeled with '0'): 1"

## plotting the time series of the cluster obtained
for(i in 1:max(LinTrend$cluster)) {
    plot.ts(X[, LinTrend$cluster == i], 
            main = paste("Cluster", i))
}


## Simulating 7 autoregressive time series, 
## where first 4 time series have a linear trend added 
set.seed(234)
T = 100 #length of time series
a <- sapply(1:4, function(x) -10 + 0.1 * (1:T) + 
            arima.sim(n = T, list(order = c(1, 0, 0), ar = c(0.6))))
b <- sapply(1:3, function(x) arima.sim(n = T, 
            list(order = c(1, 0, 0), ar = c(0.6))))
Y <- cbind(a, b)
plot.ts(Y)

## Clustering based on linear trend with rate of removal = 2 
# and confidence level for the synchronism test 90\%
LinTrend7 <- sync.cluster(Y ~ t, rate = 2, alpha = 0.1, B = 99)
   
## Sample output:
##[1] "Cluster labels:"
##[1] 1 1 1 0 2 0 2
##[1] "Number of single-element clusters (labeled with '0'): 2"
}

}
\references{
\insertAllCited{}
}
\seealso{
\code{\link{BICC}}, \code{\link{DR}}, \code{\link{sync.test}}
}
\author{
Srishti Vishwakarma, Vyacheslav Lyubchich
}
\keyword{cluster}
\keyword{synchrony}
\keyword{trend}
