% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/promaxQ.R
\name{promaxQ}
\alias{promaxQ}
\title{Conduct an Oblique Promax Rotation}
\usage{
promaxQ(
  R = NULL,
  urLoadings = NULL,
  facMethod = "fals",
  numFactors = NULL,
  power = 4,
  standardize = "Kaiser",
  epsilon = 1e-04,
  maxItr = 15000,
  faControl = NULL
)
}
\arguments{
\item{R}{(Matrix) A correlation matrix.}

\item{urLoadings}{(Matrix) An unrotated factor-structure matrix to be rotated.}

\item{facMethod}{(Character) The method used for factor extraction 
(\code{\link{faX}}). The supported options are "fals" for unweighted least 
squares, "faml" for maximum likelihood, "fapa" for iterated principal axis 
factoring, "faregLS" for regularized least squares,
"faregML" for regularized maximum likelihood, and "pca" for principal components 
 analysis. The default method  is "fals". 
\itemize{
  \item \strong{"fals"}: Factors are extracted using the unweighted least 
  squares estimation procedure using the \code{\link{fals}} function.
  \item \strong{"faml"}: Factors are extracted using the maximum likelihood 
  estimation procedure using the \code{\link[stats]{factanal}} function.
  \item \strong{"fapa"}: Factors are extracted using the iterated principal 
  axis factoring estimation procedure using the \code{\link{fapa}} function.
  \item \strong{"faregLS"}: Factors are extracted using regularized 
  least squares factor analysis using the \code{\link{fareg}} function. 
  \item \strong{"faregML"}: Factors are extracted using regularized 
  maximum likelihood factor using the \code{\link{fareg}} function. 
  \item \strong{"pca"}: Principal components are extracted. 
}}

\item{numFactors}{(Scalar) The number of factors to extract if the lambda 
matrix is not provided.}

\item{power}{(Scalar) The power with which to raise factor loadings for 
minimizing trivial loadings. The default value is 4.}

\item{standardize}{(Character) Which standardization routine is applied to the 
unrotated factor structure. The three options are "none", "Kaiser", and "CM". 
The default option is "Kaiser" as is recommended by Kaiser and others. See 
\code{\link{faStandardize}} for more details. 
\itemize{
  \item \strong{"none"}: Do \emph{not} rotate the normalized factor structure 
  matrix.
  \item \strong{"Kaiser"}: Use a factor structure matrix that has been normed 
  by Kaiser's method (i.e., normalize all rows to have a unit length).
  \item \strong{"CM"}: Use a factor structure matrix that has been normed by 
  the Cureton-Mulaik method.
}}

\item{epsilon}{(Scalar) The convergence criterion used for evaluating the 
varimax rotation. The default value is 1e-4 (i.e., .0001).}

\item{maxItr}{(Scalar) The maximum number of iterations allowed for computing 
the varimax rotation. The default value is 15,000 iterations.}

\item{faControl}{(List) A list of optional parameters passed to the factor 
extraction (\code{\link{faX}}) function.
\itemize{
  \item \strong{treatHeywood}: (Logical) In \code{fals}, if treatHeywood is 
  true, a penalized least squares function is used to bound the communality 
  estimates below 1.0. Defaults to treatHeywood = TRUE.
  \item \strong{nStart}: (Numeric) The number of starting values to be tried 
  in \code{faml}. Defaults to nStart = 10.
  \item \strong{start}: (Matrix) NULL or a matrix of starting values, each column 
  giving an initial set of uniquenesses. Defaults to start = NULL. 
  \item \strong{maxCommunality}: (Numeric) In \code{faml}, set the maximum 
  communality value for the estimated solution. Defaults to maxCommunality = .995.
  \item \strong{epsilon}: (Numeric) In \code{fapa}, the numeric threshold 
  designating when the algorithm has converged. Defaults to epsilon = 1e-4.
  \item \strong{communality}: (Character) The method used to estimate the 
  initial communality values in \code{fapa}. Defaults to communality = 'SMC'.
  \itemize{
    \item \strong{"SMC"}: Initial communalities are estimated by taking the 
    squared multiple correlations of each indicator after regressing the 
    indicator on the remaining variables.
    \item \strong{"maxr"}: Initial communalities equal the largest 
    (absolute value) correlation in each column of the correlation matrix.
    \item \strong{"unity"}: Initial communalities equal 1.0 for all variables.
  }
  \item \strong{maxItr}: (Numeric) In \code{fapa}, the maximum number of 
  iterations to reach convergence. Defaults to maxItr = 15,000.
}}
}
\value{
A list of the following elements are produced:
\itemize{
  \item \strong{loadings}: (Matrix) The oblique, promax-rotated, 
  factor-pattern matrix.
  \item \strong{vmaxLoadings}: (Matrix) The orthogonal, varimax-rotated, 
  factor-structure matrix used as the input matrix for the promax rotation.
  \item \strong{rotMatrix}: (Matrix) The (rescaled) transformation matrix 
  used in an attempt to minimize the Euclidean distance between the varimax 
  loadings and the generated promax target matrix (cf. Hendrickson & White, 
  1964; Mulaik, 2009, p. 342-343, eqn. 12.44).
  \item \strong{Phi}: (Matrix) The factor correlation matrix associated with 
  the promax solution. Phi is found by taking the inverse of the inner 
  product of the (rescaled) rotation matrix (rotMatrix) with itself (i.e., 
  \eqn{solve(T' T)}, where T is the (rescaled) rotation matrix).
  \item \strong{vmaxDiscrepancy}: (Scalar) The value of the minimized varimax 
  discrepancy function. promax does not have a rotational criterion but the 
  varimax rotation does.
  \item \strong{convergence}: (Logical) Whether the varimax rotation
  congerged.
  \item \strong{Table}: (Matrix) The table returned from \code{\link{GPForth}} 
  from the \code{GPArotation} package.
  \item \strong{rotateControl}: (List) A list containing (a) the power 
  parameter used, (b) whether the varimax rotation used Kaiser normalization, 
  (c) the varimax epsilon convergence criterion, and (d) the maximum number 
  of iterations specified.
  \itemize{
    \item \strong{power}: The power in which the varimax-rotated factor 
    loadings are raised.
    \item \strong{standardize}: Which standardization routine was used.
    \item \strong{epsilon}: The convergence criterion set for the varimax rotation.
    \item \strong{maxItr}: The maximum number of iterations allowed for 
    reaching convergence in the varimax rotation.
  }
}
}
\description{
This function is an extension of the \code{\link[stats]{promax}} function. 
This function will extract the unrotated factor loadings (with three algorithm 
options, see \code{\link{faX}}) if they are not provided. The factor 
intercorrelations (Phi) are also computed within this function.
}
\details{
\itemize{
  \item \strong{Varimax Standardization}: When conducting the varimax 
  rotation, it is recommended to standardize the factor loadings using 
  Kaiser's normalization (i.e., rescaling the factor indicators [rows] so 
  that the vectors have unit length). The standardization/normalization 
  occurs by pre-multiplying the unrotated factor structure, \strong{A}, by 
  the inverse of \strong{H}, where \strong{H}^2 is a diagonal matrix with the 
  communality estimates on the diagonal. A varimax rotation is then applied 
  to the normalized, unrotated factor structure. Then, the varimax-rotated 
  factor structure is rescaled to its original metric by pre-multiplying the 
  varimax factor structure by \strong{H}. For details, see Mulaik (2009).
  \item \strong{Oblique Procrustes Rotation of the Varimax Solution}: 
  According to Hendrickson & White (1964), an unrestricted (i.e., oblique) 
  Procrustes rotation is applied to the orthogonal varimax solution. 
  Specifically, a target matrix is generated by raising the varimax factor 
  loadings to the user-specified power (typically, power = 4) (must retain 
  the signs of the original factor loadings). This should quickly diminish 
  trivial factor loadings while retaining larger factor loadings. The 
  Procrustes rotation takes the varimax solution and rotates it toward the 
  promax-generated target matrix. For a modern description of this approach, 
  see Mulaik (2009, ch. 12, p. 342-343).
  \item \strong{Choice of a Power}: Changing the power in which varimax factor 
  loadings are raised will change the target matrix in the oblique Procrustes 
  rotation. After raising factor loadings to some power, there will be a 
  larger discrepancy between high and low loadings than before (e.g., squaring 
  factor loadings of .6 and .7 yields loadings of .36 and .49 and cubing 
  yields loadings of .216 and .343). Furthermore, increasing the power will 
  increase the number of near-zero loadings, resulting in larger factor 
  intercorrelations. Many (cf. Gorsuch, 1983; Hendrickson & White, 1964; 
  Mulaik, 2009) advocate for raising varimax loadings to the fourth power 
  (the default) but some (e.g., Gorsuch) advocate for trying power = 2 and 
  power = 6 to see if there is an improvement in the simple structure without 
  overly inflating factor correlations.
}
}
\examples{
## Generate an orthgonal factor model
lambda <- matrix(c(.41, .00, .00,
                   .45, .00, .00,
                   .53, .00, .00,
                   .00, .66, .00,
                   .00, .38, .00,
                   .00, .66, .00,
                   .00, .00, .68,
                   .00, .00, .56,
                   .00, .00, .55),
                 nrow = 9, ncol = 3, byrow = TRUE)

## Model-implied correlation (covariance) matrix
R <- lambda \%*\% t(lambda)

## Unit diagonal elements
diag(R) <- 1

## Start from just a correlation matrix
Out1 <- promaxQ(R           = R,
                facMethod   = "fals",
                numFactors  = 3,
                power       = 4,
                standardize = "Kaiser")$loadings

## Iterate the promaxQ rotation using the rotate function
Out2 <- faMain(R             = R,
               facMethod     = "fals",
               numFactors    = 3,
               rotate        = "promaxQ",
               rotateControl = list(power       = 4,
                                    standardize = "Kaiser"))$loadings

## Align the factors to have the same orientation
Out1 <- faAlign(F1 = Out2,
                F2 = Out1)$F2

## Show the equivalence of factor solutions from promaxQ and rotate
all.equal(Out1, Out2, check.attributes = FALSE)

}
\references{
Gorsuch, R. L. (1983). \emph{Factor Analysis}, 2nd. Hillsdale, 
NJ: LEA.

Hendrickson, A. E., & White, P. O. (1964). Promax: A quick 
method for rotation to oblique simple structure. \emph{British Journal of 
Statistical Psychology, 17}(1), 65-70.

Mulaik, S. A. (2009). \emph{Foundations of Factor Analysis}. 
Chapman and Hall/CRC.
}
\seealso{
Other Factor Analysis Routines: 
\code{\link{BiFAD}()},
\code{\link{Box26}},
\code{\link{GenerateBoxData}()},
\code{\link{Ledermann}()},
\code{\link{SLi}()},
\code{\link{SchmidLeiman}()},
\code{\link{faAlign}()},
\code{\link{faEKC}()},
\code{\link{faIB}()},
\code{\link{faMB}()},
\code{\link{faMain}()},
\code{\link{faScores}()},
\code{\link{faSort}()},
\code{\link{faStandardize}()},
\code{\link{faX}()},
\code{\link{fals}()},
\code{\link{fapa}()},
\code{\link{fareg}()},
\code{\link{orderFactors}()},
\code{\link{print.faMB}()},
\code{\link{print.faMain}()},
\code{\link{summary.faMB}()},
\code{\link{summary.faMain}()}
}
\author{
\itemize{
  \item Casey Giordano (Giord023@umn.edu)
  \item Niels G. Waller (nwaller@umn.edu)
}
}
\concept{Factor Analysis Routines}
