# Generated by using Rcpp::compileAttributes() -> do not edit by hand
# Generator token: 10BE3573-1514-4C36-9D1C-5A225CD40393

#' Detrended Cross-Correlation Analysis
#'
#' Fast function for computing detrended cross-correlation analysis (DCCA) on long time series, which is a 
#' bivariate extension of detrended fluctuation analysis (DFA).
#'
#' @param x A real valued vector (i.e., time series data) to be analyzed.
#' @param y A real valued vector (i.e., time series data) to be analyzed.
#' @param order is an integer indicating the polynomial order used for 
#' detrending the local windows (e.g, 1 = linear, 2 = quadratic, etc.). There 
#' is not a pre-determined limit on the order of the polynomial order but the 
#' user should avoid using a large polynomial on small windows. This can result
#' in overfitting and non-meaningful estimates. 
#' @param scales An integer vector of scales over which to compute correlation. 
#' Unlike univariate DFA, MRA does not require that scales be in log units.
#' Scale intervals can be sequential, for example, when the analysis is 
#' exploratory and no a priori hypotheses have been made about the scale of 
#' correlation. A small subset of targeted scales may also be investigated 
#' where scale-specific research questions exist. We have found that windows
#' smaller than say 8 observations create stability problems due to 
#' overfitting. This is espcially when the order of the fitting polynomial is 
#' large.
#' @import Rcpp
#' @useDynLib fractalRegression
#' @export
#'
#' @details Details of the algorithm are specified in Podobnik and Stanley (2008) and in Zebende (2011).
#' In general, the output of the algorithm are estimates of \eqn{\rho}DCCA, which range from -1 to 1 and can generally be interpreted as follows: 
#' \itemize{
#'  \item \eqn{\rho DCCA = -1.0 ->} perfect anti-cross-correlation
#'  \item \eqn{\rho DCCA =  0.0 ->} no cross-correlation
#'  \item \eqn{\rho DCCA =  1.0 ->} perfect cross-correlation
#' } 
#'
#' @return The object returned from the function is a list including the following:
#' \itemize{ 
#'  \item \code{scales} indicates the values of the scales used for estimates \eqn{\rho}DCCA
#'  \item \code{rho} includes the scale-wise estimates of \eqn{\rho}DCCA
#' }
#'
#' @references
#' 
#' Podobnik, B., & Stanley, H. E. (2008). Detrended cross-correlation analysis: a new method for analyzing two nonstationary time series. Physical review letters, 100(8), 084102.
#'
#' Zebende, G. F. (2011). DCCA cross-correlation coefficient: Quantifying level of cross-correlation. Physica A: Statistical Mechanics and its Applications, 390(4), 614-618.
#'
#'//'
#' @examples
#' 
#'
#' 
#' # Here is a simple example for running DCCA using a white noise and pink noise time series.
#' # For more detailed examples, see the vignette. 
#' 
#' noise <- rnorm(5000)
#' 
#' pink.noise <- fgn_sim(n = 5000, H = 0.9)
#'
#' scales <- ifultools::logScale(scale.min = 10, scale.max = 1250, scale.ratio = 1.1)
#' 
#' dcca.out <- dcca(noise, pink.noise, order = 1, scales = scales)
#' 
#' 
#'
dcca <- function(x, y, order, scales) {
    .Call('_fractalRegression_dcca', PACKAGE = 'fractalRegression', x, y, order, scales)
}

#' Detrended Fluctuation Anlaysis
#' 
#' Fast function for computing detrended fluctuation analysis (DFA), a widely used method for estimating long-range temporal correlations in time series data. 
#' DFA is also a form of mono-fractal analysis that indicates the degree of self-similarity across temporal scales.
#' 
#' @param x A real valued vector (i.e., time series data) to be analyzed. 
#' @param order An integer indicating the polynomial order used for 
#' detrending the local windows (e.g, 1 = linear, 2 = quadratic, etc.). There 
#' is not a pre-determined limit on the order of the polynomial order but the 
#' user should avoid using a large polynomial on small windows. This can result
#' in overfitting and non-meaningful estimates. 
#' @param verbose A logical that when = 1 indicates that the flucuation function including the log of all included scales as well as the log Rms should be 
#' returned as well as the \eqn{\alpha} or when = 0 only the estimated scaling exponent \eqn{\alpha} will be returned.
#' @param sc_min The minimum window size, specified in the number of data points (i.e., observations) to be included in the smallest window. 
#' @param sc_max The maximum window size, specified in the number of data points (i.e., observations) to be included in the largest window.
#' @param scale_ratio A scaling factor by which to create successive window sizes from `sc_min` to `sc_max`. 
#' This allows one to to maintain even spacing in logarithms while increasing
#' scale resolution.
#' @import Rcpp
#' @useDynLib fractalRegression
#' @export
#' 
#' @details Details of the algorithm are specified in detail in Peng et al. (1994) and visualized nicely in Kelty-Stephen et al. (2016).
#' The output of the algorithm is an \eqn{\alpha} (alpha) estimate which is a generalization of the Hurst Exponent. Conventional interpretation of \eqn{\alpha} is:
#' \itemize{
#'  \item \eqn{\alpha < 0.5 =} anti-correlated
#'  \item \eqn{\alpha ~= 0.5 =} uncorrelated, white noise
#'  \item \eqn{\alpha > 0.5 =} temporally correlated
#'  \item \eqn{\alpha ~= 1 =} 1/f-noise, pink noise
#'  \item \eqn{\alpha > 1 =} non-stationary and unbounded
#'  \item \eqn{\alpha ~= 1.5 =} fractional brownian motion
#' } 
#' 
#' We recommend a few points of consideration here in using this function. One is to be sure to 
#' verify there are not cross-over points in the logScale-logFluctuation plots (Peng et al., 1995; Perakakis et al ., 2009). Cross-over points 
#' (or a visible change in the slope as a function of of scale) indicate that a mono-fractal characterization 
#' does not sufficiently characterize the data. If cross-over points are evident, we recommend proceeding to using the mfdfa() to estimate the multi-fractal
#' fluctuation dynamics across scales.
#' 
#' While it is common to use only linear detrending with DFA, it is important to inspect the trends in the data to determine
#' if it would be more appropriate to use a higher order polynomial for detrending, and/or compare the DFA output for different polynomial orders (see Kantelhardt et al., 2001).
#' 
#' General recommendations for choosing the min and max scale are an sc_min = 10 and sc_max = (N/4), where N is the number of observations.
#' See Eke et al. (2002) and Gulich and Zunino (2014) for additional considerations. 
#' 
#' @return The object returned can take the following forms:
#' \itemize{ 
#'  \item If the value of verbose = 1, then a list object is returned that includes: \code{logScales}
#' the log of all included scales, \code{logRMS} the log root mean square error (RMS) per scale, and \code{alpha} the overall \eqn{\alpha} estimate.
#'  \item If the value of verbose = 0, then a list containing only `alpha` the estimated scaling exponent \eqn{\alpha} will be returned.
#' }
#' @references 
#' 
#' Eke, A., Herman, P., Kocsis, L., & Kozak, L. R. (2002). Fractal characterization of complexity in temporal physiological signals. Physiological measurement, 23(1), R1-R38.
#' 
#' Gulich, D., & Zunino, L. (2014). A criterion for the determination of optimal scaling ranges in DFA and MF-DFA. Physica A: Statistical Mechanics and its Applications, 397, 17-30.
#'
#' Kantelhardt, J. W., Koscielny-Bunde, E., Rego, H. H., Havlin, S., & Bunde, A. (2001). Detecting long-range correlations with detrended fluctuation analysis. Physica A: Statistical Mechanics and its Applications, 295(3-4), 441-454.
#' 
#' Kelty-Stephen, D. G., Stirling, L. A., & Lipsitz, L. A. (2016). Multifractal temporal correlations in circle-tracing behaviors are associated with the executive function of rule-switching assessed by the Trail Making Test. Psychological assessment, 28(2), 171-180.
#' 
#' Peng C-K, Buldyrev SV, Havlin S, Simons M, Stanley HE, and Goldberger AL (1994), Mosaic organization of DNA nucleotides, Physical Review E, 49, 1685-1689.
#' 
#' Peng C-K, Havlin S, Stanley HE, and Goldberger AL (1995), Quantification of scaling exponents and crossover phenomena in nonstationary heartbeat time series, Chaos, 5, 82-87.
#' 
#' Perakakis, P., Taylor, M., Martinez-Nieto, E., Revithi, I., & Vila, J. (2009). Breathing frequency bias in fractal analysis of heart rate variability. Biological psychology, 82(1), 82-88.
#' 
#' @examples
#' 
#' 
#' 
#' noise <- rnorm(5000)
#' 
#' dfa.noise.out <- dfa(
#'     x = noise, 
#'     order = 1, 
#'     verbose = 1, 
#'     sc_min = 16, 
#'     sc_max = length(noise)/4, 
#'     scale_ratio = 2)
#' 
#' pink.noise <- fgn_sim(n = 5000, H = 0.9)
#' 
#' dfa.pink.out <- dfa(
#'     x = pink.noise, 
#'     order = 1, 
#'     verbose = 1, 
#'     sc_min = 16, 
#'     sc_max = length(pink.noise)/4, 
#'     scale_ratio = 2)
#' 
#' anticorr.noise <- fgn_sim(n = 5000, H = 0.25)
#' 
#' dfa.anticorr.out <- dfa(
#'     x = anticorr.noise, 
#'     order = 1, 
#'     verbose = 1, 
#'     sc_min = 16, 
#'     sc_max = length(anticorr.noise)/4, 
#'     scale_ratio = 2)
#' 
#' 
#' 
#' 
dfa <- function(x, order, verbose, sc_min, sc_max, scale_ratio) {
    .Call('_fractalRegression_dfa', PACKAGE = 'fractalRegression', x, order, verbose, sc_min, sc_max, scale_ratio)
}

#' Multiscale Lagged Regression Anlaysis
#' Fast function for computing MRA on long time series
#' @param x is a real valued vector of time series data
#' @param y is a real valued vector of time series data
#' @param order is an integer indicating the polynomial order used for 
#' detrending the local windows
#' @param scales integer vector of scales over which to compute correlation. 
#' Performance is best when scales are evenly spaced in log units. Choosing a
#' logarithm base between 1 and 2 may also improve performance of regression.
#' @param lags integer indicating the maximum number of lags to include
#' @param direction string indicating a positive ('p') or negative ('n') lag
#' @import Rcpp
#' @useDynLib fractalRegression
#' @export
#' 
#' @return The object returned from the dlcca() function is a list containing rho coefficients for each lag at each of the scales. 
#'
dlcca <- function(x, y, order, scales, lags, direction) {
    .Call('_fractalRegression_dlcca', PACKAGE = 'fractalRegression', x, y, order, scales, lags, direction)
}

#' Multifractal Detrended Fluctuation Analysis
#'
#' Fast function for computing multifractal detrended fluctuation analysis (MF-DFA), a widely used method for estimating the family of long-range temporal correlations or scaling exponents in time series data. 
#' MF-DFA is also a form of multifractal analysis that indicates the degree of interaction across temporal scales.
#' 
#' @param x A real valued vector (i.e., time series data) to be analyzed. 
#' @param q A real valued vector indicating the statistical moments (q) to use 
#' in the analysis.
#' @param order is an integer indicating the polynomial order used for 
#' detrending the local windows (e.g, 1 = linear, 2 = quadratic, etc.). There 
#' is not pre-determined limit on the order of the polynomial order but the 
#' user should avoid using a large polynomial on small windows. This can result
#' in overfitting and non-meaningful estimates. 
#' @param scale_min An integer indicating the minimum window size, specified in the number of data points (i.e., observations) to be included in the smallest window. 
#' @param scale_max An integer indicating the maximum window size, specified in the number of data points (i.e., observations) to be included in the largest window. indicating largest scale to resolve
#' @param scale_ratio A scaling factor by which to create successive window sizes from `scale_min` to `scale_max`. 
#' This allows one to to maintain even spacing in logarithms while increasing
#' scale resolution.
#' @import Rcpp
#' @useDynLib fractalRegression
#' @export
#' 
#' @details Details of the algorithm are specified in detail in Kantelhardt et al. (2001; 2002) and visualized nicely in Kelty-Stephen et al. (2016).
#' 
#' Selecting the range of values for q is important. Note that MF-DFA estimates for q = 2 are equivalent to DFA. Larger values of q (q > 2) emphasize larger residuals and smaller values of q
#' (q < 2) emphasis smaller residuals (Kelty-Stephen et al., 2016). For most biomedical signals such as physiological and kinematic, a q range of -5 to 5 is common (Ihlen, 2010). However, in some cases, 
#' such as when time series are short (< 3000), it can be appropriate to limit the range of q to positive only. Kelty-Stephen et al. (2016) recommend a 
#' positive q range of 0.5 to 10 with an increment of 0.5. 
#'
#' While it is common to use only linear detrending with DFA and MF-DFA, it is important to inspect the trends in the data to determine
#' if it would be more appropriate to use a higher order polynomial for detrending, and/or compare the DFA and MF-DFA output for different polynomial orders (see Ihlen, 2012; Kantelhardt et al., 2001).
#' 
#' General recommendations for choosing the min and max scale are a scale_min = 10 and scale_max = (N/4), where N is the number of observations.
#' See Eke et al. (2002), Gulich and Zunino (2014), Ihlen (2012), and  for additional considerations and information on choosing the correct parameters. 
#'
#' @return The output of the algorithm is a list that includes:
#' \itemize{ 
#'  \item \code{log2scale} The log2 scales used for the analysis
#'  \item \code{log2Fq} The log2 of the fluctuation functions for each scale and q 
#'  \item \code{Hq} The q-order Hurst exponent (generalized Hurst exponent)
#'  \item \code{Tau} The q-order mass exponent
#'  \item \code{q} The q-order statistical moments
#'  \item \code{h} The q-order singularity exponent
#'  \item \code{Dh} The dimension of the q-order singularity exponent
#'}
#'
#' @references 
#'
#' Ihlen, E. A. F. (2012). Introduction to Multifractal Detrended Fluctuation Analysis in Matlab. Frontiers in Physiology, 3. https://doi.org/10.3389/fphys.2012.00141
#'
#' Kantelhardt, J. W., Koscielny-Bunde, E., Rego, H. H., Havlin, S., & Bunde, A. (2001). Detecting long-range correlations with detrended fluctuation analysis. Physica A: Statistical Mechanics and its Applications, 295(3-4), 441-454.
#' 
#' Kantelhardt, J. W., Zschiegner, S. A., Koscielny-Bunde, E., Havlin, S., Bunde, A., & Stanley, H. E. (2002). Multifractal detrended fluctuation analysis of nonstationary time series. Physica A: Statistical Mechanics and its Applications, 316(1-4), 87-114.
#'
#' Kelty-Stephen, D. G., Palatinus, K., Saltzman, E., & Dixon, J. A. (2013). A Tutorial on Multifractality, Cascades, and Interactivity for Empirical Time Series in Ecological Science. Ecological Psychology, 25(1), 1-62. https://doi.org/10.1080/10407413.2013.753804
#'
#' Kelty-Stephen, D. G., Stirling, L. A., & Lipsitz, L. A. (2016). Multifractal temporal correlations in circle-tracing behaviors are associated with the executive function of rule-switching assessed by the Trail Making Test. Psychological Assessment, 28(2), 171-180. https://doi.org/10.1037/pas0000177
#'
#' @examples
#'
#' 
#' 
#' noise <- rnorm(5000)
#' 
#' mf.dfa.white.out <- mfdfa(
#'     x = noise, q = c(-5:5), 
#'     order = 1, 
#'     scale_min = 16, 
#'     scale_max = length(noise)/4, 
#'     scale_ratio = 2) 
#'  
#' pink.noise <- fgn_sim(n = 5000, H = 0.9)
#' 
#' mf.dfa.pink.out <- mfdfa(
#'     x = pink.noise, 
#'     q = c(-5:5), 
#'     order = 1, 
#'     scale_min = 16, 
#'     scale_max = length(pink.noise)/4, 
#'     scale_ratio = 2)
#'
#' 
#' 
mfdfa <- function(x, q, order, scale_min, scale_max, scale_ratio) {
    .Call('_fractalRegression_mfdfa', PACKAGE = 'fractalRegression', x, q, order, scale_min, scale_max, scale_ratio)
}

colmeans <- function(X) {
    .Call('_fractalRegression_colmeans', PACKAGE = 'fractalRegression', X)
}

#' Multiscale Lagged Regression Analysis
#'
#' Fast function for computing multiscale lagged regression analysis (MLRA) on long time series. Combining DFA with ordinary least square regression, MLRA
#' is a form of fractal regression that can be used to estimate asymmetric and multiscale regression coefficients between two variables at different time-scales and temporal lags. 
#'
#' @param x A real valued vector (i.e., time series data) to be analyzed.
#' @param y A real valued vector (i.e., time series data) to be analyzed.
#' @param order is an integer indicating the polynomial order used for 
#' detrending the local windows (e.g, 1 = linear, 2 = quadratic, etc.). There 
#' is a not pre-determined limit on the order of the polynomial order but the 
#' user should avoid using a large polynomial on small windows. This can result
#' in overfitting and non-meaningful estimates. 
#' @param scales An integer vector of scales over which to compute correlation. 
#' Unlike univariate DFA, MRA does not require that scales be in log units.
#' Scale intervals can be sequential, for example, when the analysis is 
#' exploratory and no a priori hypotheses have been made about the scale of 
#' correlation. A small subset of targeted scales may also be investigated 
#' where scale-specific research questions exist. We have found that windows
#' smaller than say 8 observations create stability problems due to 
#' overfitting. This is espcially when the order of the fitting polynomial is 
#' large.
#' @param lags An integer indicating the maximum number of lags to include in the analysis.
#' @param direction A character string indicating a positive ('p') or negative ('n') lag.
#' @import Rcpp
#' @useDynLib fractalRegression
#' @export
#'
#' @details Mathematical treatment of the MLRA algorithm and its performance can be found in Kristoufek (2015) and Likens et al. (2019).
#'
#' Use of the direction parameter specifies whether the scale-wise \eqn{\beta} coefficients for positive or negative lags will be estimated.  
#'
#' Note that under conditions with linear and quadratic trends, Likens et al. (2019) found that there was a systematic positive bias in the \eqn{\beta} estimates for larger scales.
#' Using a polynomial detrending order of 2 or greater was shown to attenuate this bias. 
#'
#' @return The object returned from the mlra() function is a list containing \code{betas} the \eqn{\beta} coefficients for each lag at each of the scales. 
#'
#'
#' @references
#'
#' Kristoufek, L. (2015). Detrended fluctuation analysis as a regression framework: Estimating dependence at different scales. Physical Review E, 91(2), 022802.
#'
#' Likens, A. D., Amazeen, P. G., West, S. G., & Gibbons, C. T. (2019). Statistical properties of Multiscale Regression Analysis: Simulation and application to human postural control. Physica A: Statistical Mechanics and its Applications, 532, 121580.
#'
#' @examples
#'
#' 
#' # Here is a simple example for running MLRA using a white noise and pink noise time series.
#' # For more detailed examples, see the vignette. 
#' 
#' noise <- rnorm(5000)
#' 
#' pink.noise <- fgn_sim(n = 5000, H = 0.9)
#'
#' scales <- ifultools::logScale(scale.min = 10, scale.max = 1250, scale.ratio = 1.1)
#' 
#' mlra.out <- mlra(
#'     x = noise, 
#'     y = pink.noise, 
#'     order = 1, 
#'     scales = scales, 
#'     lags = 100, direction = 'p')
#' 
#'
#'
mlra <- function(x, y, order, scales, lags, direction) {
    .Call('_fractalRegression_mlra', PACKAGE = 'fractalRegression', x, y, order, scales, lags, direction)
}

lagn <- function(x, lags) {
    .Call('_fractalRegression_lagn', PACKAGE = 'fractalRegression', x, lags)
}

polyfit <- function(x, order) {
    .Call('_fractalRegression_polyfit', PACKAGE = 'fractalRegression', x, order)
}

detrend_var <- function(X, order) {
    .Call('_fractalRegression_detrend_var', PACKAGE = 'fractalRegression', X, order)
}

#' Multiscale Regression Anlaysis (MRA)
#'
#' Fast function for computing multiscale regression analysis (MRA) on long time series. Combining DFA with ordinary least square regression, MRA
#' is a form of fractal regression that can be used to estimate asymmetric and multiscale regression coefficients between two variables. 
#'
#' @param x A real valued vector (i.e., time series data) to be analyzed. A key
#' difference between DCCA and MRA is that MRA produces asymmetric estiamtes. 
#' That is, x is assumed to be an independent variable and y is assumed to be 
#' a dependent variable. MRA should be used when one of the time series in 
#' question is usefully cast as the independent variable. That is, x is assumed
#' to effect change in y. If no such causal relationship is anticipated, use
#' DCCA instead.
#' @param y A real valued vector (i.e., time series data) to be analyzed.
#' @param order is an integer indicating the polynomial order used for 
#' detrending the local windows (e.g, 1 = linear, 2 = quadratic, etc.). There 
#' is not a pre-determined limit on the order of the polynomial order but the 
#' user should avoid using a large polynomial on small windows. This can result
#' in overfitting and non-meaningful estimates. 
#' @param scales An integer vector of scales over which to compute correlation. 
#' Unlike univariate DFA, MRA does not require that scales be in log units.
#' Scale intervals can be sequential, for example, when the analysis is 
#' exploratory and no a priori hypotheses have been made about the scale of 
#' correlation. A small subset of targeted scales may also be investigated 
#' where scale-specific research questions exist. We have found that windows
#' smaller than say 8 observations create stability problems due to 
#' overfitting. This is espcially when the order of the fitting polynomial is 
#' large.
#' @import Rcpp
#' @useDynLib fractalRegression
#' @export
#'
#' @details Mathematical treatment of the MRA algorithm and its performance can be found in Kristoufek (2015) and Likens et al. (2019).
#'
#' Note that under conditions with linear and quadratic trends, Likens et al. (2019) found that there was a systematic positive bias in the \eqn{\beta} estimates for larger scales.
#' Using a polynomial detrending order of 2 or greater was shown to attenuate this bias. 
#'
#' @return The object returned from the mra() function is a list including the following:
#' \itemize{ 
#'  \item \code{scales} indicates the values of the scales used for estimates
#'  \item \code{betas} are the scale specific \eqn{\beta} estimates of the influence of x on y
#'  \item \code{r2} is the scale specific r-squared value of the model fit (i.e., variance in y accounted for by x at that scale)
#'  \item \code{t_observed} is the estimated t-statistic for a given \eqn{\beta} at a given scale. 
#' }
#'
#'
#' @references
#'
#' Kristoufek, L. (2015). Detrended fluctuation analysis as a regression framework: Estimating dependence at different scales. Physical Review E, 91(2), 022802.
#'
#' Likens, A. D., Amazeen, P. G., West, S. G., & Gibbons, C. T. (2019). Statistical properties of Multiscale Regression Analysis: Simulation and application to human postural control. Physica A: Statistical Mechanics and its Applications, 532, 121580.
#'
#' @examples
#'
#' 
#' # Here is a simple example for running MRA using a white noise and pink noise time series.
#' # For more detailed examples, see the vignette. 
#' 
#' noise <- rnorm(5000)
#' 
#' pink.noise <- fgn_sim(n = 5000, H = 0.9)
#'
#' scales <- ifultools::logScale(scale.min = 10, scale.max = 1250, scale.ratio = 1.1)
#' 
#' mra.out <- mra(x = noise, y = pink.noise, order = 1, scales = scales)
#' 
#'
#'
mra <- function(x, y, order, scales) {
    .Call('_fractalRegression_mra', PACKAGE = 'fractalRegression', x, y, order, scales)
}

