\encoding{latin1}
\name{Approx.fpt.density}
\alias{Approx.fpt.density}
\title{Approximating First-Passage-Time Densities}
\description{
  \code{Approx.fpt.density} computes values of the approximate first-passage-time (f.p.t.) density, for conditioned and unconditioned problems. \cr
  
  For the unconditioned case, values of the approximate densities for f.p.t. problems conditioned to suitable values of the initial distribution 
  are also calculated. 
}
\usage{
Approx.fpt.density(dp, t0, T, id, S, env = NULL, variableStep = TRUE, 
                   from.t0 = FALSE, to.T = FALSE, r = 4000, zeroSlope = 0.01, 
                   p0.tol = 8, k = 3, m = 100, n = 250, p = 0.2, alpha = 1, 
                   skip = TRUE, tol = 0.001, it.max)               
}
\arguments{
  \item{dp}{an object of class \dQuote{diffproc} defining a family of diffusion processes.}
  \item{t0, T}{lower and upper limits of the considered time interval. Must be finite.}
  \item{id}{a numerical value specifying a fixed initial value of the process in the time instant specified in the \code{t0} argument
  (for a conditioned f.p.t. problem), or a four-component list specifying the initial distribution of the process (for an unconditioned f.p.t. problem).
  In the last case the first component is the \R mathematical expression of the density as a character string, the second
  one is a label to denote the distribution in the reports generated by the \code{report} method, the third one is the label to
  denote the distribution for LaTeX, and the four one is a label to denote the distribution in the graphs generated by the \code{plot} method.}
  \item{S}{numerical value of a constant boundary or character string with the mathematical expression of a time 
  dependent boundary.}
  \item{env}{a named list of objects of numeric or character type specifying the values of names which occur in
  the mathematical expressions in objects \code{dp} and \code{S}, or of names which occur in the specification of the previous values.
  Defaults to NULL, interpreted as an empty list. It is copied into a temporary environment for evaluating the mathematical expressions 
  in the \code{dp} object and \code{S}.} 
  \item{variableStep}{a logical value indicating whether a variable integration step is used.}
  \item{from.t0}{a logical value indicating whether the approximation should be calculated from the lower end of the
interval considered, \eqn{t_0}{t0}.}
  \item{to.T}{a logical value indicating whether the approximation should be calculated to the upper end of the
interval considered, \eqn{T}.}
  \item{r}{number of points at which the FPTL function is evaluated.}
  \item{zeroSlope}{maximum slope required to consider that a growing function is constant.}
  \item{p0.tol}{controls where the First-Passage-Time Location function begins to increase significantly.}
  \item{k}{controls whether the First-Passage-Time Location function decreases very slowly.}
  \item{m}{Number of equally spaced values to select from the specified initial distribution in the case of an unconditioned f.p.t. problem.} 
  \item{n}{Number of points used to determine the integration step in subintervals \eqn{[t_i^*, t_{max,i}^+]}{[t[i]*, tmax[i]^+]} 
\eqn{i=1, \ldots, m}{i=1,..., m}, from interesting instants provided by the FPTL function.}
	\item{p}{Ratio of n used to determine the integration step in subintervals  
\eqn{[t_{max,i}^+, t_{i+1}^*]}{[tmax[i]^+, t[i+1]*]}, \eqn{i=1, \ldots, m}{i=1,..., m}, \eqn{[t_0, t_1^*]}{[t0, t[1]*]} and 
\eqn{[t_{max,m}^{+}, T]}{[tmax[m]^+, T]}.}
  \item{alpha}{Parameter used to determine the integration step in subintervals 
\eqn{[t_{max,i}^+, t_{i+1}^*]}{[tmax[i]^+, t[i+1]*]}, \eqn{i=1, \ldots, m}{i=1,..., m}, \eqn{[t_0, t_1^*]}{[t0, t[1]*]} and 
\eqn{[t_{max,m}^{+}, T]}{[tmax[m]^+, T]}, in order to reduce the computational cost of approximating the f.p.t. density function 
in those cases where \eqn{t_{i+1}^* - t_{max,i}^+ >> t_{max,i}^{-} - \thinspace t_i^*}{t[i+1]* - tmax[i]^+ >> tmax[i]^- - t[i]*}, 
for some \eqn{i}, \eqn{t_1^* - t_0 >> t_{max,1}^{-} - \thinspace t_1^*}{t[1]* - t0 >> tmax[1]^- - t[1]*} or 
\eqn{T - t_{max,m}^+ >> t_{max,m}^{-} - \thinspace t_m^*}{T - tmax[m]^+ >> tmax[m]^- - t[m]*}, respectively.} 
  \item{skip}{a logical value indicating whether the intervals at which the FPTL function is near zero could be
avoided.} 
  \item{tol}{If the cumulative integral of the approximation is greater than or equal to 1 - tol the algorithm is stopped.}
  \item{it.max}{If the number of iterations required for the approximation process is greater than it.max, the function asks for permission to continue.
  Defaults to 50000 for a conditioned f.p.t. problem, and 1000000 for an unconditioned f.p.t. problem.}
}
\details{
For an unconditioned f.p.t. problem, this function allows to compute directly the approximate f.p.t. density from an object of class \dQuote{diffproc}. 
In contrast to the approximation of a conditioned f.p.t. density by using the \code{Approx.cfpt.density} function, in this case it is not necessary for users to make 
previous calls to the functions \code{FPTL} and \code{summary.FPTL}. \cr

For a diffusion process \eqn{\{X(t), t_0 \leq t \leq T \}}{{X(t), t0 \le t \le T}} with non-degenerate initial distribution 
the unconditioned f.p.t. variable, through a continuous boundary \eqn{S(t)}, is defined as

\ifelse{latex}{\deqn{T_{S(t), X(t_0)} = \left\{
\begin{array}{lll}
\mbox{Inf} \ \{ t \geq t_0 \ : \ X(t) > S(t) \} & & \mbox{if \ } X(t_0) < S(t_0) \vspace{5pt} \\
\mbox{Inf} \ \{ t \geq t_0 \ : \ X(t) < S(t) \} & & \mbox{if \ } X(t_0) > S(t_0)
\end{array}
 \right.}}{\deqn{T = Inf{ t \ge t0 : X(t) > S(t)} ,} 
if \eqn{X(t0) < S(t0)}, and
\deqn{T = Inf{ t \ge t0 : X(t) < S(t)} ,} 
if \eqn{X(t0) > S(t0)}.}

and its density function can be obtained by means of the numerical integration, in the range of variation of \eqn{X(t_0)}{X(t0)}, 
of the corresponding f.p.t. densities conditioned to values of \eqn{X(t_0)}{X(t0)}, weighted by the initial density function. \cr

For each conditioned problem related to an unconditioned f.p.t. problem, the \code{Approx.fpt.density} function makes internal calls to  
\code{\link{FPTL}} and \code{\link{summary.fptl}} functions, in order to localize each conditioned f.p.t. variable, according to the 
\code{zeroSlope}, \code{p0.tol} and \code{k} arguments. Then, the function makes a internall
call to \code{\link{Integration.Steps}} function, in order to determine the suitable subintervals and 
integration steps to be used to approximate the unconditioned f.p.t. density and conditioned f.p.t. densities according to the \code{variableStep}, 
\code{from.t0}, \code{to.T}, \code{n}, \code{p} and \code{alpha} arguments. \cr

From this information, a suitable sequence of time instants in \eqn{[t_0, T]}{[t0, T]} for the approximation of the f.p.t. densities is available.
For each time instant t in such sequence, the \code{Approx.fpt.density} function computes the value of the f.p.t. density conditioned to 
\code{m} values of the initial distribution (equally spaced in its range of variation), and then it computes the value of the unconditioned f.p.t density. \cr

For the approximation of each conditioned f.p.t density, the numerical procedure proposed by Buonocore et al. (1987), 
based on the composite trapezoid method, has been implemented. This method has also been used in the numerical integration for obtaining
the unconditioned f.p.t. density. \cr

The mathematical expression of the boundary \code{S} should be a function of \eqn{t} and may include the argument \code{t0},
the name \code{x0}, to refer to an initial value of the process, and the parameters specified in the \code{env} argument. 
The function checks if the mathematical expression shows syntax errors and if \R can compute its symbolic derivative with respect to \eqn{t}. \cr
	
The \code{env} argument is a list of tagged values in \code{name = value} form with \code{name} other than \eqn{x}, \eqn{t}, \eqn{y} and \eqn{s}.
To name the expression of a function \eqn{h(u)} as a character string we can use 
\eqn{\grave{} \thinspace}{}\if{text}{`}\if{html}{\out{&#96;}}\code{h(u)}\eqn{\grave{} \ }{}\if{text}{`}\if{html}{\out{&#96;}} \code{= value} 
if we want to show its dependence on \eqn{u}, or \code{h = value} otherwise. \cr

The \code{env} argument is copied into a temporary environment for evaluating the mathematical expressions in objects \code{dp} and \code{S}. 
\R looks for the objects not found into this temporary environment in the parent.frame() environment. \cr

By default the function does not compute the approximate f.p.t. density from the time instant \eqn{t_0}{t0}, but from a more suitable time instant 
provided by the First-Passage-Time Location (FPTL) function. It also uses a variable integration step. \cr

If \code{skip = TRUE}, for each subinterval in which it could be possible to avoid calculating all the conditioned f.p.t. densities,
the function checks the value of the approximate unconditioned f.p.t. density
at the lower end of such subinterval. If it is almost 0, the approximate unconditioned f.p.t. density
calculation is avoided in the subinterval, and it continues from the upper end of the subinterval
considering a zero value of the approximate unconditioned f.p.t. density. \cr

Similarly, if \code{to.T = FALSE}, the function checks the cumulative value of the integral for each upper end of the
subintervals for which it is not possible to avoid the application of the numerical algorithm for each conditioned problem. 
If it is greater than or equal to 1 - \code{tol}, the approximation procedure
is stopped. In any case, the procedure is stopped at the upper end of the last subinterval,
and if the cumulative value of the integral is less than 1 - \code{tol} the function issues a warning. \cr
}
\value{
The \code{Approx.fpt.density} function computes and returns an object of class \dQuote{fpt.density}. This object 
is a three-component list:
  \item{x}{a sequence of suitable time instants in \eqn{[t_0, \ T]}{[t0, T]} according to the arguments in the function call.}
  \item{y}{the approximate f.p.t. density function values on the x sequence for the unconditioned or conditioned problem at hand.}
  \item{y.x0}{NULL for a conditioned f.p.t. problem or a matrix with the values, by columns, of the approximate f.p.t. 
  densities conditioned to each considered value \eqn{x_0}{x0} selected from the initial distribution for an unconditioned f.p.t. problem.} \cr
  
It also includes six additional attributes:
\tabular{rl}{
\code{Call} \tab the unevaluated function call, substituting each object of length 1 (referred \cr 
\code{} \tab by name in call) by its value. \cr
\code{Steps} \tab matrix of subintervals and integration steps to consider for computing the \cr 
\code{} \tab approximate f.p.t. density. \cr 
\code{cumIntegral} \tab vector of the values of the cumulative integral of the approximation for \cr
\code{} \tab each subinterval considered. \cr
\code{skips} \tab a list that contains, for each subinterval, the indexes of the initial values \cr
\code{} \tab for which the calculation of the approximate conditioned f.p.t. densities \cr
\code{} \tab has been avoided. \cr
\code{CPUTime} \tab matrix of user and system times, by columns, required to approximate \cr
\code{} \tab the density for each subinterval considered, by rows. \cr
\code{summary.fptl} \tab the object of class \dQuote{summary.fptl} that results in the internal calls to the \cr
\code{} \tab \code{summary.fptl} function which is used as \code{sfptl} argument in the internal \cr
\code{} \tab call to the \code{Integration.Steps} function. \cr
}

\code{x} is the vector result of the concatenation of the sequences of equally spaced values in the suitable subintervals 
determined by the \code{\link{Integration.Steps}} function.                                                                                          
}
\references{
Buonocore, A., Nobile, A.G. and Ricciardi, L.M. (1987) A new integral equation for the evaluation of
first-passage-time probability densities. \emph{Adv. Appl. Probab.}, \bold{19}, 784--800.

Romn, P., Serrano, J. J., Torres, F. (2008) First-passage-time location function: Application to determine first-passage-time densities in diffusion processes. \emph{Comput. Stat. Data Anal.}, \bold{52}, 4132--4146.

P. Romn-Romn, J.J. Serrano-Prez, F. Torres-Ruiz. (2012) An R package for an efficient approximation of first-passage-time densities for diffusion processes based on the FPTL function. \emph{Applied Mathematics and Computation}, \bold{218}, 8408--8428.

P. Romn-Romn, J.J. Serrano-Prez, F. Torres-Ruiz. (2014) More general problems on first-passage times for diffusion processes: A new version of the fptdApprox R package. \emph{Applied Mathematics and Computation}, \bold{244}, 432--446.
}
\author{Patricia Romn-Romn, Juan J. Serrano-Prez and Francisco Torres-Ruiz.}
\seealso{
\code{\link{diffproc}} about creation of class \dQuote{diffproc} objects.

\code{\link{is.fpt.density}} to test for objects of class \dQuote{fpt.density}.

\code{\link{print.fpt.density}} to show objects of class \dQuote{fpt.density}. 

\code{\link{report.fpt.density}} to generate a report.

\code{\link{plot.fpt.density}} for graphical display.

\code{\link{FPTL}} to evaluate the FPTL function and create objects of class \dQuote{fptl}.

\code{\link{summary.fptl}} to locate the f.p.t. variable and create objects of class \dQuote{summary.fptl}.
}
\examples{
## Continuing the diffproc(.) example:
\dontshow{Lognormal <- diffproc(c("m*x","sigma^2*x^2","dnorm((log(x)-(log(y)+(m-sigma^2/2)*(t-s)))/(sigma*sqrt(t-s)),0,1)/(sigma*sqrt(t-s)*x)", "plnorm(x,log(y)+(m-sigma^2/2)*(t-s),sigma*sqrt(t-s))"))}
## Making an efficient approximation of the f.p.t. density
## in the case of a conditioned f.p.t. problem. (optimal 
## variable integration steps and small computational cost)
yyy.cp <- Approx.fpt.density(dp = Lognormal, t0 = 0, T = 18, id = 1, S = 
                        "4.5 + 4*t^2 + 7*t*sqrt(t)*sin(6*sqrt(t))", 
                        env = list(m = 0.48, sigma = 0.07))
yyy.cp

\dontrun{
## Making a less efficient approximation of the f.p.t. density 
## (optimal fixed integration step but high computational cost related to 
##  the efficient approximation)
yyy1.cp <- Approx.fpt.density(dp = Lognormal, t0 = 0, T = 18, id = 1, S = 
                        "4.5 + 4*t^2 + 7*t*sqrt(t)*sin(6*sqrt(t))", 
                        env = list(m = 0.48, sigma = 0.07), 
                        variableStep = FALSE, from.t0 = TRUE, to.T = 
                        TRUE, skip = FALSE)
yyy1.cp

## Making an efficient approximation of the f.p.t. density 
## in the case of an unconditioned f.p.t. problem.
yyy.ucp <- Approx.fpt.density(dp = Lognormal, t0 = 0, T = 18, id = 
                list("dlnorm(x,-0.005,0.1)", "Lambda(-0.005,0.1)", 
                "\\\\Lambda(-0.005,0.1)", "Lognormal(-0.005,0.1)"),
                S = "4.5 + 4*t^2 + 7*t*sqrt(t)*sin(6*sqrt(t))", 
                env = list(m = 0.48, sigma = 0.07), m=25)                 
yyy.ucp}
}
\keyword{classes}
\keyword{list}
\keyword{methods}
\keyword{print}
