% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/flist.R
\name{flist}
\alias{flist}
\title{Create a formatting list}
\usage{
flist(..., type = "column", lookup = NULL, simplify = TRUE)
}
\arguments{
\item{...}{A set of formatting objects.}

\item{type}{The type of formatting list.  Valid values are 'row' or 'column'.
The default value is 'column'.}

\item{lookup}{A lookup vector.  Used for looking up the format from 
the formatting list.  This parameter is only used for 'row' type 
formatting lists.}

\item{simplify}{Whether to simplify the results to a vector.  Valid values 
are TRUE or FALSE.  Default is TRUE.  If the value is set to FALSE, the 
return type will be a list.}
}
\value{
A vector or list of formatted values.  The type of return value 
can be controlled with the \code{simplify} parameter.  The default return
type is a vector.
}
\description{
A formatting list contains more than one formatting object
}
\details{
To apply more than one formatting object to a vector, use a formatting
list.  There are two types of formatting list: column and row.  The column
type formatting lists applies all formats to all values in the
vector.  The row type formatting list can apply a different format to 
each value in the vector.  

Further, there are two styles of row type list: ordered and lookup.  The
ordered style applies each format in the list to the vector values
in the order specified.  The
ordered style will recycle the formats as needed.  The lookup style 
formatting list uses a lookup to determine which format from the list to
apply to a particular value of the vector.  The lookup column values should
correspond to names on the formatting list.  

Examples of column type and row type formatting lists are given below.
}
\examples{
#' ## Example 1: Formatting List - Column Type ##
# Set up data
v1 <- c(Sys.Date(), Sys.Date() + 30, Sys.Date() + 60)

# Create formatting list
fl1 <- flist("\%B", "The month is: \%s")

# Apply formatting list to vector
fapply(v1, fl1)


## Example 2: Formatting List - Row Type ordered ##
# Set up data
# Notice each row has a different data type
l1 <- list("A", 1.263, as.Date("2020-07-21"), 
          "B", 5.8732, as.Date("2020-10-17"))
          
# These formats will be recycled in the order specified           
fl2 <- flist(type = "row",
             c(A = "Label A", B = "Label B"),
             "\%.1f",
             "\%d\%b\%Y")

fapply(l1, fl2)


## Example 3: Formatting List - Row Type with lookup ##
# Set up data
# Notice each row has a different data type
l2 <- list(2841.258, "H", as.Date("2020-06-19"),
           "L", as.Date("2020-04-24"), 1382.8865)
v3 <- c("num", "char", "date", "char", "date", "num")

# Create formatting list
fl3 <- flist(type = "row", lookup = v3,
             num = function(x) format(x, digits = 2, nsmall = 1, 
                                  big.mark=","),
             char = value(condition(x == "H", "High"),
                     condition(x == "L", "Low"),
                     condition(TRUE, "NA")),
             date = "\%d\%b\%Y")

# Apply formatting list to vector, using lookup
fapply(l2, fl3)
}
\seealso{
\code{\link{fapply}} for information on how formats are applied
to a vector, \code{\link{value}} for how to create a user-defined format,
and \code{\link{as.flist}} to convert an existing list of formats 
to a formatting
list. Also see \link{FormattingStrings} for details on how to use
formatting strings.
}
