% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/epifit.R
\name{epifit}
\alias{epifit}
\title{Model fitting function for epifit package}
\usage{
epifit(modeleq = NULL, equations = "", initeq = "", itereq = "",
  endeq = "", data = NULL, subset = NULL, weight = NULL,
  na.action = NULL, opt = c("newrap", "BFGS", "CG", "Nelder-Mead", "SANN"),
  tol1 = 1e-08, tol2 = 1e-08, maxiter = 200, init = NULL,
  verbatim = 0, ...)
}
\arguments{
\item{modeleq}{a character string specifying the model. See \sQuote{Details}.}

\item{equations}{a character string specifying R expressions executed before \sQuote{modeleq}. Multiple expressions separated by \sQuote{;} is allowed. See \sQuote{Details}.}

\item{initeq}{a character string specifying R expressions executed once at the first time. Typical use is specifying initial values for parameters using R expressions. Multiple expressions separated by \sQuote{;} is allowed.}

\item{itereq}{a character string specifying R expressions executed during Cox regression at each time of event occur. Typical use is incorporating time-dependent variables. Not yet implemented completely.}

\item{endeq}{a character string specifying R expressions executed at the end of calculations.}

\item{data}{a data.frame in which to interpret the variables named in the formula, or in the subset and the weights argument.}

\item{subset}{expression indicating which subset of the rows of data should be used in the fit. All observations are included by default.}

\item{weight}{vector of case weights. If weights is a vector of integers, then the estimated coefficients are equivalent to estimating the model from data with the individual cases replicated as many times as indicated by weights. Only supported \sQuote{breslow} and \sQuote{efron} ties specification in Cox regression models and other likelihood specifications.}

\item{na.action}{a missing-data filter function. This is applied when data.frame is supplied as \sQuote{data} parameter. Default is \code{options()$na.action}.}

\item{opt}{a character string specifying the method for optimization. When sQuote{newrap} is specified, \code{nlm} function that uses Newton-type algorithm used for obtaining maximum likelihood estimate. For the rest of specifications (\sQuote{BFGS} for quasi-Newton method, \sQuote{CG} for conjugate gradients method, \sQuote{SANN} for simulated annealing, \sQuote{Nelder-Mead} for Nelder and Mead simplex method), \code{optim} is used. The default is \sQuote{newrap}.}

\item{tol1}{a numeric value specifying \code{gtol} in nlm, \code{abstol} in optim.}

\item{tol2}{a numeric value specifying \code{stol} in nlm, \code{reltol} in optim.}

\item{maxiter}{a integer value specifying the maximum number of iterations. Defaults is 200.}

\item{init}{initial values for the parameters specified in the form of vector.}

\item{verbatim}{a integer value from 0 (minimum) to 2 (maximum) controlling the amount of information printed during calculation.}

\item{...}{for the arguments used in the inner functions (currently not used).}
}
\value{
a list containing the result of model fitting including parameter estimates, variance of parameter estimates, log likelihood and so on.
}
\description{
This function maximizes an arbitrary likelihood including generalized linear models and Cox partial likelihood.
}
\details{
This function provides flexible model fitting. The main model specification is written in \code{modeleq}. \code{modeleq} consisits of two parts separated by \sQuote{~}. The distribution is specified in the first part, and the second part includes variable name which follows the specified distribution in the first part. Available distributional specifications are \sQuote{cox}(Cox partial likelihood), \sQuote{pois}(Poisson distribution), \sQuote{norm}(normal distribution), \sQuote{binom}(binomial distribution), \sQuote{nbinom}(negative binomial distribution), \sQuote{gamma}(gamma distribution) and \sQuote{weibull}(Weibull distribution). Options can be specified for some distribution specification after \sQuote{/} in the distribution specification part. Multiple options separated by \sQuote{,} can also be specified.

For Cox regressions, time and status variable must be specified in parentheses like \code{cox(time, status)}. Some options are also available for Cox regressions, and \sQuote{efron}, \sQuote{breslow}, \sQuote{discrete} and \sQuote{average} is available for tie handling method. \code{ties(discrete)} specification corresponds to \sQuote{exact} ties specification in \code{coxph} function, and \code{ties(average)} corresonds to \sQuote{exact} specification in SAS PHREG procecure. See references for further details. Strata option which specifies a variable indicating strata is also available in Cox regressions. Subset option which has same functinality as subset argument below is also available for Cox regressions and other distribution specifications. For other distribution specifications, parameters must be specified in parentheses. For poisson distribution, mean parameter must be specified as \code{pois(mean)}. Note that each parameter specificaiton can be a variable or R equation. For other distributions, \code{norm(mean, variance)}, \code{binom(size, probability)}, \code{nbinom(size, probability)}, \code{gamma(shape, scale)}, \code{weibull(shape, scale)}.

When distributions are specified (not Cox regressions), additional R expressions can be specified in \code{equations} argument. R expressions are parsed to make variable list. Variables which exist in data.frame or the global environment must be vector, and the rest of variables are regarded as parameters. If you define variable \sQuote{mu} in \code{equations}, you can use \sQuote{mu} in \code{modeleq} argument. Refer Poisson regression examples below.
}
\examples{
library(survival)

# Make sample data
set.seed(123)
nsub <- 20
follow <- 5
x <- rnorm(nsub)
rate <- exp(-0.5 + x)
etime <- rweibull(nsub, 1, 1/rate)
status <- as.integer(etime < follow)
time <- pmin(follow, etime)
dat <- data.frame(status, x, time)

coxph(Surv(time, status)~x, data=dat)
modeleq <- "cox(time,status)~exp(beta*x)"
epifit(modeleq=modeleq, data=dat)

glm(status ~ x + offset(log(time)), family=poisson(), data=dat)
equations <- "mu <- time*exp(beta0 + beta1*x)"
modeleq <- "pois(mu) ~ status"
epifit(modeleq=modeleq, equations=equations, data=dat)

# The simplest test data set from coxph function
test1 <- list(time=c(4,3,1,1,2,2,3),
              status=c(1,1,1,0,1,1,0),
              x=c(0,2,1,1,1,0,0),
              sex=c(0,0,0,0,1,1,1))

# Cox regressions with strata
coxph(Surv(time, status) ~ x + strata(sex), data=test1)
modeleq <- "cox(time,status)/strata(sex)~exp(beta*x)"
epifit(modeleq=modeleq, data=test1)

# Tie specification example in Cox regressions
coxph(Surv(time, status) ~ x + strata(sex), data=test1, ties="breslow")
modeleq <- "cox(time,status)/strata(sex),ties(breslow)~exp(beta*x)"
epifit(modeleq=modeleq, data=test1)

# Average partial likelihood
modeleq <- "cox(time,status)/strata(sex),ties(average)~exp(beta*x)"
epifit(modeleq=modeleq, data=test1)

# Conditional logistic regression for matched case-control studies
# hypothetical data
conlog <- data.frame(strata=c(1,1,2,2,3,3,4,4,5,5), outcome=c(1,0,1,0,1,0,1,0,1,0),
                     cov=c(1,3,2,1,5,2,4,2,2,2))
# Make dummy survival time so that all the cases in a matched set have the same survival
# time value, and the corresponding controls are censored at later times
conlog <- cbind(conlog, dummy=(2 - conlog$outcome))
coxph(Surv(dummy, outcome)~cov + strata(strata), ties="exact", data=conlog)
modeleq <- "cox(dummy,outcome)/ties(discrete),strata(strata)~exp(beta*cov)"
epifit(modeleq=modeleq, data=conlog)


# Joint model example (for demonstrating technical specifications)
nsub <- 1000
follow <- 10
x <- rnorm(nsub)
dose <- rweibull(nsub, 0.5, 1/(2.84)^2)
rate <- exp(-1 + x)*(1 + 0.5*dose)

# Generate survival data
etime <- rweibull(nsub, 1, 1/rate)
status <- as.integer(etime < follow)
time <- pmin(follow, etime)
dat2 <- data.frame(event=status, py=time, x, dose, model=1)

# Generate person-year table (baseline is different)
py <- runif(nsub)*follow
rate2 <- exp(-0.5 + 0.5*x)*(1 + 0.5*dose)
event <- sapply(rate2*py, function(x){rpois(1, x)})
dat3 <- cbind(pytable(event, py, cbind(x,dose)), model=2)
dat4 <- rbind(dat2, dat3)

modeleq <- c("cox(py,event)/subset(model==1)~exp(beta0*x)*(1 + beta*dose)",
             "pois(py*exp(beta1 + beta2*x)*(1 + beta*dose))/subset(model==2) ~ event")
epifit(modeleq, data=dat4)
}
\references{
DeLong, D. M., Guirguis, G.H., and So, Y.C. (1994). Efficient computation of subset selection probabilities with application to Cox regression. \emph{Biometrika} \strong{81}, 607-611.

Gail, M. H., Lubin, J. H., and Rubinstein, L. V. (1981). Likelihood calculations for matched case-control studies and survival studies with tied death times. \emph{Biometrika} \strong{68}, 703-707.
}
\seealso{
\code{\link{AIC.epifit}},
\code{\link{print.epifit}}
}

