#' plot.tsd plot result of tsd() that best describe temperature-dependent sex determination
#' @title Plot results of tsd() that best describe temperature-dependent sex determination
#' @author Marc Girondot
#' @return Nothing
#' @param x A result file generated by tsd()
#' @param ... Parameters for plot()
#' @param show.observations Should the observations be shown
#' @param las.x las parameter for x axis
#' @param las.y las parameter for y axis
#' @param lab.PT Label to describe pivotal temperature
#' @param lab.TRT Label to describe transitional range of temperature
#' @param males.freq Should the graph uses males frequency [TRUE] or females [FALSE]
#' @param col.TRT The color of TRT
#' @param col.TRT.CI The color of CI of TRT based on range.CI
#' @param col.PT.CI The color of CI of PT based on range.CI
#' @param show.CI Do the CI for the curve should be shown
#' @description Plot the estimates that best describe temperature-dependent sex determination.
#' @references Girondot, M. 1999. Statistical description of temperature-dependent sex determination using maximum likelihood. Evolutionary Ecology Research, 1, 479-486.
#' @references Godfrey, M.H., Delmas, V., Girondot, M., 2003. Assessment of patterns of temperature-dependent sex determination using maximum likelihood model selection. Ecoscience 10, 265-272.
#' @references Hulin, V., Delmas, V., Girondot, M., Godfrey, M.H., Guillon, J.-M., 2009. Temperature-dependent sex determination and global change: are some species at greater risk? Oecologia 160, 493-506.
#' @references Girondot M., Submited. On the concept of embryological thermosensitive period for sex determination in reptiles.
#' @examples
#' \dontrun{
#' CC_AtlanticSW <- subset(DatabaseTSD, RMU=="Atlantic, SW" & 
#'                           Species=="Caretta caretta" & Sexed!=0)
#' tsdL <- with (CC_AtlanticSW, tsd(males=Males, females=Females, 
#'                                  temperatures=Incubation.temperature-Correction.factor, 
#'                                  equation="logistic"))
#' plot(tsdL)
#' }
#' @method plot tsd
#' @export



plot.tsd <- function(x, ..., show.observations=TRUE, males.freq=TRUE, 
	las.x=1, las.y=1, lab.PT=paste("Pivotal ", x$type), 
	lab.TRT=paste0("Transitional range of ",  x$type, "s l=",l*100,"%"), 
	col.TRT="gray", col.TRT.CI=rgb(0.8, 0.8, 0.8, 0.5), 
  col.PT.CI=rgb(0.8, 0.8, 0.8, 0.5), show.CI=TRUE) {

# show.observations=TRUE; males.freq=TRUE; las.x=1; las.y=1; lab.PT="Pivotal"; lab.TRT=paste0("Transitional range l=95%"); col.TRT="gray"; col.TRT.CI=rgb(0.8, 0.8, 0.8, 0.5); col.PT.CI=rgb(0.8, 0.8, 0.8, 0.5); show.CI=TRUE
  
  range.CI.qnorm <- qnorm(1-((1-x$range.CI)/2))
  l <- x$l
  males <- x$males
  females <- x$females
  N <- x$N
  temperatures <- x$temperatures
  equation <- x$equation  
  par <- x$par
  res <- x$SE
 
  result <- x
 
  if (equation!="GSD") {

   L <- list(...)
   
   xlll <- ifelse(x$type=="temperature", expression("Temperatures in " * degree * "C"), 
                  "Durations in days")

	if (males.freq) {
		L1 <- modifyList(list(x=temperatures, y=males/N, bty="n", type="n", xlab=xlll, ylab="male frequency"), L)
	} else {
		L1 <- modifyList(list(x=temperatures, y=females/N, bty="n", type="n", xlab=xlll, ylab="female frequency"), L)	
	}
  L1 <- modifyList(L1, list(ylim=c(0,1), xaxt="n", las=las.y))
  
  if (is.null(L$xlim)) {
    L1 <- modifyList(L1, list(xlim=c(floor(min(temperatures)), floor(1+max(temperatures)))))
  }
  
  L2 <- L1[!(names(L1) %in% c("errbar.tick", "errbar.lwd"
                              , "errbar.lty", "errbar.col"
                              , "errbar.y.polygon"
                              , "errbar.y.polygon.list"))]
  
  a <- do.call(plot, L2) 
  
  x2 <- (par("usr")[1]+par("usr")[2]*26)/27
  x1 <- x2*26-par("usr")[2]/0.04
  
  cex.x <- par("cex.axis")
  if (!is.null(L$cex.axis)) cex.x <- L$cex.axis
  
  axis(1, at=x1:x2, las=las.x, cex.axis=cex.x)
  
  # je trace la TRT centree sur P
  
  
if (equation!="GSD") {
  polygon(c(result$TRT_limits[1], result$TRT_limits[1], result$TRT_limits[2], result$TRT_limits[2]), c(0,1,1,0), border=NA, col=col.TRT)  
  # limites de la limite basse de la TRT
  polygon(c(result$TRT_limits[1]+range.CI.qnorm*result$SE_TRT_limits[1], result$TRT_limits[1]+range.CI.qnorm*result$SE_TRT_limits[1], result$TRT_limits[1]-range.CI.qnorm*result$SE_TRT_limits[1], result$TRT_limits[1]-range.CI.qnorm*result$SE_TRT_limits[1]), c(0,1,1,0), border=NA, col=col.TRT.CI)
  # limites de la limite haute de la TRT
  polygon(c(result$TRT_limits[2]+range.CI.qnorm*result$SE_TRT_limits[2], result$TRT_limits[2]+range.CI.qnorm*result$SE_TRT_limits[2], result$TRT_limits[2]-range.CI.qnorm*result$SE_TRT_limits[2], result$TRT_limits[2]-range.CI.qnorm*result$SE_TRT_limits[2]), c(0,1,1,0), border=NA, col=col.TRT.CI)
  # limites de la PT
  polygon(c(par["P"]-range.CI.qnorm*res["P"], par["P"]-range.CI.qnorm*res["P"], par["P"]+range.CI.qnorm*res["P"], par["P"]+range.CI.qnorm*res["P"]), c(0,1,1,0), border=NA, col=col.PT.CI)
  par(xpd=TRUE)
  segments(par["P"], 0, par["P"], 1.05, lty=4)
  segments(result$TRT_limits[1], 0, result$TRT_limits[1], 1.15, lty=3)
  segments(result$TRT_limits[2], 0, result$TRT_limits[2], 1.15, lty=3)
  text(x=par["P"], y=1.1, lab.PT)
  text(x=par["P"], y=1.2, lab.TRT)
  
}

  if (show.observations) {
	if (males.freq) {  
  	  b <- HelpersMG::.BinomialConfidence(males,N)
  	  L1 <- modifyList(list(x=temperatures, y=males/N, bty="n", type="p", ylim=c(0,1), y.plus = b[,3], y.minus = b[,2]), L)
  	} else {
  	  b <- HelpersMG::.BinomialConfidence(females,N)
  	  L1 <- modifyList(list(x=temperatures, y=females/N, bty="n", type="p", ylim=c(0,1), y.plus = b[,3], y.minus = b[,2]), L)
  	}
  L1 <- modifyList(L1, list(ylim=c(0,1), xlab="", ylab="", main="", axes=FALSE, xlim=c(x1, x2)))
  
  par(xpd=FALSE)

  par(new=TRUE)
  
  a <- do.call(plot_errbar, L1) 
  }
  
  if (!is.null(result$CI)) {

x <- result$CI["temperatures", ]
p <- result$CI["mean", ]
   
  par(new=TRUE)
	if (males.freq) {   
  L1 <- modifyList(list(x=x, y=p, bty="n"), L)
  } else {
  L1 <- modifyList(list(x=x, y=1-p, bty="n"), L)
  }
  L1 <- modifyList(L1, list(ylim=c(0,1), axes=FALSE, xlab="", ylab="", type="l", main="", xlim=c(x1, x2)))

  L2 <- L1[!(names(L1) %in% c("errbar.tick", "errbar.lwd"
                              , "errbar.lty", "errbar.col"
                              , "errbar.y.polygon"
                              , "errbar.y.polygon.list"))]
  
  a <- do.call(plot, L2)

if (show.CI) {
  
  pm <- result$CI["2.5%", ]
  pp <- result$CI["97.5%", ]
  
	par(new=TRUE)
  if (males.freq) {
    L1 <- modifyList(list(x=x, y=pm, bty="n"), L)
  } else {
    L1 <- modifyList(list(x=x, y=1-pm, bty="n"), L)
  }
  L1 <- modifyList(L1, list(ylim=c(0,1), axes=FALSE, xlab="", ylab="", type="l", main="", lty=2, xlim=c(x1, x2)))
  
  L2 <- L1[!(names(L1) %in% c("errbar.tick", "errbar.lwd"
                              , "errbar.lty", "errbar.col"
                              , "errbar.y.polygon"
                              , "errbar.y.polygon.list"))]
  
  a <- do.call(plot, L2) 
  
  par(new=TRUE)
  if (males.freq) {   
    L1 <- modifyList(list(x=x, y=pp, bty="n"), L)
  } else {
    L1 <- modifyList(list(x=x, y=1-pp, bty="n"), L)
  }
  L1 <- modifyList(L1, list(ylim=c(0,1), axes=FALSE, xlab="", ylab="", type="l", main="", lty=2, xlim=c(x1, x2)))
  
  L2 <- L1[!(names(L1) %in% c("errbar.tick", "errbar.lwd"
                              , "errbar.lty", "errbar.col"
                              , "errbar.y.polygon"
                              , "errbar.y.polygon.list"))]
  
  a <- do.call(plot, L2) 
}
}
}
}
