% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ale.R
\name{ale}
\alias{ale}
\alias{ale.default}
\alias{ale.ranger}
\alias{ale.explainer}
\title{Accumulated Local Effects (ALE)}
\usage{
ale(object, ...)

\method{ale}{default}(
  object,
  v,
  data,
  pred_fun = stats::predict,
  trafo = NULL,
  which_pred = NULL,
  w = NULL,
  breaks = "Sturges",
  right = TRUE,
  discrete_m = 5L,
  outlier_iqr = 2,
  ale_n = 50000L,
  ale_bin_size = 200L,
  seed = NULL,
  ...
)

\method{ale}{ranger}(
  object,
  v,
  data,
  pred_fun = NULL,
  trafo = NULL,
  which_pred = NULL,
  w = NULL,
  breaks = "Sturges",
  right = TRUE,
  discrete_m = 5L,
  outlier_iqr = 2,
  ale_n = 50000L,
  ale_bin_size = 200L,
  seed = NULL,
  ...
)

\method{ale}{explainer}(
  object,
  v = colnames(data),
  data = object$data,
  pred_fun = object$predict_function,
  trafo = NULL,
  which_pred = NULL,
  w = object$weights,
  breaks = "Sturges",
  right = TRUE,
  discrete_m = 5L,
  outlier_iqr = 2,
  ale_n = 50000L,
  ale_bin_size = 200L,
  seed = NULL,
  ...
)
}
\arguments{
\item{object}{Fitted model.}

\item{...}{Further arguments passed to \code{pred_fun()}, e.g., \code{type = "response"} in
a \code{glm()} or (typically) \code{prob = TRUE} in classification models.}

\item{v}{Vector of variable names to calculate statistics.}

\item{data}{Matrix or data.frame.}

\item{pred_fun}{Prediction function, by default \code{stats::predict}.
The function takes three arguments (names irrelevant): \code{object}, \code{data}, and \code{...}.}

\item{trafo}{How should predictions be transformed?
A function or \code{NULL} (default). Examples are \code{log} (to switch to link scale)
or \code{exp} (to switch from link scale to the original scale).}

\item{which_pred}{If the predictions are multivariate: which column to pick
(integer or column name). By default \code{NULL} (picks last column).}

\item{w}{Optional vector with case weights. Can also be a column name in \code{data}.}

\item{breaks}{An integer, vector, string or function specifying the bins
of the numeric X variables as in \code{\link[graphics:hist]{graphics::hist()}}. The default is "Sturges".
To allow varying values of \code{breaks} across variables, it can be a list of the
same length as \code{v}, or a \emph{named} list with \code{breaks} for certain variables.}

\item{right}{Should bins be right-closed? The default is \code{TRUE}.
Vectorized over \code{v}. Only relevant for numeric X.}

\item{discrete_m}{Numeric X variables with up to this number of unique values
should not be binned and treated as a factor (after calculating partial dependence)
The default is 5. Vectorized over \code{v}.}

\item{outlier_iqr}{Outliers of a numeric X are capped via the boxplot rule, i.e.,
outside \code{outlier_iqr} * IQR from the quartiles. The default is 2 is more
conservative than the usual rule to account for right-skewed distributions.
Set to 0 or \code{Inf} for no capping. Note that at most 10k observations are sampled
to calculate quartiles. Vectorized over \code{v}.}

\item{ale_n}{Size of the data used for calculating ALE.
The default is 50000. For larger \code{data} (and \code{w}), \code{ale_n} rows are randomly
sampled. Each variable specified by \code{v} uses the same subsample. Set to 0 to omit.}

\item{ale_bin_size}{Maximal number of observations used per bin for ALE calculations.
If there are more observations in a bin, \code{ale_bin_size} indices are
randomly sampled. The default is 200. Applied after subsampling regarding \code{ale_n}.}

\item{seed}{Optional random seed (an integer) used for:
\itemize{
\item ALE: select background data if \code{n > ale_n} and for bins > \code{ale_bin_size}.
\item Capping X: quartiles are selected based on 10k observations.
}}
}
\value{
A list (of class "EffectData") with a data.frame of statistics per feature. Use
single bracket subsetting to select part of the output.
}
\description{
Calculates ALE for one or multiple \code{X} variables.

The concept of ALE was introduced in Apley et al. (2020) as an alternative to
partial dependence (PD). The Ceteris Paribus clause behind PD is a blessing and
a curse at the same time:
\itemize{
\item Blessing: The interpretation is easy and similar to what we know from linear
regression  (just averaging out interaction effects).
\item Curse: The model is applied to very unlikely or even impossible feature
combinations, especially with strongly dependent features.
}

ALE fixes the curse as follows: Partial dependence is calculated for the lower and
upper endpoint of a bin, using all (or a sample) of observations falling into this
bin. Its slope provides the \emph{local effect} over the bin.
This is repeated for all bins, and the values are \emph{accumulated}. Since the resulting
sum starts at 0, one typically shifts the result vertically, e.g., to the average
prediction. This is not done by \code{\link[=ale]{ale()}}, however.

The function is a convenience wrapper around \code{\link[=feature_effects]{feature_effects()}}, which calls
the barebone implementation \code{\link[=.ale]{.ale()}} to calculate ALE. The ALE values calculated
by \code{\link[=feature_effects]{feature_effects()}} are vertically shifted to the same (weighted) average than the
partial dependence curve, for optimal comparability.
}
\section{Methods (by class)}{
\itemize{
\item \code{ale(default)}: Default method.

\item \code{ale(ranger)}: Default method.

\item \code{ale(explainer)}: Default method.

}}
\examples{
fit <- lm(Sepal.Length ~ ., data = iris)
M <- ale(fit, v = "Petal.Length", data = iris)
M |> plot()

M2 <- ale(fit, v = colnames(iris)[-1], data = iris, breaks = 5)
plot(M2, share_y = "all")  # Only numeric variables shown
}
\references{
Apley, Daniel W., and Jingyu Zhu. 2020. \emph{Visualizing the Effects of Predictor Variables in Black Box Supervised Learning Models.}
Journal of the Royal Statistical Society Series B: Statistical Methodology,
82 (4): 1059–1086. doi:10.1111/rssb.12377.
}
\seealso{
\code{\link[=feature_effects]{feature_effects()}}, \code{\link[=.ale]{.ale()}}
}
