% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dynamic_proj_covariates.R
\name{dynamic_proj_covariates}
\alias{dynamic_proj_covariates}
\title{Combine explanatory variable rasters into covariates for each projection date.}
\usage{
dynamic_proj_covariates(
  dates,
  varnames,
  drive.folder,
  user.email,
  local.directory,
  spatial.ext,
  spatial.mask,
  spatial.res.degrees,
  resample.method,
  cov.file.type,
  prj = "+proj=longlat +datum=WGS84",
  cov.prj,
  save.directory,
  save.drive.folder,
  static.rasters,
  static.varnames,
  static.resample.method,
  static.moving.window.matrix,
  static.GEE.math.fun
)
}
\arguments{
\item{dates}{a character string, vector of dates in format "YYYY-MM-DD".}

\item{varnames}{a character string, the unique names for each explanatory variable.}

\item{drive.folder}{optional; a character string or vector, Google Drive folder or folders to read
projection covariate rasters from. Folder must be uniquely named within Google Drive. Do not
provide path.}

\item{user.email}{optional; a character string, user email for initialising Google Drive. Required
if \code{drive.folder} or \code{save.drive.folder} used.}

\item{local.directory}{optional; a character string or vector, path to local directory or
directories to read projection covariate rasters from.}

\item{spatial.ext}{optional; the spatial extent to crop explanatory variable
rasters to. Object of class \code{SpatExtent}, \code{SpatRaster}, an \code{sf} polygon or
numeric vector listing xmin, xmax, ymin and ymax in order.}

\item{spatial.mask}{an object of class \code{Raster}, \code{sf} or \code{Spatial}, representing a mask in which
NA cells in the mask layer are removed from the projection covariates.}

\item{spatial.res.degrees}{optional; a numeric value, the spatial resolution in degrees for
projection rasters to be resampled to. Required if \code{spatial.ext} given.}

\item{resample.method}{a character string or vector length of varnames, specifying resampling
method to use. One of \code{near} and \code{bilinear}. See details for more information.}

\item{cov.file.type}{a character string, the type of file to export projection covariates as. One
of: \code{tif} (SpatRaster with multiple layers) or \code{csv}(data frame).}

\item{prj}{a character string, the coordinate reference system desired for projection covariates.
Default is "+proj=longlat +datum=WGS84".}

\item{cov.prj}{a character string, the coordinate reference system desired for output projection
covariates. Default is assumed to be the same as \code{prj}.}

\item{save.directory}{optional; a character string, path to local directory to save projection
covariates to.}

\item{save.drive.folder}{optional; a character string, Google Drive folder to save projection
covariates to. Folder must be uniquely named within Google Drive. Do not provide path.}

\item{static.rasters}{a RasterStack of one or more rasters to be added to covariates for each date.}

\item{static.varnames}{a character string or vector, the unique names for each
explanatory variable in order of rasters in \code{static.raster} stack.}

\item{static.resample.method}{a character string or vector length of \code{static.varnames}, specifying resampling
method to use on static rasters provided. One of \code{near} and \code{bilinear}. See details for more information..}

\item{static.moving.window.matrix}{optional; a matrix of weights with an odd number
of sides, representing the spatial neighbourhood of cells (“moving
window”) to calculate \code{GEE.math.fun} across from record co-ordinate. See
details for more information.}

\item{static.GEE.math.fun}{optional; a character string, the mathematical function to
compute across the specified spatial matrix for each cell in \code{spatial.ext}}
}
\value{
Exports combined covariates in "csv" or "tif" file for each projection date to the local
directory or Google Drive folder.
}
\description{
Explanatory variable rasters are imported, resampled to a given spatial resolution and extent,
stacked and then exported as a covariate data frame or raster stack for each projection date.
}
\section{Input variable rasters}{
For each projection date, the rasters for each explanatory variable are imported from a local
directory or Google Drive folder.

Such rasters should be uniquely named "tif" files within the directory or drive folder and
contain the variable name (as stated in \code{varnames}) and projection date in format "YYYY-MM-DD".
If more than one “tif” file in the Google Drive folder or local directory matches the projection
date and explanatory variable name, then the function will error.
}

\section{Processing rasters}{
If required, rasters are cropped and resampled to the same spatial extent and
resolution. If \code{spatial.mask} is given, then cells with NA in this mask layer
are removed from the returned projection covariates. See \code{terra::mask()} in R
package \code{terra} for details (Hijmans et al., 2022).

Rasters are then stacked and reprojected if \code{cov.prj} is different to \code{prj}.

Note: if explanatory variable rasters are not of the same spatial resolution and extent, then the
function will error. Resample methods (\code{resample.method}) include:
\itemize{
\item \code{near}:  Each cell acquires the value of its nearest neighbour cell in the original raster. This
is typically used for categorical variables.
\item \code{bilinear}: the distance-weighted average of the four nearest cells are used to estimate a new
cell value. This is typically used for continuous variables.
}

If only one \code{resample.method} is given, but these are more than one explanatory variables, the
same \code{resample.method} is used for all.
}

\section{Output covariates}{
The raster stacks are then converted into data frames or remain as raster stacks depending on
\code{cov.file.type}. Column names or raster layer names will be the unique explanatory variable names
(\code{varnames}). These are exported to the local directory or Google Drive folder with file names
containing the relevant projection date in "YYYY-MM-DD" format.
}

\section{Google Drive compatibility}{
If \code{drive.folder} or \code{save.drive.folder} given, please ensure the folder name is unique within
your Google Drive. Do not provide the path if the folder is nested within others.

If one of \code{drive.folder} or \code{save.drive.folder} are used then user.email is required to access the
appropriate Google Drive user account. This requires users to have installed R package
\code{googledrive} and initialised Google Drive with valid log-in credentials. Please follow
instructions on \url{https://googledrive.tidyverse.org/}.
}

\section{Static rasters}{
If static datasets are to be added into the dynamic projection covariates,
then five arguments are available to specify their inclusion.

Please provide the static rasters in RasterStack format, specifying any
spatial buffering needed (using \code{static.moving.window.matrix} and
\code{static.GEE.math.fun} as described below).

The resample method or methods for rasters within the static raster stack need
to be specified too using \code{static.resample.method}. Please also provide
\code{static.varnames} to name the static variables in the covariate data exported.

#' # Spatial buffering of static rasters (optional)

Using the \code{focal} function from \code{terra} R package (Hijmans et al., 2022),
\code{GEE.math.fun} is calculated across the spatial buffer area from the record
co-ordinate. The spatial buffer area used is specified by the argument
\code{moving.window.matrix}, which dictates the neighbourhood of cells
surrounding the cell containing the occurrence record to include in this
calculation.

See function \code{get_moving_window()} to generate appropriate
\code{moving.window.matrix}.
}

\section{Mathematical function}{
\code{GEE.math.fun} specifies the mathematical function to be calculated over the
spatial buffered area and temporal period. Options are limited to Google
Earth Engine ImageCollection Reducer functions
(\url{https://developers.google.com/earth-engine/apidocs/}) for which an
analogous R function is available. This includes: "allNonZero","anyNonZero",
"count", "first","firstNonNull", "last", "lastNonNull", "max","mean",
"median","min", "mode","product", "sampleStdDev", "sampleVariance",
"stdDev", "sum" and "variance".
}

\examples{
\dontshow{if (googledrive::drive_has_token()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}

data("sample_extent_data")

# Set extraction variables
projectiondates <- dynamic_proj_dates("2018-01-01", "2018-12-01", interval = 3,interval.level =
"month")
variablenames <- c("eight_sum_prec", "year_sum_prec")
spatial.res.metres <- 500
cov_resolution <- 0.05

# Get Google Drive email
user.email<-as.character(gargle::gargle_oauth_sitrep()$email)
\dontshow{
projectiondates <- projectiondates[1]
spatial.res.metres <-20000
cov_resolution <- 5
}
extract_dynamic_raster(dates=projectiondates,
                      datasetname = "UCSB-CHG/CHIRPS/DAILY",
                      bandname="precipitation",
                      user.email = user.email,
                      spatial.res.metres = spatial.res.metres,
                      GEE.math.fun = "sum",
                      temporal.direction = "prior",
                      temporal.res = 56,
                      spatial.ext = sample_extent_data,
                      varname = variablenames[1],
                      save.directory = tempdir())


extract_dynamic_raster(dates=projectiondates,
                     datasetname = "UCSB-CHG/CHIRPS/DAILY",
                     bandname="precipitation",
                     user.email = user.email,
                     spatial.res.metres = spatial.res.metres,
                     GEE.math.fun = "sum",
                     temporal.direction = "prior",
                     temporal.res = 364,
                     spatial.ext = sample_extent_data,
                     varname = variablenames[2],
                     save.directory = tempdir())

dynamic_proj_covariates(dates = projectiondates,
                       varnames = variablenames,
                       local.directory = tempdir(),
                       spatial.ext = sample_extent_data,
                       spatial.mask = sample_extent_data,
                       spatial.res.degrees = cov_resolution,
                       resample.method = c("bilinear","bilinear"),
                       cov.file.type = "csv",
                       prj="+proj=longlat +datum=WGS84",
                       save.directory = tempdir())
\dontshow{\}) # examplesIf}
}
\references{
Hijmans, R.J., Bivand, R., Forner, K., Ooms, J., Pebesma, E. and Sumner, M.D.,
2022. Package 'terra'. Maintainer: Vienna, Austria.
}
