#' Composed error multivariate distribution
#'
#' Probablitiy density function, distribution and random number generation for the composed error multivariate distribution.
#'
#' @param x1 vector of quantiles for margin \code{1}.
#' @param mu1 vector of \eqn{\mu} for margin \code{1}.
#' @param sigma_v1 vector of \eqn{\sigma_V} for margin 1. Must be positive.
#' @param par_u1 vector of \eqn{\sigma_U} for margin \code{1}. Must be positive.
#' @param s1 \eqn{s=-1} for production and \eqn{s=1} for cost function for margin \code{1}.
#' @param x2 vector of quantiles for margin \code{2}.
#' @param mu2 vector of \eqn{\mu} for margin \code{2}.
#' @param sigma_v2 vector of \eqn{\sigma_V} for margin \code{2}. Must be positive.
#' @param par_u2 vector of \eqn{\sigma_U} for margin \code{2}. Must be positive.
#' @param s2 \eqn{s=-1} for production and \eqn{s=1} for cost function for margin \code{2}.
#' @param delta matrix of copula parameter. Must have at least one column.
#' @param family_mv string vector, specifying the the margin one and two, as well as the copula.
#' For the margins the distributions \code{normhnorm} or \code{normexp} are available. For the family_cop:\cr
#' `independent` = Independence copula \cr
#' `normal` = Gaussian copula \cr
#' `clayton` = Clayton copula \cr
#' `gumbel` = Gumbel copula \cr
#' `frank` = Frank copula \cr
#' `joe` = Joe copula \cr
#' @param deriv derivative of order \code{deriv} of the log density. Available are \code{0} and \code{2}.
#' @param tri optional, index arrays for upper triangular matrices, generated by \code{\link[mgcv:trind.generator]{trind.generator()}} and supplied to \code{chainrule()}.
#' @param log.p logical; if \code{TRUE}, probabilities p are given as \code{log(p)}.
#' @param check logical; if TRUE, check inputs.
#'
#' @return \code{dcomperr_mv} gives the density, \code{pcomperr_mv} the distribution and \code{rcomperr_mv} generates random numbers, with given parameters. If the derivatives are calculated these are provided as the attributes \code{gradient}, \code{hessian}, \code{l3} and \code{l4} of the output of the density.
#'
#' @details A bivariate random vector \eqn{(Y_1,Y_2)} follows a composed error multivariate distribution \eqn{f_{Y_1,Y_2}(y_1,y_2)}, which can be rewritten using Sklars' theorem via a copula
#' \deqn{f_{Y_1,Y_2}(y_1,y_2)=c(F_{Y_1}(y_1),F_{Y_2}(y_2),\delta) \cdot f_{Y_1}(y_1) f_{Y_2}(y_2) \qquad,}
#' where \eqn{c(\cdot)} is a copula function and \eqn{F_{Y_m}(y_m)},\eqn{f_{Y_m}(y_m)} are the marginal cdf and pdf respectively. \eqn{delta} is the copula parameter.
#'
#' @examples
#' set.seed(1337)
#' x2<-10;x1<-5
#' mu2<-7;mu1<-2
#' sigma_v2<-6;sigma_v1<-3
#' par_u2<-3;par_u1<-2
#' s2<-s1<--1
#' delta<-matrix(0.5,ncol=1, nrow=100)
#' family_mv=c("normhnorm","normhnorm","normal")
#'
#' pdf<-dcomperr_mv(x1, mu1, sigma_v1, par_u1, s1,
#'                  x2, mu2, sigma_v2, par_u2, s2,
#'                  delta[1, , drop=FALSE], family_mv, deriv = 2, tri=NULL, log.p=FALSE)
#'
#' cdf<-pcomperr_mv(q1=x1, mu1, sigma_v1, par_u1, s1,
#'                  q2=x2, mu2, sigma_v2, par_u2, s2,
#'                  delta[1, , drop=FALSE], family_mv, log.p=FALSE)
#' r<-rcomperr_mv(n=100, mu1, sigma_v1, par_u1, s1,
#'                mu2, sigma_v2, par_u2, s2,
#'                delta, family_mv)
#'
#' @references
#' \itemize{
#' \item \insertRef{aigner1977formulation}{dsfa}
#' }
#'
#' @export
#dcomperr_mv
dcomperr_mv<-function(x1=0, mu1=0, sigma_v1=1, par_u1=1, s1=-1,
                      x2=0, mu2=0, sigma_v2=1, par_u2=1, s2=-1,
                      delta=0, family_mv=c("normhnorm","normhnorm","normal"),
                      deriv = 0, tri=NULL, log.p=FALSE, check=TRUE){

  #Initialize values
  fy1<-NULL
  fy2<-NULL

  #Margin 1
  fy1<-dcomperr(x=x1, mu=mu1, sigma_v=sigma_v1, par_u=par_u1, s=s1, family=family_mv[1], deriv = deriv, log.p = TRUE, check=check)
  Fy1<-pcomperr(q=x1, mu=mu1, sigma_v=sigma_v1, par_u=par_u1, s=s1, family=family_mv[1], deriv = deriv, log.p = FALSE, check=check)

  #Margin 2
  fy2<-dcomperr(x=x2, mu=mu2, sigma_v=sigma_v2, par_u=par_u2, s=s2, family=family_mv[2], deriv = deriv, log.p = TRUE, check=check)
  Fy2<-pcomperr(q=x2, mu=mu2, sigma_v=sigma_v2, par_u=par_u2, s=s2, family=family_mv[2], deriv = deriv, log.p = FALSE, check=check)

  #Evaluate copula at probability integral transformed observations
  value<-dcop(W=as.matrix(cbind(Fy1,Fy2)), delta=delta, family_cop=family_mv[3], deriv = deriv, log.p = log.p)

  #Calculate log multivariate density
  out<-sum(value, fy1, fy2)
  names(out)<-NULL

  if(deriv>0){
    #Get number of parameters
    npar_fy1<-ncol(attr(fy1,"gradient"))
    npar_fy2<-ncol(attr(fy2,"gradient"))
    npar_cop<-ncol(attr(value,"gradient"))
    npar_comperr_mv<-npar_fy1+npar_fy2+npar_cop-2

    #Check if index arrays for upper triangular is available, else create it
    if(is.null(tri)){
      tri<-mgcv::trind.generator(npar_comperr_mv)
    }

    l1<-cbind(attr(fy1,"gradient"),attr(fy2,"gradient"),attr(value,"gradient")[,-c(1:2)])
    l1[,-c((npar_fy1+npar_fy2+1):npar_comperr_mv)]<-l1[,-c((npar_fy1+npar_fy2+1):npar_comperr_mv)]+
                                                    cbind(attr(value,"gradient")[,1]*attr(Fy1,"gradient"),
                                                          attr(value,"gradient")[,2]*attr(Fy2,"gradient"))

    l2_index<-unique(c(tri$i2[(1:npar_fy1),(1:npar_fy1)],
                       tri$i2[(npar_fy1+1):(npar_fy1+npar_fy2),(npar_fy1+1):(npar_fy1+npar_fy2)],
                       tri$i2[(npar_fy1+npar_fy2+1):npar_comperr_mv,(npar_fy1+npar_fy2+1):npar_comperr_mv]))
    l2<-matrix(0, nrow=length(value), ncol=max(tri$i2))

    tri_cop<-mgcv::trind.generator(npar_cop)

    l2[,l2_index]<-cbind(attr(fy1,"hessian"),attr(fy2,"hessian"),attr(value,"hessian")[,max(tri_cop$i2)])

    cop_index<-matrix(max(tri_cop$i2), nrow=npar_comperr_mv, ncol=npar_comperr_mv)
    cop_index[(1:npar_fy1),(1:npar_fy1)]<-1
    cop_index[(npar_fy1+1):(npar_fy1+npar_fy2),(1:npar_fy1)]<-2
    cop_index[(npar_fy1+npar_fy2+1):npar_comperr_mv,(1:npar_fy1)]<-3
    cop_index[(npar_fy1+1):(npar_fy1+npar_fy2),(npar_fy1+1):(npar_fy1+npar_fy2)]<-4
    cop_index[(npar_fy1+npar_fy2+1):npar_comperr_mv,(npar_fy1+1):(npar_fy1+npar_fy2)]<-5
    cop_index[upper.tri(cop_index,diag=FALSE)]<-0
    cop_index<-cop_index+t(cop_index)-diag(diag(cop_index))
    cop_index<-cop_index[lower.tri(cop_index,diag=TRUE)]

    F_index<-matrix(1:npar_comperr_mv, nrow=npar_comperr_mv, ncol=npar_comperr_mv, byrow=FALSE)
    tF_index<-t(F_index)

    F_index<-F_index[lower.tri(F_index,diag=TRUE)]
    tF_index<-tF_index[lower.tri(tF_index,diag=TRUE)]

    l2<-l2+attr(value,"hessian")[,cop_index]*
            cbind(attr(Fy1,"gradient"),attr(Fy2,"gradient"),0)[,F_index]*
            cbind(attr(Fy1,"gradient"),attr(Fy2,"gradient"),0)[,tF_index]

    cop_index_grad<-c(rep(1,ncol(attr(Fy1,"hessian"))),rep(2,ncol(attr(Fy2,"hessian"))))
    l2[,l2_index[-length(l2_index)]]<-l2[,l2_index[-length(l2_index)]]+attr(value,"gradient")[,cop_index_grad]*cbind(attr(Fy1,"hessian"),attr(Fy2,"hessian"))

    #Set gradient and hessian as attributes
    attr(out,"gradient")<-l1
    attr(out,"hessian")<-l2
  }

  if(!log.p){
    out<-exp(out)
  }

  #Return out
  return(out)
}

#' @describeIn dcomperr_mv distribution function for the composed error multivariate distribution.
#' @param q1 vector of quantiles for margin 1.
#' @param q2 vector of quantiles for margin 2.
#' @export
pcomperr_mv<-function(q1=0, mu1=0, sigma_v1=1, par_u1=1, s1=-1,
                      q2=0, mu2=0, sigma_v2=1, par_u2=1, s2=-1,
                      delta=0, family_mv=c("normhnorm","normhnorm","normal"),
                      log.p=FALSE, check=TRUE){

  #Margin 1
  Fy1<-pcomperr(q=q1, mu=mu1, sigma_v=sigma_v1, par_u=par_u1, s=s1, family=family_mv[1], deriv = 0, log.p = FALSE, check=check)

  #Margin 2
  Fy2<-pcomperr(q=q2, mu=mu2, sigma_v=sigma_v2, par_u=par_u2, s=s2, family=family_mv[2], deriv = 0, log.p = FALSE, check=check)

  #Evaluate cdf of copula at probability integral transformed observations
  out<-pcop(W=cbind(Fy1,Fy2), delta=delta, family_cop=family_mv[3], log.p = log.p)
  names(out)<-NULL

  #Return out
  return(out)
}

#' @describeIn dcomperr_mv random number generation for the composed error multivariate distribution.
#' @param n number of observations.
#' @export
rcomperr_mv<-function(n, mu1=0, sigma_v1=1, par_u1=1, s1=-1,
                         mu2=0, sigma_v2=1, par_u2=1, s2=-1,
                        delta=0, family_mv=c("normhnorm","normhnorm","normal"), check=TRUE){

  #Generate pseudo observations
  obs<-rcop(n=n, delta=delta, family_cop=family_mv[3])

  #Margin 1
  y1<-qcomperr(p=obs[,1], mu=mu1, sigma_v=sigma_v1, par_u=par_u1, s=s1, family=family_mv[1], check=check)

  #Margin 2
  y2<-qcomperr(p=obs[,2], mu=mu2, sigma_v=sigma_v2, par_u=par_u2, s=s2, family=family_mv[2], check=check)


  #Combind y1 and y2
  out<-cbind(y1,y2)

  #Return out
  return(out)
}
