#' Dot-and-Whisker Plots of Regression Coefficients from Tidy Data Frames
#'
#' \code{dwplot} is a function for quickly and easily generating dot-and-whisker plots of regression models saved in tidy data frames.
#'
#' @param df A data.frame including the variables \code{term} (names of independent variables), \code{estimate} (corresponding coefficient estimates), \code{std.error} (corresponding standard errors), and optionally \code{model} (when multiple models are desired on a single plot) such as generated those by \code{\link[broom]{tidy}}.
#' @param alpha A number setting the criterion of the confidence intervals. The default value is .05, corresponding to 95-percent confidence intervals.
#' @param dodge_size A number (typically between 0 and 0.3) indicating how much vertical separation should be between different models' coefficients when multiple models are graphed in a single plot.  Lower values tend to look better when the number of independent variables is small.
#'
#' @details \code{dwplot} visualizes regression results saved in tidy data.frames by, e.g., \code{\link[broom]{tidy}} as dot-and-whisker plots generated by \code{\link[ggplot2]{ggplot}}.
#'
#' Because the function takes a data.frame as input, it is easily employed for a wide range of models, and because the output is a \code{ggplot} object, it can be further customized with any additional arguments and layers supported by \code{ggplot2}.
#'
#' @references
#' Kastellec, Jonathan P. and Leoni, Eduardo L. 2007. "Using Graphs Instead of Tables in Political Science." Perspectives on Politics, 5(4):755-771.
#'
#' @return The function returns a \code{ggplot} object.
#'
#' @import ggplot2 dplyr
#' @importFrom stats qnorm
#'
#' @examples
#' library(broom)
#'
#' # Plot regression coefficients from a single model
#' data(mtcars)
#' m1 <- lm(mpg ~ wt + cyl + disp, data = mtcars)
#' m1_df <- tidy(m1) # create data.frame of regression results
#'
#'dwplot(m1_df) +
#'     scale_y_discrete(breaks = 4:1, labels=c("Intercept", "Weight", "Cylinders", "Displacement")) +
#'     theme_bw() + xlab("Coefficient") + ylab("") +
#'     geom_vline(xintercept = 0, colour = "grey50", linetype = 2) +
#'     theme(legend.position="none")
#'
#' # Plot regression coefficients from multiple models
#' library(dplyr)
#' by_trans <- mtcars %>% group_by(am) %>%
#'     do(tidy(lm(mpg ~ wt + cyl + disp, data = .))) %>% rename(model=am)
#'
#' dwplot(by_trans, dodge_size = .05) +
#'     scale_y_discrete(breaks = 4:1, labels=c("Intercept", "Weight", "Cylinders", "Displacement")) +
#'     theme_bw() + xlab("Coefficient Estimate") + ylab("") +
#'     geom_vline(xintercept = 0, colour = "grey60", linetype = 2) +
#'     ggtitle("Predicting Gas Mileage, OLS Estimates") +
#'     theme(plot.title = element_text(face="bold"),
#'           legend.justification=c(1,0), legend.position=c(1,0),
#'           legend.background = element_rect(colour="grey80"),
#'           legend.title.align = .5) +
#'     scale_colour_grey(start = .4, end = .8,
#'                       name = "Transmission",
#'                       breaks = c(0, 1),
#'                       labels = c("Automatic", "Manual"))
#'
#' @export

dwplot <- function(df, alpha = .05, dodge_size = .15) {

  n_vars <- length(unique(df$term))
  model <- NULL
  dodge_size = dodge_size

  if ("model" %in% names(df)) n_models <- length(unique(df$model)) else {
    if (length(df$term) == n_vars) {
      df$model <- 1
      n_models <- 1
    } else stop("Please add a variable named \'model\' to distinguish different models")
  }

  m_names <- unique(df$model)
  v_names <- df$term

  y_ind <- rep(seq(n_vars, 1), n_models)
  df$y_ind  <- y_ind

  estimate <- as.numeric(df$estimate)
  df$estimate <- estimate
  df$std.error <- as.numeric(df$std.error)

  if(alpha < 0 | alpha > 1) stop("Value of alpha for the confidential intervals should be between 0 and 1.")

  ci <- 1 - alpha/2
  lb <- c(df$estimate - qnorm(ci) * df$std.error)
  ub <- c(df$estimate + qnorm(ci) * df$std.error)

  df <- cbind(df, lb, ub)

  if(n_models == 1) shift <- 0 else
    shift <- seq(dodge_size, -dodge_size, length.out=n_models)

  shift_index <- data.frame(model = unique(df$model), shift, stringsAsFactors = FALSE)
  df <- dplyr::left_join(df, shift_index)

  if (length(y_ind)!=length(v_names)) v_names <- unique(v_names)

  p <- ggplot(df, aes(x = estimate, y = y_ind+shift, colour=factor(model))) +
      geom_point(na.rm = TRUE) +
      geom_segment(aes(x = lb,
                       xend = ub,
                       y = y_ind + shift, yend = y_ind + shift,
                       colour=factor(model)), na.rm = TRUE) +
      scale_y_discrete(breaks=y_ind, labels=v_names) +
      coord_cartesian(ylim=c(.5, n_vars+.5)) +
      ylab("")

  if (!"model" %in% names(df) | length(unique(df$model)) == 1) p <- p + theme(legend.position="none")

  return(p)
}
