% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit.R
\name{fit_one_layer}
\alias{fit_one_layer}
\title{MCMC sampling for one layer GP}
\usage{
fit_one_layer(
  x,
  y,
  nmcmc = 10000,
  trace = TRUE,
  g_0 = 0.01,
  theta_0 = 0.1,
  true_g = NULL,
  settings = list(l = 1, u = 2, alpha = list(g = 1.5, theta = 1.5), beta = list(g =
    3.9, theta = 3.9/1.5))
)
}
\arguments{
\item{x}{vector or matrix of input locations}

\item{y}{vector of response values}

\item{nmcmc}{number of MCMC iterations}

\item{trace}{logical indicating whether to print iteration progress}

\item{g_0}{initial value for \code{g}}

\item{theta_0}{initial value for \code{theta}}

\item{true_g}{if true nugget is known it may be specified here (set to a 
small value to make fit deterministic).  Note - values that are too 
small may cause numerical issues in matrix inversions.}

\item{settings}{hyperparameters for proposals and priors on \code{g} and 
\code{theta}}
}
\value{
a list of the S3 class "\code{gp}" with elements:
\itemize{
  \item \code{x}: copy of input matrix
  \item \code{y}: copy of response vector
  \item \code{nmcmc}: number of MCMC iterations
  \item \code{settings}: copy of proposal/prior settings
  \item \code{g}: vector of MCMC samples for \code{g}
  \item \code{theta}: vector of MCMC samples for \code{theta}
  \item \code{time}: computation time in seconds
}
}
\description{
Conducts MCMC sampling of hyperparameters for a one layer 
    GP.  Covariance structure is based on inverse exponentiated squared 
    euclidean distance with length scale parameter "\code{theta}" governing 
    the strength of the correlation and nugget parameter "\code{g}" 
    governing noise.
}
\details{
Utilizes Metropolis Hastings sampling of the length scale and
    nugget parameters with proposals and priors controlled by \code{settings}.
    Proposals for \code{g} and \code{theta} follow a uniform sliding window 
    scheme, e.g. 
    
    \code{g_star <- runif(1, l * g_t / u, u * g_t / l)}, 
    
    with defaults \code{l = 1} and \code{u = 2} provided in \code{settings}.
    Priors on \code{g} and \code{theta} follow Gamma distributions with 
    shape parameter (\code{alpha}) and rate parameter (\code{beta}) provided 
    in \code{settings}.  These priors are designed for "\code{x}" scaled 
    to [0,1] and "\code{y}" scaled to have mean 0 and variance 1.  
    
    The output object of class 
    "\code{gp}" is designed for use with \code{continue}, \code{trim}, and 
    \code{predict}.
}
\examples{
# Toy example (runs in less than 5 seconds) --------------------------------
# This example uses a small number of MCMC iterations in order to run quickly
# More iterations are required to get appropriate fits
# Function defaults are recommended (see additional example below)

f <- function(x) {
  if (x <= 0.4) return(-1)
  if (x >= 0.6) return(1)
  if (x > 0.4 & x < 0.6) return(10*(x-0.5))
}
x <- seq(0.05, 0.95, length = 7)
y <- sapply(x, f)
x_new <- seq(0, 1, length = 100)

# Fit model and calculate EI
fit <- fit_one_layer(x, y, nmcmc = 500)
fit <- trim(fit, 400)
fit <- predict(fit, x_new, lite = TRUE, store_all = TRUE)
ei <- EI(fit)

\donttest{
# One Layer and EI ---------------------------------------------------------

f <- function(x) {
  sin(5 * pi * x) / (2 * x) + (x - 1) ^ 4
}
  
# Training data
x <- seq(0.5, 2, length = 30)
y <- f(x) + rnorm(30, 0, 0.01)
  
# Testing data
xx <- seq(0.5, 2, length = 100)
yy <- f(xx)
  
# Standardize inputs and outputs
xx <- (xx - min(x)) / (max(x) - min(x))
x <- (x - min(x)) / (max(x) - min(x))
yy <- (yy - mean(y)) / sd(y)
y <- (y - mean(y)) / sd(y)
  
# Conduct MCMC
fit <- fit_one_layer(x, y, nmcmc = 10000)
plot(fit) # investigate trace plots
fit <- trim(fit, 8000, 2)
  
# Predict and calculate EI
fit <- predict(fit, xx, lite = TRUE, store_all = TRUE)
ei <- EI(fit)
  
# Visualize Fit
plot(fit)
par(new = TRUE) # overlay EI
plot(xx, ei$value, type = 'l', lty = 2, axes = FALSE, xlab = '', ylab = '')

# Select next design point
x_new <- xx[which.max(ei$value)]

# Evaluate fit
rmse(yy, fit$mean) # lower is better
}

}
\references{
Sauer, A, RB Gramacy, and D Higdon. 2020. "Active Learning for Deep Gaussian 
    Process Surrogates." arXiv:2012.08015. \cr\cr
Gramacy, RB. \emph{Surrogates: Gaussian Process Modeling, Design, and 
    Optimization for the Applied Sciences}. Chapman Hall, 2020.
}
