% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MOCClassif.R
\name{MOCClassif}
\alias{MOCClassif}
\title{MOC (Multi-Objective Counterfactual Explanations) for Classification Tasks}
\description{
MOC (Dandl et. al 2020) solves a multi-objective optimization problem to find counterfactuals. The four objectives
to minimize are:
\enumerate{
\item {\code{dist_target}: Distance to \code{desired_prob} (classification tasks) or \code{desired_prob} (regression tasks).}
\item {\code{dist_x_interest}: Dissimilarity to \code{x_interest} measured by Gower's dissimilarity measure (Gower 1971).}
\item {\code{no_changed}: Number of feature changes.}
\item {\code{dist_train}: (Weighted) sum of dissimilarities to the \code{k} nearest data points in \code{predictor$data$X}.}
}

For optimization, it uses the NSGA II algorithm (Deb et. al 2002) with mixed integer evolutionary
strategies (Li et al. 2013) and some tailored adjustments for the counterfactual search (Dandl et al. 2020).
Default values for the hyperparameters are based on Dandl et al. 2020.
}
\details{
Several population initialization strategies are available:
\enumerate{
\item {\code{random}: Feature values of new individuals are sampled from the feature value ranges in \code{predictor$data$X}.
Some features values are randomly reset to their initial value in \code{x_interest}.}
\item {\code{sd}: Like \code{random}, except that the sample ranges of numerical features are limited to one standard
deviation from their initial value in \code{x_interest}.}
\item {\code{icecurve}: As in \code{random}, feature values are sampled from the feature value ranges in \code{predictor$data$X}.
Then, however, features are reset with probabilities relative to their importance: the higher the importance
of a feature, the higher the probability that its values differ from its value in \code{x_interest}.
The feature importance is measured using ICE curves (Goldstein et al. 2015).}
\item {\code{traindata}: Contrary to the other strategies, feature values are drawn from (non-dominated) data points
in \code{predictor$data$X}; if not enough non-dominated data points are available, remaining individuals
are initialized by random sampling. Subsequently, some features values are randomly reset to their initial value
in \code{x_interest} (as for \code{random}).}
}

If \code{use_conditional_mutator} is set to TRUE, a conditional mutator samples
feature values from the conditional distribution given the other feature values
with the help of transformation trees (Hothorn and Zeileis 2017).
For details see Dandl et al. 2020.
}
\examples{
if (require("randomForest")) {
 \donttest{
  # Train a model
  rf = randomForest(Species ~ ., data = iris)
  # Create a predictor object
  predictor = iml::Predictor$new(rf, type = "prob")
  # Find counterfactuals for x_interest
  moc_classif = MOCClassif$new(predictor, n_generations = 15L, quiet = TRUE)
 
  cfactuals = moc_classif$find_counterfactuals(
    x_interest = iris[150L, ], desired_class = "versicolor", desired_prob = c(0.5, 1)
  )
  # Print the counterfactuals
  cfactuals$data
  # Plot evolution of hypervolume and mean and minimum objective values
  moc_classif$plot_statistics()
  }
}

}
\references{
Dandl, S., Molnar, C., Binder, M., and Bischl, B. (2020).
"Multi-Objective Counterfactual Explanations". In: Parallel Problem
Solving from Nature – PPSN XVI, edited by Thomas Bäck, Mike Preuss,
André Deutz, Hao Wang, Carola Doerr, Michael Emmerich, and Heike Trautmann, 448–469,
Cham, Springer International Publishing, \doi{10.1007/978-3-030-58112-1_31}.

Deb, K., Pratap, A., Agarwal, S., & Meyarivan, T. A. M. T. (2002).
"A fast and elitist multiobjective genetic algorithm: NSGA-II".
IEEE transactions on evolutionary computation, 6(2), 182-197.

Goldstein, A., Kapelner, A., Bleich, J., and Pitkin, E. (2015).
"Peeking Inside the Black Box: Visualizing
Statistical Learning with Plots of Individual Conditional Expectation".
Journal of Computational and Graphical Statistics 24 (1): 44–65.
\doi{10.1080/10618600.2014.907095}.

Gower, J. C. (1971). A general coefficient of similarity and some of its properties. Biometrics, 27, 623–637.

Hothorn, T., Zeileis, A. (2017), "Transformation Forests".
Technical Report, \href{https://arxiv.org/abs/1701.02110}{arXiv 1701.02110}.

Li, Rui, L., Emmerich, M. T. M., Eggermont, J. Bäck, T., Schütz, M., Dijkstra, J., Reiber, J. H. C. (2013).
"Mixed Integer Evolution Strategies for Parameter Optimization."
Evolutionary Computation 21 (1): 29–64. \doi{10.1162/EVCO_a_00059}.
}
\concept{get_dominated_hv}
\concept{plot_search}
\concept{plot_statistics}
\section{Super classes}{
\code{\link[counterfactuals:CounterfactualMethod]{counterfactuals::CounterfactualMethod}} -> \code{\link[counterfactuals:CounterfactualMethodClassif]{counterfactuals::CounterfactualMethodClassif}} -> \code{MOCClassif}
}
\section{Active bindings}{
\if{html}{\out{<div class="r6-active-bindings">}}
\describe{
\item{\code{optimizer}}{(\link[bbotk]{OptimInstanceMultiCrit}) \cr
The object used for optimization.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-MOCClassif-new}{\code{MOCClassif$new()}}
\item \href{#method-MOCClassif-plot_statistics}{\code{MOCClassif$plot_statistics()}}
\item \href{#method-MOCClassif-get_dominated_hv}{\code{MOCClassif$get_dominated_hv()}}
\item \href{#method-MOCClassif-plot_search}{\code{MOCClassif$plot_search()}}
\item \href{#method-MOCClassif-clone}{\code{MOCClassif$clone()}}
}
}
\if{html}{\out{
<details open><summary>Inherited methods</summary>
<ul>
<li><span class="pkg-link" data-pkg="counterfactuals" data-topic="CounterfactualMethod" data-id="print"><a href='../../counterfactuals/html/CounterfactualMethod.html#method-CounterfactualMethod-print'><code>counterfactuals::CounterfactualMethod$print()</code></a></span></li>
<li><span class="pkg-link" data-pkg="counterfactuals" data-topic="CounterfactualMethodClassif" data-id="find_counterfactuals"><a href='../../counterfactuals/html/CounterfactualMethodClassif.html#method-CounterfactualMethodClassif-find_counterfactuals'><code>counterfactuals::CounterfactualMethodClassif$find_counterfactuals()</code></a></span></li>
</ul>
</details>
}}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-MOCClassif-new"></a>}}
\if{latex}{\out{\hypertarget{method-MOCClassif-new}{}}}
\subsection{Method \code{new()}}{
Create a new \code{MOCClassif} object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{MOCClassif$new(
  predictor,
  epsilon = NULL,
  fixed_features = NULL,
  max_changed = NULL,
  mu = 20L,
  termination_crit = "gens",
  n_generations = 175L,
  p_rec = 0.71,
  p_rec_gen = 0.62,
  p_mut = 0.73,
  p_mut_gen = 0.5,
  p_mut_use_orig = 0.4,
  k = 1L,
  weights = NULL,
  lower = NULL,
  upper = NULL,
  init_strategy = "icecurve",
  use_conditional_mutator = FALSE,
  quiet = FALSE,
  distance_function = "gower"
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{predictor}}{(\link[iml]{Predictor})\cr
The object (created with \code{iml::Predictor$new()}) holding the machine learning model and the data.}

\item{\code{epsilon}}{(\code{numeric(1)} | \code{NULL})\cr
If not \code{NULL}, candidates whose prediction for the \code{desired_class} is farther away from the interval \code{desired_prob}
than \code{epsilon} are penalized. \code{NULL} (default) means no penalization.}

\item{\code{fixed_features}}{(\code{character()} | \code{NULL})\cr
Names of features that are not allowed to be changed. \code{NULL} (default) allows all features to be changed.}

\item{\code{max_changed}}{(\code{integerish(1)} | \code{NULL})\cr
Maximum number of feature changes. \code{NULL} (default) allows any number of changes.}

\item{\code{mu}}{(\code{integerish(1)})\cr
The population size. Default is \code{20L}.}

\item{\code{termination_crit}}{(\code{character(1)}|\code{NULL})\cr
Termination criterion, currently, two criterions are implemented: "gens" (default),
which stops after \code{n_generations} generations,  and "genstag", which stops after
the hypervolume did not improve for \code{n_generations} generations
(the total number of generations is limited to 500).}

\item{\code{n_generations}}{(\code{integerish(1)})\cr
The number of generations. Default is \code{175L}.}

\item{\code{p_rec}}{(\code{numeric(1)})\cr
Probability with which an individual is selected for recombination. Default is \code{0.71}.}

\item{\code{p_rec_gen}}{(\code{numeric(1)})\cr
Probability with which a feature/gene is selected for recombination. Default is \code{0.62}.}

\item{\code{p_mut}}{(\code{numeric(1)})\cr
Probability with which an individual is selected for mutation. Default is \code{0.73}.}

\item{\code{p_mut_gen}}{(\code{numeric(1)})\cr
Probability with which a feature/gene is selected for mutation. Default is \code{0.5}.}

\item{\code{p_mut_use_orig}}{(\code{numeric(1)})\cr
Probability with which a feature/gene is reset to its original value in \code{x_interest} after mutation. Default is \code{0.4}.}

\item{\code{k}}{(\code{integerish(1)})\cr
The number of data points to use for the forth objective. Default is \code{1L}.}

\item{\code{weights}}{(\code{numeric(1) | numeric(k)} | \code{NULL})\cr
The weights used to compute the weighted sum of dissimilarities for the forth objective. It is either a single value
or a vector of length \code{k}. If it has length \code{k}, the i-th element specifies the weight of the i-th closest data point.
The values should sum up to \code{1}. \code{NULL} (default) means all data points are weighted equally.}

\item{\code{lower}}{(\code{numeric()} | \code{NULL})\cr
Vector of minimum values for numeric features.
If \code{NULL} (default), the element for each numeric feature in \code{lower} is taken as its minimum value in \code{predictor$data$X}.
If not \code{NULL}, it should be named with the corresponding feature names.}

\item{\code{upper}}{(\code{numeric()} | \code{NULL})\cr
Vector of maximum values for numeric features.
If \code{NULL} (default), the element for each numeric feature in \code{upper} is taken as its maximum value in \code{predictor$data$X}.
If not \code{NULL}, it should be named with the corresponding feature names.}

\item{\code{init_strategy}}{(\code{character(1)})\cr
The population initialization strategy. Can be \code{icecurve} (default), \code{random}, \code{sd} or \code{traindata}. For more information,
see the \code{Details} section.}

\item{\code{use_conditional_mutator}}{(\code{logical(1)})\cr
Should a conditional mutator be used? The conditional mutator generates plausible feature values based
on the values of the other feature. Default is \code{FALSE}.}

\item{\code{quiet}}{(\code{logical(1)})\cr
Should information about the optimization status be hidden? Default is \code{FALSE}.}

\item{\code{distance_function}}{(\verb{function()} | \code{'gower'} | \code{'gower_c'})\cr
The distance function to be used in the second and fourth objective.
Either the name of an already implemented distance function
('gower' or 'gower_c') or a function.
If set to 'gower' (default), then Gower's distance (Gower 1971) is used;
if set to 'gower_c', a C-based more efficient version of Gower's distance is used.
A function must have three arguments  \code{x}, \code{y}, and \code{data} and should
return a \code{double} matrix with \code{nrow(x)} rows and maximum \code{nrow(y)} columns.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-MOCClassif-plot_statistics"></a>}}
\if{latex}{\out{\hypertarget{method-MOCClassif-plot_statistics}{}}}
\subsection{Method \code{plot_statistics()}}{
Plots the evolution of the mean and minimum objective values together with the dominated hypervolume over
the generations. All values for a generation are computed based on all non-dominated individuals that emerged until
that generation.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{MOCClassif$plot_statistics(centered_obj = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{centered_obj}}{(\code{logical(1)})\cr
Should the objective values be centered? If set to \code{FALSE}, each objective value is visualized in a separate plot,
since they (usually) have different scales. If set to \code{TRUE} (default), they are visualized in a single plot.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-MOCClassif-get_dominated_hv"></a>}}
\if{latex}{\out{\hypertarget{method-MOCClassif-get_dominated_hv}{}}}
\subsection{Method \code{get_dominated_hv()}}{
Calculates the dominated hypervolume of each generation.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{MOCClassif$get_dominated_hv()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A \code{data.table} with the dominated hypervolume of each generation.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-MOCClassif-plot_search"></a>}}
\if{latex}{\out{\hypertarget{method-MOCClassif-plot_search}{}}}
\subsection{Method \code{plot_search()}}{
Visualizes two selected objective values of all emerged individuals in a scatter plot.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{MOCClassif$plot_search(objectives = c("dist_target", "dist_x_interest"))}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{objectives}}{(\code{character(2)})\cr
The two objectives to be shown in the plot. Possible values are "dist_target", "dist_x_interest, "no_changed",
and "dist_train".}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-MOCClassif-clone"></a>}}
\if{latex}{\out{\hypertarget{method-MOCClassif-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{MOCClassif$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
