% 2023-12-15 A. Papritz correction of spelling errors and revision of text
% cd R.user.home/constrainedKriging/constrainedKriging_02-7/pkg/man/
% R CMD Rdconv -t html -o bla.html preCKrige.Rd ; open bla.html;
% R CMD Rd2pdf --force preCKrige.Rd;
% hunspell -d en_GB-ise,en_GB-ize,en_GB-large -t -i mac  preCKrige.Rd
% 2023-11-23 A. Papritz correction of example preCK_2
% 2023-12-09 A. Papritz correction of spelling errors and minor revision of
\name{preCKrige}
\docType{methods}
\alias{preCKrige-methods}
\alias{preCKrige}
\alias{preCKrige.points}
\alias{preCKrige.polygons}
\alias{preCKrige.pointsDF}
\alias{preCKrige.polygonsDF}
\alias{preCKrige,SpatialPoints,ANY,covmodel-method}
\alias{preCKrige,SpatialPointsDataFrame,ANY,covmodel-method}
\alias{preCKrige,SpatialPolygons,ANY,covmodel-method}
\alias{preCKrige,SpatialPolygonsDataFrame,ANY,covmodel-method}

\title{Spatial Variance-Covariance Matrices for Sets of Points and Polygons}
%%
\description{The function \code{preCKrige} computes (approximated) spatial
variance-covariance matrices for user-defined sets of points or polygons
(blocks) of any shape for two-dimensional isotropic random fields.  The
areas of a set of polygons (polygon neighbourhood configuration) are
approximated by pixels and the block-block covariances are approximated by
averaging covariances between the pixels used to approximate the polygons.

The object returned by \code{preCKrige} is needed by \code{\link{CKrige}}
for computing spatial point or block predictions by constrained,
covariance-matching constrained or universal (external drift) Kriging.
}
%%
\usage{
preCKrige(newdata, neighbours, model, \ldots)
%
\S4method{preCKrige}{SpatialPoints,ANY,covmodel}(newdata, neighbours, model)
%
\S4method{preCKrige}{SpatialPointsDataFrame,ANY,covmodel}(newdata, neighbours, model)
%
\S4method{preCKrige}{SpatialPolygons,ANY,covmodel}(newdata, neighbours, model,
  pwidth = 0, pheight = 0, napp = 1, ncores = 1L,
  fork = !identical( .Platform[["OS.type"]], "windows"))
%
\S4method{preCKrige}{SpatialPolygonsDataFrame,ANY,covmodel}(newdata, neighbours,
  model, pwidth = 0, pheight = 0, napp = 1, ncores = 1L,
  fork = !identical( .Platform[["OS.type"]], "windows"))
}

%% arguments
\arguments{

  %% newdata
  \item{newdata}{either an object of the class
  \dQuote{\code{SpatialPointsDataFrame}}
  or \dQuote{\code{SpatialPoints}} that contains the coordinates of the
  prediction points and optionally additional information (covariates) stored
  in the \code{data} slot of the \code{SpatialPointsDataFrame},
  or an object of the class \dQuote{\code{SpatialPolygonsDataFrame}} or
  \dQuote{\code{SpatialPolygons}}
  with the coordinates of the polygons (blocks) for which predictions are
  computed and optionally additional information (covariates) stored in the
  \code{data} slot of the
  \code{SpatialPolygonsDataFrame}.}

  %%% neighbours
  \item{neighbours}{a list of length \var{n} with integer vectors as
  components.  \var{n} is equal to the number of points if
  \code{newdata} is an object of class
  \dQuote{\code{SpatialPointsDataFrame}} or
  \dQuote{\code{SpatialPoints}} or equal to number of polygons (blocks)
  if \code{newdata} is an object of class
  \dQuote{\code{SpatialPolygonsDataFrame}} or
  \dQuote{\code{SpatialPolygons}}.

  The \var{i}th list component defines the neighbours of the \var{i}th
  point or \var{i}th polygon (block) in \code{newdata}, which form
  jointly with the \var{i}th point or polygon the so-called \emph{point}
  or \emph{polygon neighbourhood configuration}.  If \code{newdata} is
  an object of class \dQuote{\code{SpatialPolygonsDataFrame}} or
  \dQuote{\code{SpatialPolygons}} the \var{i}th list component contains
  the indices of the neighbouring polygons for the \var{i}th polygon.
  If \code{newdata} is an object of class \dQuote{\code{SpatialPoints}}
  or \dQuote{\code{SpatialPointsDataFrame}} the \var{i}th list component
  contains the row indices of the neighbouring points in the point
  coordinate matrix.  The \var{i}th list component is set to
  \code{integer(0)} if the \var{i}th polygon or \var{i}th point have no
  (defined) neighbours.  By default, the points or polygons have no
  neighbours.

  See the second example below where the function \code{poly2nb} of the
  package \pkg{spdep} is used to build a list of neighbours for target
  polygons of the  data set \code{\link{meuse.blocks}}.}

  %%& model
  \item{model}{an object of class \dQuote{\code{covmodel}}.  The object
  contains the parameters of the isotropic covariance function,
  generated by the function \code{\link{covmodel}}.}
  %%
  \item{\dots}{further arguments if \code{newdata} is of class
  \dQuote{\code{SpatialPolygonsDataFrame}} or\cr
  \dQuote{\code{SpatialPolygons}}.}

  %%pwidth
  \item{pwidth}{a positive numeric scalar, defines the width of the
  pixels used to approximate the polygon (block) areas.}

  %% pheight
  \item{pheight}{a positive numeric scalar, defines the height of the
  pixels used to approximate the polygon (block) areas.}

  %% napp
  \item{napp}{a positive integer scalar.  \code{napp} > 1 reduces the
  block-block variance-covariance approximation error.  By default,
  \code{napp} = 1, see \emph{Details.}}

  %% ncores
  \item{ncores}{a positive integer scalar with the number of CPUs to
  use for parallel computations.}

  %% fork
  \item{fork}{a logical scalar to control whether parallel
  computations are done by forking using \code{\link[parallel]{mclapply}}
  (non-windows OSes) or by socket clusters using
  \code{\link[parallel]{parLapply}} (windows OS).}

}
% end arguments
%
%
%% details
\details{ If the object \code{newdata} is of class
\dQuote{\code{SpatialPolygonsDataFrame}} or
\dQuote{\code{SpatialPolygons}} then\cr \code{preCKrige} searches the
polygon neighbourhood configuration (defined by \code{neighbours})
with the largest bounding box and generates a pixel grid that
completely covers the largest bounding box.  Subsequently, the
covariance matrix of this set of pixels is calculated by the
\pkg{spatialCovariance} package and the polygon (block) areas of each
polygon neighbourhood configuration are approximated by intersecting
the polygons with the shifted pixel grid, which yields a pixel
representation of the polygon neighbourhood configuration.  Finally,
the block-block covariances of the polygons are approximated by
averaging the covariances of the pixel representation of the
polygon neighbourhood configuration.

By default, \code{napp = 1}, which means that the approximation of the
block-block variance-covariance matrix for each polygon neighbourhood
configuration is computed just once.  If \code{napp} > 1 the
approximation of the block-block variance-covariance matrix for one
polygon neighbourhood configuration is based on the mean of
\code{napp} repetitions of the approximation to reduce the
approximation error.  Each of the \code{napp} block-block
variance-covariance approximations are based on a new, randomly
shifted pixel gird which results each time in a new pixel
representation of the polygon neighbourhood configuration.  Large
values of the argument \code{napp} increases the computation time.

There is a plot method \code{\link{plot.preCKrigePolygons}} for
\code{preCKrige} output objects of class\cr
\dQuote{\code{preCKrigePolygons}} to visually control the polygon
(block) area approximation by the pixels.  }
% end details
%%
\note{
A polygon (block) is treated as point if the polygon area is smaller than
the (defined) pixel area or if all pixel centroids of the generated pixel grid
lie outside the polygon (block) area.  If a pixel centroid lies
inside a polygon that has a smaller area than a pixel, the pixel is
allocated to the polygon (block) by which it shares the largest area.

The point-point correlations are calculated via the internal function
\code{CorrelationFct} (this function implements a subset of the
covariance models available previously in the function
\code{CovarianceFct} of the archived package \pkg{RandomFields},
version 2.0.71) and the point-block covariances are calculated by the C
function \code{PointRectCov} of the package.}
% end note

%% value
\value{\code{preCKrige} returns an S4 object, either of class
\dQuote{\code{preCKrigePolygons}} if
\code{newdata} is of class\cr \dQuote{\code{SpatialPolygons}} or
\dQuote{\code{SpatialPolygonsDataFrame}} or an S4 object of class
\dQuote{\code{preCKrigePoints}} if
\code{newdata} is of class \dQuote{\code{SpatialPoints}} or
\dQuote{\code{SpatialPointsDataFrame}}.

\emph{Notation}:
\tabular{ll}{
  \eqn{n} \tab number of polygons or points in \code{newdata},
               \var{i} = 1, \ldots, \eqn{n}\cr
  \eqn{m_i} \tab size of point or polygon neighbourhood configuration \cr
  \tab \eqn{m_i} = 1 + number of (defined) neighbours of the \var{i}th point
               or \var{i}th polygon \cr
  \eqn{r_{\mathrm{pix}}}{r_pix} \tab number of pixel grid rows \cr
  \eqn{c_{\mathrm{pix}}}{c_pix}  \tab number of pixel grid columns \cr
  \eqn{n_{\mathrm{pix}}}{n_pix}  \tab number of pixels in pixel grid
     \eqn{n_{\mathrm{pix}} = r_{\mathrm{pix}} \cdot c_{\mathrm{pix}}}{n_pix = r_pix * c_pix} \cr
}


%%% preCKrigePoints
An object of class \dQuote{\code{preCKrigePoints}} contains the
following slots:

\item{covmat}{a list of length \eqn{n}, the \eqn{i}th list component
contains the point-point covariance matrix of the \eqn{i}th prediction
point and its neighbours, i.e. of the \eqn{i}th point neighbourhood
configuration.}

\item{posindex}{a list of length \eqn{n}, the \eqn{i}th list component
contains a vector with the row indices of the \eqn{m_i - 1} neighbours
in the \eqn{i}th point neighbourhood configuration.}

\item{model}{an object of class \dQuote{\code{covmodel}} with the
parameters of the used covariance function.}

\item{data}{a data frame, which is the \code{data} slot of the
\code{SpatialPointsDataFrame} object.  This data frame is used to
build the design matrix of the prediction points by the
\code{\link{CKrige}} function.  \code{data} is empty with
\code{dim(data)} = (0, 0) if \code{newdata} is an object of class
\dQuote{\code{SpatialPoints}}.}

\item{coords}{a matrix with \code{dim(coords)} = (\eqn{n}, 2) with the
coordinates of the prediction points.}


%%% preCKrigePolygons
An object of class \dQuote{\code{preCKrigePolygons}} contains the
following slots:

\item{covmat}{a list of length \eqn{n}, the \eqn{i}th list component
contains the approximated block-block covariance matrix of the
\eqn{i}th polygon and its neighbours, i.e. of the \eqn{i}th polygon
neighbourhood configuration.}

\item{se.covmat}{a list of length \eqn{n}, the \eqn{i}th list
component contains a matrix with the standard errors of the
approximated block-block covariances of the \eqn{i}th polygon
neighbourhood configuration. Values are equal to \code{NaN} for
argument \code{napp = 1}, see \emph{Details}.}

\item{pixconfig}{a list of lists of length \eqn{n}, the \eqn{i}th list
component contains a list with the information about the pixels used
for the covariance approximation of the \eqn{i}th polygon
neighbourhood configuration.  The components of \code{pixconfig} are
described below.}

\item{pixcovmat}{a matrix, \code{dim(matrix)} =
(\eqn{n_{\mathrm{pix}}}{n_pix}, \eqn{n_{\mathrm{pix}}}{n_pix} ), with
the covariance matrix of the pixels.}

\item{model}{an object of class \dQuote{\code{covmodel}} with the
parameters of the used covariance function.}

\item{data}{a data frame which is the \code{data} slot of the
\code{SpatialPolygonsDataFrame} object.  This data frame is used to
build the design matrix of the prediction polygons by the
\code{\link{CKrige}} function.  \code{data} is empty with
\code{dim(data)} = (0, 0) if \code{newdata} is an object of class
\dQuote{\code{SpatialPolygons}}.}

\item{polygons}{a \code{SpatialPolygons} object.  A list of length
\eqn{n} with the polygons of the \code{newdata} object.}


The \eqn{i}th component of \code{pixconfig} is a list with the
following 10 components:

\item{pixcenter}{a matrix with \code{dim(pixcenter)} =
(\eqn{n_{\mathrm{pix}}}{n_pix}, 2) with the coordinates of the pixels
centroids for the \eqn{i}th polygon neighbourhood configuration.}

\item{rowwidth}{\code{preCKrige} input argument \code{pheight}.}

\item{colwidth}{\code{preCKrige} input argument \code{pwidth}.}

\item{nrows}{a numeric scalar with number of rows
\eqn{r_{\mathrm{pix}}}{r_pix} of the pixel grid.}

\item{ncols}{a numeric scalar with number of columns
\eqn{c_{\mathrm{pix}}}{c_pix} of the pixel grid.}

\item{no.pix.in.poly}{a numeric vector of length \eqn{m_i}, each
number indicates by how many pixels a polygon of the \eqn{i}th polygon
configuration is approximated.}

\item{sa.polygons}{a logical vector of length \eqn{m_i}, TRUE means
that the \eqn{i}th polygon is treated as a point because its area is
smaller than the area of a pixel, and FALSE means that the polygon is
approximated by pixels, see \emph{Note} for more details.}

\item{polygon.centroids}{a matrix with \code{dim(polygon.centroids)} =
(\eqn{m_i}, 2) with the coordinates of the polygon centroids of the
\eqn{i}th polygon neighbourhood configuration.}

\item{posindex}{an integer vector of length \eqn{m_i} with indices of
the \eqn{i}th polygon and its neighbours as defined by the argument
\code{neighbours}.}

\item{pix.in.poly}{is a binary matrix with \code{dim(pix.in.poly)} =
(\eqn{n_{\mathrm{pix}}}{n_pix}, \eqn{m_i}).  \code{pix.in.poly[k, j] = 1}
indicates that the centroid of the \var{k}th pixel lies in the
\var{j}th polygon, and \code{pix.in.poly[k, j] = 0} indicates that the
\var{k}th pixel centroid does not lie in the \var{j}th
polygon.} }

% end value

\references{
Hofer, C. and Papritz, A. (2011).  constrainedKriging: an R-package
for customary, constrained and covariance-matching constrained point
or block Kriging.  \emph{Computers & Geosciences}. \bold{37}, 1562--1569,
\doi{10.1016/j.cageo.2011.02.009}
}


\author{
Christoph Hofer, \email{christoph.hofer@alumni.ethz.ch}
}

\seealso{ \code{\link{CKrige}} }

\examples{
\donttest{
### first example
### load data
data(meuse, package = "sp")
data(meuse.blocks)

### plot blocks
plot(meuse.blocks)

### compute the approximated block variance of each block in meuse.blocks
### without any neighbouring blocks (default, required for in universal
### and constrained Kriging) for an exponential covariance function without
### a measurement error, a nugget  = 0.15 (micro scale white noise process),
### a partial sill variance = 0.15 and a scale parameter = 192.5
### approximation of block variance by pixel of size 75m x 75m
preCK_1 <- preCKrige(newdata = meuse.blocks, model = covmodel(modelname =
    "exponential", mev = 0, nugget = 0.05, variance = 0.15,
    scale = 192.5), pwidth = 75, pheight = 75)

### plot block approximation for block 59
plot(preCK_1, 59)

### second example
### define neighbours by using the poly2nb function
### of the spdep package
if(!requireNamespace("spdep", quietly = TRUE)){
  stop("install package spdep to run example")
}
neighbours <- spdep::poly2nb(meuse.blocks)
class(neighbours)
### neighbours should be an object of class "list"
class(neighbours) <- "list"
### compute the approximated block variance-covariance
### matrices of each block in meuse.blocks without the
### defined block neighbours
preCK_2 <- preCKrige(newdata = meuse.blocks, neighbours = neighbours,
  model = covmodel("exponential", nugget = 0.05, variance = 0.15,
    scale = 192.5), pwidth = 75, pheight = 75)

### plot block approximation of block 59 and its
### block neighbours
plot(preCK_2, 59)
}
}

\keyword{methods}
