% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/clubSandwich.R
\name{vcovCR}
\alias{vcovCR}
\alias{vcovCR.default}
\title{Cluster-robust variance-covariance matrix}
\usage{
vcovCR(obj, cluster, type, target, inverse_var, form, ...)

\method{vcovCR}{default}(obj, cluster, type, target = NULL,
  inverse_var = FALSE, form = "sandwich", ...)
}
\arguments{
\item{obj}{Fitted model for which to calculate the variance-covariance matrix}

\item{cluster}{Expression or vector indicating which observations belong to 
the same cluster. For some classes, the cluster will be detected 
automatically if not specified.}

\item{type}{Character string specifying which small-sample adjustment should 
be used, with available options \code{"CR0"}, \code{"CR1"}, \code{"CR1p"},
\code{"CR1S"}, \code{"CR2"}, or \code{"CR3"}. See "Details" section of 
\code{\link{vcovCR}} for further information.}

\item{target}{Optional matrix or vector describing the working 
variance-covariance model used to calculate the \code{CR2} and \code{CR4} 
adjustment matrices. If a vector, the target matrix is assumed to be 
diagonal. If not specified, \code{vcovCR} will attempt to infer a value.}

\item{inverse_var}{Optional logical indicating whether the weights used in 
fitting the model are inverse-variance. If not specified, \code{vcovCR} 
will attempt to infer a value.}

\item{form}{Controls the form of the returned matrix. The default 
\code{"sandwich"} will return the sandwich variance-covariance matrix. 
Alternately, setting \code{form = "meat"} will return only the meat of the 
sandwich and setting \code{form = B}, where \code{B} is a matrix of 
appropriate dimension, will return the sandwich variance-covariance matrix 
calculated using \code{B} as the bread.}

\item{...}{Additional arguments available for some classes of objects.}
}
\value{
An object of class \code{c("vcovCR","clubSandwich")}, which consists 
  of a matrix of the estimated variance of and covariances between the 
  regression coefficient estimates. The matrix has several attributes: 
  \describe{ \item{type}{indicates which small-sample adjustment was used} 
  \item{cluster}{contains the factor vector that defines independent 
  clusters} \item{bread}{contains the bread matrix} \item{v_scale}{constant 
  used in scaling the sandwich estimator} \item{est_mats}{contains a list of 
  estimating matrices used to calculate the sandwich estimator} 
  \item{adjustments}{contains a list of adjustment matrices used to calculate
  the sandwich estimator} \item{target}{contains the working 
  variance-covariance model used to calculate the adjustment matrices. This 
  is needed for calculating small-sample corrections for Wald tests.} }
}
\description{
This is a generic function, with specific methods defined for 
  \code{\link[stats]{lm}}, \code{\link[plm]{plm}}, \code{\link[stats]{glm}}, 
  \code{\link[nlme]{gls}}, \code{\link[nlme]{lme}},
  \code{\link[robumeta]{robu}}, \code{\link[metafor]{rma.uni}}, and
  \code{\link[metafor]{rma.mv}} objects.

\code{vcovCR} returns a sandwich estimate of the variance-covariance matrix 
of a set of regression coefficient estimates.
}
\details{
\code{vcovCR} returns a sandwich estimate of the variance-covariance matrix 
of a set of regression coefficient estimates.

Several different small sample corrections are available, which run 
  parallel with the "HC" corrections for heteroskedasticity-consistent 
  variance estimators, as implemented in \code{\link[sandwich]{vcovHC}}. The
  "CR2" adjustment is recommended (Pustejovsky & Tipton, 2017; Imbens &
  Kolesar, 2016). See Pustejovsky and Tipton (2017) and Cameron and Miller
  (2015) for further technical details. Available options include: \describe{
  \item{"CR0"}{is the original form of the sandwich estimator (Liang & Zeger,
  1986), which does not make any small-sample correction.} 
  \item{"CR1"}{multiplies CR0 by \code{m / (m - 1)}, where \code{m} is the 
  number of clusters.} 
  \item{"CR1p"}{multiplies CR0 by \code{m / (m - p)}, where \code{m} is the 
  number of clusters and \code{p} is the number of covariates.} 
  \item{"CR1S"}{multiplies CR0 by \code{(m (N-1)) / [(m - 
  1)(N - p)]}, where \code{m} is the number of clusters, \code{N} is the 
  total number of observations, and \code{p} is the number of covariates. 
  Some Stata commands use this correction by default.} 
  \item{"CR2"}{is the
  "bias-reduced linearization" adjustment proposed by Bell and McCaffrey
  (2002) and further developed in Pustejovsky and Tipton (2017). The
  adjustment is chosen so that the variance-covariance estimator is exactly
  unbiased under a user-specified working model.} 
  \item{"CR3"}{approximates the leave-one-cluster-out jackknife variance estimator (Bell & McCaffrey,
  2002).} }
}
\examples{

# simulate design with cluster-dependence
m <- 8
cluster <- factor(rep(LETTERS[1:m], 3 + rpois(m, 5)))
n <- length(cluster)
X <- matrix(rnorm(3 * n), n, 3)
nu <- rnorm(m)[cluster]
e <- rnorm(n)
y <- X \%*\% c(.4, .3, -.3) + nu + e
dat <- data.frame(y, X, cluster, row = 1:n)

# fit linear model
lm_fit <- lm(y ~ X1 + X2 + X3, data = dat)
vcov(lm_fit)

# cluster-robust variance estimator with CR2 small-sample correction
vcovCR(lm_fit, cluster = dat$cluster, type = "CR2")

# compare small-sample adjustments
CR_types <- paste0("CR",c("0","1","1S","2","3"))
sapply(CR_types, function(type) 
       sqrt(diag(vcovCR(lm_fit, cluster = dat$cluster, type = type))))

}
\references{
Bell, R. M., & McCaffrey, D. F. (2002). Bias reduction in
standard errors for linear regression with multi-stage samples. Survey
Methodology, 28(2), 169-181.

Cameron, A. C., & Miller, D. L. (2015). A Practitioner's Guide to
Cluster-Robust Inference. \emph{Journal of Human Resources, 50}(2), 317-372.
\doi{10.3368/jhr.50.2.317}

Imbens, G. W., & Kolesar, M. (2016). Robust standard errors in small samples:
Some practical advice. \emph{Review of Economics and Statistics, 98}(4),
701-712. \doi{10.1162/rest_a_00552}

Liang, K.-Y., & Zeger, S. L. (1986). Longitudinal data analysis using
generalized linear models. \emph{Biometrika, 73}(1), 13-22.
\doi{10.1093/biomet/73.1.13}

Pustejovsky, J. E. & Tipton, E. (2017). Small sample methods for
cluster-robust variance estimation and hypothesis testing in fixed effects
models. \emph{Journal of Business and Economic Statistics}. In Press.
\doi{10.1080/07350015.2016.1247004}
}
\seealso{
\code{\link{vcovCR.lm}}, \code{\link{vcovCR.plm}}, 
  \code{\link{vcovCR.glm}}, \code{\link{vcovCR.gls}},
  \code{\link{vcovCR.lme}}, \code{\link{vcovCR.robu}},
  \code{\link{vcovCR.rma.uni}}, \code{\link{vcovCR.rma.mv}}
}
