\name{seas.sum}
\alias{seas.sum}
\title{Seasonal sum data object}
\description{
  Create a seasonal sum object used for analysis of precipitation data (among other things, such as recharge rates).
}
\synopsis{
seas.sum(dat, start, end, width = 11, param, prime, a.cut = 0.3, 
     na.cut = 0.2, unit = "mm", id, name)
}
\usage{
# minimum usage
seas.sum(dat)

# all options
seas.sum(dat, start, end, width = 11, param, prime, a.cut = 0.3, 
     na.cut = 0.2, unit = "mm", id, name)
}
\arguments{
  \item{dat}{time-varying \code{data.frame} with parameters which are summed, such as precipitation}
  \item{start}{start year; if omitted minimum year will be used}
  \item{end}{end year; if omitted will use same as \code{start}, and if \code{start} is omitted, will use maximum year}
  \item{width}{a number specifying the width of the bin (factor) in days, or \code{"mon"} for Months (see \code{\link{mkfact}} for others)}
  \item{param}{a \code{\link{character}} array of the \code{\link{names}} from \code{dat} that are to be summed, such as \code{c("rain","snow","precip")}, if available}
  \item{prime}{a single parameter from \code{param} which is the prime variable of interest, such as \code{"precip"}; this is the parameter used for comparison with \code{a.cut} and \code{na.cut} in the resulting \code{active} and \code{na} dimensions}
  \item{a.cut}{cut-off value for the day to be considered an \emph{active} or \sQuote{wet day} (based on the \code{prime} parameter); a trace amount of 0.3 mm is suggested}
  \item{na.cut}{cut-off fraction of missing values; can be single value or a vector for \code{c(annual,seasonal)}; details given below}
  \item{unit}{unit for \code{seas.sum} object; useful for future plotting}
  \item{id}{unique station identifier used to extract a subset of data from \code{dat}}
  \item{name}{provide a name for the seasonal sum object; otherwise generated from \code{id} (if available)}
}
\details{
  This function is used to discretize and sum time-varying data in a \code{\link{data.frame}} for analysis in \emph{seasonal} and \emph{annual} parts. This is particularly useful for calculating normals of rates, such as precipitation and recharge. This function simply sums up each parameter in each bin for each year, and provides the results in several multi-dimensional arrays.

  Sums are \emph{not} normalized, and represent a sum for the number of days in the bin (seasonal data) or year (for annual data). Seasonal data can be normalized by the number of days (for a rate per day) or by the number of active days where \code{prime > a.cut}. Both arrays of these numbers of days are provided for normalizing.
  
  For annual sums, years with \emph{many} missing values are ignored (receiving a value of \code{NA}) since it has insufficient data for a complete sum. The amount of allowable \code{NA} values is controlled by \code{na.cut}, which is a fraction of \code{NA} values for the whole year (default is 0.2).
  
  The seasonal sums are calculated independently from the annual sums. Individual bins from each year with \emph{many} missing values are ignored, where the amount of allowable \code{NA} values is controlled by \code{na.cut} (fraction of \code{NA}s in each bin of each individual year; default is 0.2).
}
\value{
  Returns a \code{seas.sum} object, which is a \code{\link{list}} with the following properties:
  \item{ann}{
    \code{\link{array}} of annual data; the dimensions are:
    \item{\code{[[1]]}}{years}
    \item{\code{[[2]]}}{annual sums of parameters (\code{param}) (if original units were mm/day, they are now mm/year); also the total number of active days (\code{active}), existing days (\code{days}), and missing days (\code{na}) in the year.}
  }
  \item{seas}{
    \code{\link{array}} of seasonal data; the dimensions are:
    \item{\code{[[1]]}}{years}
    \item{\code{[[2]]}}{bins, or seasonal factors generated by \code{\link{mkfact}}}
    \item{\code{[[3]]}}{sums of \code{param}, \code{active}, \code{days}, and \code{na} in each bin of each year.}
  }
  \item{call}{function \code{\link{call}}}
  \item{years}{years (same as \code{ann[[1]]} and \code{seas[[1]]})}
  \item{param}{parameters which the sums represent (part of \code{ann[[2]]} and \code{seas[[3]]})}
  \item{prime}{\code{prime} parameter}
  \item{unit}{unit of parameter}
  \item{width}{\code{width} argument passed to \code{\link{mkfact}}
  \item{bins}{names of bins returned by \code{\link{mkfact}} (same as \code{seas[[2]]})}}
  \item{precip.only}{value used in argument (modified if insufficient data found in \code{dat})}
  \item{na.cut}{value used in argument}
  \item{a.cut}{value used in argument}
  \item{id}{value used in argument; or found in \code{dat$id} if present and not given as argument}
  \item{name}{either given by \code{name} argument or generated from \code{id} or from \code{dat} object name}
}
\author{M.W. Toews}
\seealso{\code{\link{image.seas.sum}} to view the result result, \code{\link{seas.norm}} to view a \dQuote{normal}}
\examples{
loc <- Sys.getlocale()
on.exit(Sys.setlocale(locale=loc))
Sys.setlocale(loc="C")
data(mscstn)
data(mscdata)

dat.ss <- seas.sum(mscdata,id=1108447,width="mon")

# Annual data
dat.ss$ann

# Demonstrate how to slice through a cubic array
dat.ss$seas["1990",,]
dat.ss$seas[,"Feb",] # month names are locale specific
dat.ss$seas[,,"precip"]

# Simple calculation on an array
(monthly.mean <- apply(dat.ss$seas[,,"precip"],2,mean,na.rm=TRUE))
barplot(monthly.mean,ylab="Mean monthly total (mm/month)",
	main="Un-normalized mean precipitation in Vancouver, BC")
text(6.5,150,paste("Un-normalized rates given 'per month' should be",
	"avoided since ~3-9\% error is introduced",
	"to the analysis between months",sep="\n"))

# Normalized precip
norm.monthly <- dat.ss$seas[,,"precip"]/dat.ss$seas[,,"days"]
norm.monthly.mean <- apply(norm.monthly,2,mean,na.rm=TRUE)
print(round(norm.monthly,2))
print(round(norm.monthly.mean,2))
barplot(norm.monthly.mean,ylab="Normalized mean monthly total (mm/day)",
	main="Normalized mean precipitation in Vancouver, BC")

# Better graphics of data
dat.ss <- seas.sum(mscdata,id=1108447)
image(dat.ss)
}
\keyword{datagen}
