% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/morrisCI.r
\name{morrisCI}
\alias{morrisCI}
\title{Analytical ordered-binary-Y confidence intervals, using the Morris (1988) algorithm}
\usage{
morrisCI(
  y,
  n,
  phat = y/n,
  conf = 0.9,
  narrower = TRUE,
  alternate = wilsonCI,
  ...
)
}
\arguments{
\item{y}{integer or numeric vector, the pointwise Binomial counts}

\item{n}{integer or numeric vector, the pointwise sample sizes}

\item{phat}{numeric vector, the point estimates. Defaults to \code{y/n}, but when called by \code{\link{isotInterval}} is overridden by the actual CIR/IR point estimate.}

\item{conf}{numeric, the interval's confidence level as a fraction in (0,1). Default 0.9.}

\item{narrower}{logical, if the \code{alternate}-produced interval is narrower at any point, should it replace the Morris result? Also, can we enforce straightforward monotonocity to narrow the bounds? Default \code{TRUE}.}

\item{alternate}{function to use for alternate pointwise interval. Default \code{wilconCI}.}

\item{...}{parameters passed on to \code{alternate}.}
}
\value{
A two-column matrix with the same number of rows as \code{length(phat)}, containing the calculated lower and upper bounds, respectively.
}
\description{
Analytical confidence intervals for CIR and IR, using the recursive algorithm by Morris (1988), equation (4.3), for ordered-binary-Y point estimates. Optionally, the intervals are narrowed further using a backup (unordered) interval estimate at each individual x value.
}
\details{
The default for backup is Wilson's (\code{wilconCI}). Also available are Jeffrys' (\code{jeffCI}) and Agresti-Coull (\code{agcouCI}).
}
\note{
This function found and corrected a typo in equation (4.3), namely the use of G_(j+1) in the recursion. The recursion cannot start in this way. Rather, it is the use of theta_(j+1) that delivers information from adjacent doses. Or perhaps in other words, there is only one G function rather than a different one for each dose. The correction has been verified by reproducing the numbers in the Morris (1988) example (Table 1), and also approved by the original author.
}
\examples{
# Interesting run (#664) from a simulated up-and-down ensemble:
# (x will be auto-generated as dose levels 1:5)
dat=doseResponse(y=c(1/7,1/8,1/2,1/4,4/17),wt=c(7,24,20,12,17))
# The experiment's goal is to find the 30th percentile
slow1=cirPAVA(dat,full=TRUE)
# Default interval (Morris+Wilson); same as you get by directly calling 'quickIsotone'
int1=isotInterval(slow1)
# Morris without Wilson; the 'narrower=FALSE' argument is passed on to 'morrisCI'
int1_0=isotInterval(slow1,narrower=FALSE)
# Wilson without Morris
int2=isotInterval(slow1,intfun=wilsonCI)
# Agresti=Coull (the often-used "plus 2")
int3=isotInterval(slow1,intfun=agcouCI)
# Jeffrys (Bayesian-inspired) is also available
int4=isotInterval(slow1,intfun=jeffCI)

### Showing the data and the intervals
par(mar=c(3,3,4,1),mgp=c(2,.5,0),tcl=-0.25)
plot(dat,ylim=c(0,0.65),refsize=4,las=1,main="Forward-Estimation CIs") # uses plot.doseResponse()

# The true response function; true target is where it crosses the y=0.3 line
lines(seq(0,7,0.1),pweibull(seq(0,7,0.1),shape=1.1615,scale=8.4839),col=4)

lines(int1$ciLow,lty=2,col=2,lwd=2) 
lines(int1$ciHigh,lty=2,col=2,lwd=2) 

lines(int1_0$ciLow,lty=2) 
lines(int1_0$ciHigh,lty=2) 

lines(int2$ciLow,lty=2,col=3) 
lines(int2$ciHigh,lty=2,col=3) 
# Plotting the remaining 2 is skipped, as they are very similar to Wilson.

# Note how the default (red) boundaries take the tighter of the two options everywhere, 
# except for one place (dose 1 upper bound) where they go even tighter thanks to monotonicity 
# enforcement. This can often happen when sample size is uneven; since bounds tend to be 
# conservative it is rather safe to do.

legend('topleft',pch=c(NA,'X',NA,NA,NA),lty=c(1,NA,2,2,2),col=c(4,1,2,1,3),lwd=c(1,1,2,1,1),legend
=c('True Curve','Observations','Morris+Wilson (default)','Morris only','Wilson only'),bty='n')

}
\references{
Morris, M., 1988. Small-sample confidence limits for parameters under inequality constraints with application to quantal bioassay. Biometrics 44, 1083-1092.
}
\seealso{
\code{\link{isotInterval}}
}
\author{
Assaf P. Oron \code{<assaf.oron.at.gmail.com>}
}
