% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cgalMesh.R
\name{cgalMesh}
\alias{cgalMesh}
\title{R6 class to represent a CGAL mesh}
\description{
R6 class to represent a CGAL mesh.
}
\examples{

## ------------------------------------------------
## Method `cgalMesh$new`
## ------------------------------------------------

library(cgalMeshes)
meshFile <- system.file(
  "extdata", "bigPolyhedron.off", package = "cgalMeshes"
)
mesh <- cgalMesh$new(meshFile)
rglmesh <- mesh$getMesh(normals = FALSE)
\donttest{library(rgl)
open3d(windowRect = 50 + c(0, 0, 512, 512), zoom = 0.9)
shade3d(rglmesh, color = "tomato")
plotEdges(
  mesh$vertices(), mesh$edges(), color = "darkred"
)}

## ------------------------------------------------
## Method `cgalMesh$area`
## ------------------------------------------------

library(rgl)
mesh <- cgalMesh$new(cube3d())$triangulate()
mesh$area()

## ------------------------------------------------
## Method `cgalMesh$boundsVolume`
## ------------------------------------------------

library(rgl)
mesh <- cgalMesh$new(tetrahedron3d())
mesh$boundsVolume() # TRUE
mesh$reverseOrientation()
mesh$boundsVolume() # TRUE

## ------------------------------------------------
## Method `cgalMesh$centroid`
## ------------------------------------------------

\donttest{library(cgalMeshes)
library(rgl)
mesh <- cgalMesh$new(icosahedron3d())
mesh$centroid()}

## ------------------------------------------------
## Method `cgalMesh$clip`
## ------------------------------------------------

# cube clipped to sphere ####
library(cgalMeshes)
library(rgl)
mesh    <- cgalMesh$new(cube3d())$triangulate()
clipper <- cgalMesh$new(sphereMesh(r= sqrt(2)))
mesh$clip(clipper, clipVolume = TRUE)
rglmesh <- mesh$getMesh(normals = FALSE)
\donttest{open3d(windowRect = 50 + c(0, 0, 512, 512))
view3d(45, 45, zoom = 0.9)
shade3d(rglmesh, col = "darkorange")}

# Togliatti surface clipped to a ball ####
\donttest{library(rmarchingcubes)
library(rgl)
library(cgalMeshes)
# Togliatti surface equation: f(x,y,z) = 0
f <- function(x, y, z) {
  64*(x-1) *
    (x^4 - 4*x^3 - 10*x^2*y^2 - 4*x^2 + 16*x - 20*x*y^2 + 5*y^4 + 16 - 20*y^2) - 
    5*sqrt(5-sqrt(5))*(2*z - sqrt(5-sqrt(5))) * 
    (4*(x^2 + y^2 - z^2) + (1 + 3*sqrt(5)))^2
}
# grid
n <- 200L
x <- y <- seq(-5, 5, length.out = n)
z <- seq(-4, 4, length.out = n)
Grid <- expand.grid(X = x, Y = y, Z = z)
# calculate voxel
voxel <- array(with(Grid, f(X, Y, Z)), dim = c(n, n, n))
# calculate isosurface
contour_shape <- contour3d(
  griddata = voxel, level = 0, x = x, y = y, z = z
)
# make rgl mesh (plotted later)
rglMesh <- tmesh3d(
  vertices = t(contour_shape[["vertices"]]),
  indices  = t(contour_shape[["triangles"]]),
  normals  = contour_shape[["normals"]],
  homogeneous = FALSE
)
# make CGAL mesh
mesh <- cgalMesh$new(rglMesh)
# clip to sphere of radius 4.8
sphere <- sphereMesh(r = 4.8)
clipper <- cgalMesh$new(sphere)
mesh$clip(clipper, clipVolume = FALSE)
rglClippedMesh <- mesh$getMesh()
# plot
open3d(windowRect = 50 + c(0, 0, 900, 450))
mfrow3d(1L, 2L)
view3d(0, -70, zoom = 0.8)
shade3d(rglMesh, color = "firebrick")
next3d()
view3d(0, -70, zoom = 0.8)
shade3d(rglClippedMesh, color = "firebrick")
shade3d(sphere, color = "yellow", alpha = 0.15)}

## ------------------------------------------------
## Method `cgalMesh$connectedComponents`
## ------------------------------------------------

\donttest{library(cgalMeshes)
library(rmarchingcubes)
# isosurface function (slice of a seven-dimensional toratope)
f <- function(x, y, z, a) {
  (sqrt(
    (sqrt((sqrt((x*sin(a))^2 + (z*cos(a))^2) - 5)^2 + (y*sin(a))^2) - 2.5)^2 + 
      (x*cos(a))^2) - 1.25
  )^2 + (sqrt((sqrt((z*sin(a))^2 + (y*cos(a))^2) - 2.5)^2) - 1.25)^2
}
# make grid
n <- 200L
x <- seq(-10, 10, len = n)
y <- seq(-10, 10, len = n)
z <- seq(-10, 10, len = n)
Grid <- expand.grid(X = x, Y = y, Z = z)
# compute isosurface
voxel <- array(with(Grid, f(X, Y, Z, a = pi/2)), dim = c(n, n, n))
isosurface <- contour3d(voxel, level = 0.25, x = x, y = y, z = z)
# make CGAL mesh
mesh <- cgalMesh$new(
  vertices = isosurface[["vertices"]], faces = isosurface[["triangles"]]
)
# connected components
components <- mesh$connectedComponents()
ncc <- length(components)
# plot
library(rgl)
colors <- rainbow(ncc)
open3d(windowRect = 50 + c(0, 0, 512, 512))
view3d(30, 50)
for(i in 1L:ncc) {
  rglMesh <- components[[i]]$getMesh()
  shade3d(rglMesh, color = colors[i])
}}

## ------------------------------------------------
## Method `cgalMesh$convexParts`
## ------------------------------------------------

\donttest{library(cgalMeshes)
library(rgl)
mesh <- cgalMesh$new(pentagrammicPrism)$triangulate()
cxparts <- mesh$convexParts()
ncxparts <- length(cxparts)
colors <- hcl.colors(ncxparts, palette = "plasma")
open3d(windowRect = 50 + c(0, 0, 512, 512))
view3d(20, -20, zoom = 0.8)
for(i in 1L:ncxparts) {
  cxmesh <- cxparts[[i]]$getMesh(normals = FALSE)
  shade3d(cxmesh, color = colors[i])
}}

## ------------------------------------------------
## Method `cgalMesh$copy`
## ------------------------------------------------

library(rgl)
mesh <- cgalMesh$new(cube3d())
tmesh <- mesh$copy()$triangulate()
tmesh$isTriangle() # TRUE
mesh$isTriangle() # FALSE

## ------------------------------------------------
## Method `cgalMesh$distance`
## ------------------------------------------------

# cube example ####
library(cgalMeshes)
mesh <- cgalMesh$new(rgl::cube3d())$triangulate()
points <- rbind(
  c(0, 0, 0),
  c(1, 1, 1)
)
mesh$distance(points) # should be 1 and 0

# cyclide example ####
\donttest{library(cgalMeshes)
a <- 100; c <- 30; mu <- 80
mesh <- cgalMesh$new(cyclideMesh(a, c, mu, nu = 100L, nv = 100L))
O2 <- c(c, 0, 0)
# should be a - mu = 20 (see ?cyclideMesh):
mesh$distance(O2)}    

## ------------------------------------------------
## Method `cgalMesh$edges`
## ------------------------------------------------

library(rgl)
mesh <- cgalMesh$new(dodecahedron3d())
head(mesh$edges())

## ------------------------------------------------
## Method `cgalMesh$fair`
## ------------------------------------------------

library(cgalMeshes)
rglHopf <- HopfTorusMesh(nu = 100, nv = 100)
hopf <- cgalMesh$new(rglHopf)
# squared norms of the vertices
normsq <- apply(hopf$vertices(), 1L, crossprod)
# fair the region where the squared norm is > 19
indices <- which(normsq > 19)
hopf$fair(indices)
rglHopf_faired <- hopf$getMesh()
# plot
\donttest{library(rgl)
open3d(windowRect = 50 + c(0, 0, 900, 450))
mfrow3d(1L, 2L)
view3d(0, 0, zoom = 0.8)
shade3d(rglHopf, color = "orangered")
next3d()
view3d(0, 0, zoom = 0.8)
shade3d(rglHopf_faired, color = "orangered")}

## ------------------------------------------------
## Method `cgalMesh$geoDists`
## ------------------------------------------------

\donttest{# torus ####
library(cgalMeshes)
library(rgl)
rglmesh <- torusMesh(R = 3, r = 2, nu = 90, nv = 60)
mesh <- cgalMesh$new(rglmesh)
# estimated geodesic distances
geodists <- mesh$geoDists(1L)
# normalization to (0, 1)
geodists <- geodists / max(geodists)
# color each vertex according to its geodesic distance from the source
fcolor <- colorRamp(viridisLite::turbo(200L))
colors <- fcolor(geodists)
colors <- rgb(colors[, 1L], colors[, 2L], colors[, 3L], maxColorValue = 255)
rglmesh[["material"]] <- list("color" = colors)
# plot
open3d(windowRect = 50 + c(0, 0, 512, 512), zoom = 0.8)
shade3d(rglmesh)
wire3d(rglmesh, color = "black")
if(!rgl.useNULL()) {
  play3d(spin3d(axis = c(1, 1, 1), rpm = 5), duration = 20)  
}}

# a trefoil knot (taken from `?rgl::cylinder3d`) ####
\donttest{library(cgalMeshes)
library(rgl)
theta <- seq(0, 2*pi, length.out = 50L)
knot <- cylinder3d(
  center = cbind(
    sin(theta) + 2*sin(2*theta), 
    2*sin(3*theta), 
    cos(theta) - 2*cos(2*theta)),
  e1 = cbind(
    cos(theta) + 4*cos(2*theta), 
    6*cos(3*theta), 
    sin(theta) + 4*sin(2*theta)),
  radius = 0.8, 
  closed = TRUE)
knot <- subdivision3d(knot, depth = 2)
mesh <- cgalMesh$new(knot)$triangulate()
rglmesh <- mesh$getMesh()
# estimated geodesic distances
geodists <- mesh$geoDists(1L)
# normalization to (0, 1)
geodists <- geodists / max(geodists)
# color each vertex according to its geodesic distance from the source
fcolor <- colorRamp(viridisLite::inferno(200L))
colors <- fcolor(geodists)
colors <- rgb(colors[, 1L], colors[, 2L], colors[, 3L], maxColorValue = 255)
rglmesh[["material"]] <- list("color" = colors)
# plot
open3d(windowRect = 50 + c(0, 0, 512, 512), zoom = 0.8)
shade3d(rglmesh)
if(!rgl.useNULL()) {
  play3d(spin3d(axis = c(1, 1, 0), rpm = 5), duration = 20)  
}}

## ------------------------------------------------
## Method `cgalMesh$getMesh`
## ------------------------------------------------

library(rgl)
mesh <- cgalMesh$new(cube3d())$triangulate()
mesh$getMesh(normals = FALSE)

## ------------------------------------------------
## Method `cgalMesh$intersection`
## ------------------------------------------------

\donttest{library(cgalMeshes)
library(rgl)
# take two cubes
rglmesh1 <- cube3d()
rglmesh2 <- translate3d(cube3d(), 1, 1, 1)
mesh1 <- cgalMesh$new(rglmesh1)
mesh2 <- cgalMesh$new(rglmesh2)
# the two meshes must be triangle
mesh1$triangulate()
mesh2$triangulate()
# intersection
imesh <- mesh1$intersection(mesh2)
rglimesh <- imesh$getMesh(normals = FALSE)
# extract edges for plotting
extEdges <- exteriorEdges(imesh$edges())
# plot
open3d(windowRect = 50 + c(0, 0, 512, 512), zoom = 0.9)
shade3d(rglimesh, color = "red")
plotEdges(imesh$vertices(), extEdges)
shade3d(rglmesh1, color = "yellow", alpha = 0.2)
shade3d(rglmesh2, color = "cyan", alpha = 0.2)}

## ------------------------------------------------
## Method `cgalMesh$isOutwardOriented`
## ------------------------------------------------

library(rgl)
mesh <- cgalMesh$new(tetrahedron3d())
mesh$isOutwardOriented() # TRUE
mesh$reverseOrientation()
mesh$isOutwardOriented() # FALSE

## ------------------------------------------------
## Method `cgalMesh$isTriangle`
## ------------------------------------------------

library(rgl)
mesh <- cgalMesh$new(cube3d())
mesh$isTriangle()

## ------------------------------------------------
## Method `cgalMesh$orientToBoundVolume`
## ------------------------------------------------

# two disjoint tetrahedra ####
vertices <- rbind(
  c(0, 0, 0),
  c(2, 2, 0),
  c(2, 0, 2),
  c(0, 2, 2),
  c(3, 3, 3),
  c(5, 5, 3),
  c(5, 3, 5),
  c(3, 5, 5)
)
faces <- rbind(
  c(3, 2, 1),
  c(3, 4, 2),
  c(1, 2, 4),
  c(4, 3, 1),
  c(5, 6, 7),
  c(6, 8, 7),
  c(8, 6, 5),
  c(5, 7, 8)
)
mesh <- cgalMesh$new(vertices = vertices, faces = faces)
mesh$boundsVolume() # FALSE
mesh$orientToBoundVolume()
mesh$boundsVolume() # TRUE

## ------------------------------------------------
## Method `cgalMesh$reverseOrientation`
## ------------------------------------------------

library(rgl)
mesh <- cgalMesh$new(tetrahedron3d())
mesh$isOutwardOriented() # TRUE
mesh$reverseOrientation()
mesh$isOutwardOriented() # FALSE

## ------------------------------------------------
## Method `cgalMesh$selfIntersects`
## ------------------------------------------------

library(rgl)
mesh <- cgalMesh$new(dodecahedron3d())
mesh$selfIntersects()

## ------------------------------------------------
## Method `cgalMesh$subtract`
## ------------------------------------------------

\donttest{library(cgalMeshes)
library(rgl)
# take two cubes
rglmesh1 <- cube3d()
rglmesh2 <- translate3d(cube3d(), 1, 1, 1)
mesh1 <- cgalMesh$new(rglmesh1)
mesh2 <- cgalMesh$new(rglmesh2)
# the two meshes must be triangle
mesh1$triangulate()
mesh2$triangulate()
# difference
mesh <- mesh1$subtract(mesh2)
rglmesh <- mesh$getMesh(normals = FALSE)
# extract edges for plotting
extEdges <- exteriorEdges(mesh$edges())
# plot
open3d(windowRect = 50 + c(0, 0, 512, 512), zoom = 0.9)
shade3d(rglmesh, color = "red")
plotEdges(mesh$vertices(), extEdges)
shade3d(rglmesh2, color = "cyan", alpha = 0.2)}

## ------------------------------------------------
## Method `cgalMesh$triangulate`
## ------------------------------------------------

library(rgl)
mesh <- cgalMesh$new(cube3d())
mesh$isTriangle() # FALSE
# warning: triangulating the mesh modifies it
mesh$triangulate()
mesh$isTriangle() # TRUE

## ------------------------------------------------
## Method `cgalMesh$union`
## ------------------------------------------------

\donttest{library(cgalMeshes)
library(rgl)
# take two cubes
rglmesh1 <- cube3d()
rglmesh2 <- translate3d(cube3d(), 1, 1, 1)
mesh1 <- cgalMesh$new(rglmesh1)
mesh2 <- cgalMesh$new(rglmesh2)
# the two meshes must be triangle
mesh1$triangulate()
mesh2$triangulate()
# union
umesh <- mesh1$union(mesh2)
rglumesh <- umesh$getMesh(normals = FALSE)
# extract edges for plotting
extEdges <- exteriorEdges(umesh$edges())
# plot
open3d(windowRect = 50 + c(0, 0, 512, 512), zoom = 0.9)
shade3d(rglumesh, color = "red")
plotEdges(umesh$vertices(), extEdges)}

## ------------------------------------------------
## Method `cgalMesh$volume`
## ------------------------------------------------

library(rgl)
mesh <- cgalMesh$new(cube3d())$triangulate()
mesh$volume()
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-cgalMesh-new}{\code{cgalMesh$new()}}
\item \href{#method-cgalMesh-print}{\code{cgalMesh$print()}}
\item \href{#method-cgalMesh-area}{\code{cgalMesh$area()}}
\item \href{#method-cgalMesh-boundsVolume}{\code{cgalMesh$boundsVolume()}}
\item \href{#method-cgalMesh-centroid}{\code{cgalMesh$centroid()}}
\item \href{#method-cgalMesh-clip}{\code{cgalMesh$clip()}}
\item \href{#method-cgalMesh-connectedComponents}{\code{cgalMesh$connectedComponents()}}
\item \href{#method-cgalMesh-convexParts}{\code{cgalMesh$convexParts()}}
\item \href{#method-cgalMesh-copy}{\code{cgalMesh$copy()}}
\item \href{#method-cgalMesh-distance}{\code{cgalMesh$distance()}}
\item \href{#method-cgalMesh-edges}{\code{cgalMesh$edges()}}
\item \href{#method-cgalMesh-fair}{\code{cgalMesh$fair()}}
\item \href{#method-cgalMesh-geoDists}{\code{cgalMesh$geoDists()}}
\item \href{#method-cgalMesh-getMesh}{\code{cgalMesh$getMesh()}}
\item \href{#method-cgalMesh-intersection}{\code{cgalMesh$intersection()}}
\item \href{#method-cgalMesh-isClosed}{\code{cgalMesh$isClosed()}}
\item \href{#method-cgalMesh-isOutwardOriented}{\code{cgalMesh$isOutwardOriented()}}
\item \href{#method-cgalMesh-isTriangle}{\code{cgalMesh$isTriangle()}}
\item \href{#method-cgalMesh-isValid}{\code{cgalMesh$isValid()}}
\item \href{#method-cgalMesh-orientToBoundVolume}{\code{cgalMesh$orientToBoundVolume()}}
\item \href{#method-cgalMesh-removeSelfIntersections}{\code{cgalMesh$removeSelfIntersections()}}
\item \href{#method-cgalMesh-reverseOrientation}{\code{cgalMesh$reverseOrientation()}}
\item \href{#method-cgalMesh-selfIntersects}{\code{cgalMesh$selfIntersects()}}
\item \href{#method-cgalMesh-subtract}{\code{cgalMesh$subtract()}}
\item \href{#method-cgalMesh-triangulate}{\code{cgalMesh$triangulate()}}
\item \href{#method-cgalMesh-union}{\code{cgalMesh$union()}}
\item \href{#method-cgalMesh-vertices}{\code{cgalMesh$vertices()}}
\item \href{#method-cgalMesh-volume}{\code{cgalMesh$volume()}}
\item \href{#method-cgalMesh-writeMeshFile}{\code{cgalMesh$writeMeshFile()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-new"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-new}{}}}
\subsection{Method \code{new()}}{
Creates a new \code{cgalMesh} object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$new(mesh, vertices, faces, clean = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{mesh}}{there are four possibilities for this argument: it can be 
missing, in which case the arguments \code{vertices} and \code{faces} 
must be given, or it can be the path to a mesh file (accepted formats: 
\code{off}, \code{obj}, \code{stl}, \code{ply}, \code{ts}, \code{vtp}),
or it can be a \strong{rgl} mesh (i.e. a \code{mesh3d} object), or it 
can be a list containing (at least) the fields \code{vertices} 
(numeric matrix with three columns) and \code{faces} (matrix of 
integers or list of vectors of integers)}

\item{\code{vertices}}{if \code{mesh} is missing, must be a numeric matrix with 
three columns}

\item{\code{faces}}{if \code{mesh} is missing, must be either a matrix of 
integers (each row gives the vertex indices of a face) or a list of 
vectors of integers (each one gives the vertex indices of a face)}

\item{\code{clean}}{Boolean, no effect if the mesh is given by a file, 
otherwise it indicates whether to clean the mesh (merge duplicated 
vertices and duplicated faces, remove isolated vertices); set to 
\code{FALSE} if you know your mesh is already clean}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{cgalMesh} object.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(cgalMeshes)
meshFile <- system.file(
  "extdata", "bigPolyhedron.off", package = "cgalMeshes"
)
mesh <- cgalMesh$new(meshFile)
rglmesh <- mesh$getMesh(normals = FALSE)
library(rgl)
open3d(windowRect = 50 + c(0, 0, 512, 512), zoom = 0.9)
shade3d(rglmesh, color = "tomato")
plotEdges(
  mesh$vertices(), mesh$edges(), color = "darkred"
)
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-print"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-print}{}}}
\subsection{Method \code{print()}}{
Print a \code{cgalMesh} object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$print(...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{ignored}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
No value returned, just prints some information about the mesh.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-area"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-area}{}}}
\subsection{Method \code{area()}}{
Compute the area of the mesh. The mesh must be triangle 
  and must not self-intersect.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$area()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A number, the mesh area.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(rgl)
mesh <- cgalMesh$new(cube3d())$triangulate()
mesh$area()
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-boundsVolume"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-boundsVolume}{}}}
\subsection{Method \code{boundsVolume()}}{
Check whether the mesh bounds a volume. The mesh must be 
  triangle.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$boundsVolume()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A Boolean value, whether the mesh bounds a volume.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(rgl)
mesh <- cgalMesh$new(tetrahedron3d())
mesh$boundsVolume() # TRUE
mesh$reverseOrientation()
mesh$boundsVolume() # TRUE
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-centroid"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-centroid}{}}}
\subsection{Method \code{centroid()}}{
Centroid of the mesh. The mesh must be triangle.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$centroid()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
The Cartesian coordinates of the centroid of the mesh.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(cgalMeshes)
library(rgl)
mesh <- cgalMesh$new(icosahedron3d())
mesh$centroid()
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-clip"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-clip}{}}}
\subsection{Method \code{clip()}}{
Clip mesh to the volume bounded by another mesh. 
  \strong{WARNING}: the reference mesh is then replaced by its 
  clipped version.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$clip(clipper, clipVolume)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{clipper}}{a \code{cgalMesh} object; it must represent a closed 
triangle mesh which doesn't self-intersect}

\item{\code{clipVolume}}{Boolean, whether the clipping has to be done on the 
volume bounded by the reference mesh rather than on its surface (i.e. 
the reference mesh will be kept closed if it is closed); if 
\code{TRUE}, the mesh to be clipped must not self-intersect}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The modified \code{cgalObject}.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# cube clipped to sphere ####
library(cgalMeshes)
library(rgl)
mesh    <- cgalMesh$new(cube3d())$triangulate()
clipper <- cgalMesh$new(sphereMesh(r= sqrt(2)))
mesh$clip(clipper, clipVolume = TRUE)
rglmesh <- mesh$getMesh(normals = FALSE)
open3d(windowRect = 50 + c(0, 0, 512, 512))
view3d(45, 45, zoom = 0.9)
shade3d(rglmesh, col = "darkorange")

# Togliatti surface clipped to a ball ####
library(rmarchingcubes)
library(rgl)
library(cgalMeshes)
# Togliatti surface equation: f(x,y,z) = 0
f <- function(x, y, z) {
  64*(x-1) *
    (x^4 - 4*x^3 - 10*x^2*y^2 - 4*x^2 + 16*x - 20*x*y^2 + 5*y^4 + 16 - 20*y^2) - 
    5*sqrt(5-sqrt(5))*(2*z - sqrt(5-sqrt(5))) * 
    (4*(x^2 + y^2 - z^2) + (1 + 3*sqrt(5)))^2
}
# grid
n <- 200L
x <- y <- seq(-5, 5, length.out = n)
z <- seq(-4, 4, length.out = n)
Grid <- expand.grid(X = x, Y = y, Z = z)
# calculate voxel
voxel <- array(with(Grid, f(X, Y, Z)), dim = c(n, n, n))
# calculate isosurface
contour_shape <- contour3d(
  griddata = voxel, level = 0, x = x, y = y, z = z
)
# make rgl mesh (plotted later)
rglMesh <- tmesh3d(
  vertices = t(contour_shape[["vertices"]]),
  indices  = t(contour_shape[["triangles"]]),
  normals  = contour_shape[["normals"]],
  homogeneous = FALSE
)
# make CGAL mesh
mesh <- cgalMesh$new(rglMesh)
# clip to sphere of radius 4.8
sphere <- sphereMesh(r = 4.8)
clipper <- cgalMesh$new(sphere)
mesh$clip(clipper, clipVolume = FALSE)
rglClippedMesh <- mesh$getMesh()
# plot
open3d(windowRect = 50 + c(0, 0, 900, 450))
mfrow3d(1L, 2L)
view3d(0, -70, zoom = 0.8)
shade3d(rglMesh, color = "firebrick")
next3d()
view3d(0, -70, zoom = 0.8)
shade3d(rglClippedMesh, color = "firebrick")
shade3d(sphere, color = "yellow", alpha = 0.15)
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-connectedComponents"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-connectedComponents}{}}}
\subsection{Method \code{connectedComponents()}}{
Decomposition into connected components.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$connectedComponents(triangulate = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{triangulate}}{Boolean, whether to triangulate the connected 
components.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A list of \code{cgalMesh} objects, one for each connected 
  component.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(cgalMeshes)
library(rmarchingcubes)
# isosurface function (slice of a seven-dimensional toratope)
f <- function(x, y, z, a) {
  (sqrt(
    (sqrt((sqrt((x*sin(a))^2 + (z*cos(a))^2) - 5)^2 + (y*sin(a))^2) - 2.5)^2 + 
      (x*cos(a))^2) - 1.25
  )^2 + (sqrt((sqrt((z*sin(a))^2 + (y*cos(a))^2) - 2.5)^2) - 1.25)^2
}
# make grid
n <- 200L
x <- seq(-10, 10, len = n)
y <- seq(-10, 10, len = n)
z <- seq(-10, 10, len = n)
Grid <- expand.grid(X = x, Y = y, Z = z)
# compute isosurface
voxel <- array(with(Grid, f(X, Y, Z, a = pi/2)), dim = c(n, n, n))
isosurface <- contour3d(voxel, level = 0.25, x = x, y = y, z = z)
# make CGAL mesh
mesh <- cgalMesh$new(
  vertices = isosurface[["vertices"]], faces = isosurface[["triangles"]]
)
# connected components
components <- mesh$connectedComponents()
ncc <- length(components)
# plot
library(rgl)
colors <- rainbow(ncc)
open3d(windowRect = 50 + c(0, 0, 512, 512))
view3d(30, 50)
for(i in 1L:ncc) {
  rglMesh <- components[[i]]$getMesh()
  shade3d(rglMesh, color = colors[i])
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-convexParts"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-convexParts}{}}}
\subsection{Method \code{convexParts()}}{
Decomposition into convex parts. The mesh must be triangle.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$convexParts(triangulate = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{triangulate}}{Boolean, whether to triangulate the convex parts}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A list of \code{cgalMesh} objects, one for each convex part.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(cgalMeshes)
library(rgl)
mesh <- cgalMesh$new(pentagrammicPrism)$triangulate()
cxparts <- mesh$convexParts()
ncxparts <- length(cxparts)
colors <- hcl.colors(ncxparts, palette = "plasma")
open3d(windowRect = 50 + c(0, 0, 512, 512))
view3d(20, -20, zoom = 0.8)
for(i in 1L:ncxparts) {
  cxmesh <- cxparts[[i]]$getMesh(normals = FALSE)
  shade3d(cxmesh, color = colors[i])
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-copy"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-copy}{}}}
\subsection{Method \code{copy()}}{
Copy the mesh.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$copy()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A new \code{cgalMesh} object.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(rgl)
mesh <- cgalMesh$new(cube3d())
tmesh <- mesh$copy()$triangulate()
tmesh$isTriangle() # TRUE
mesh$isTriangle() # FALSE
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-distance"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-distance}{}}}
\subsection{Method \code{distance()}}{
Distance from one or more points to the mesh. The mesh 
  must be triangle.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$distance(points)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{points}}{either one point given as a numeric vector or several 
points given as a numeric matrix with three columns}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A numeric vector providing the distances between the given 
  point(s) to the mesh.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# cube example ####
library(cgalMeshes)
mesh <- cgalMesh$new(rgl::cube3d())$triangulate()
points <- rbind(
  c(0, 0, 0),
  c(1, 1, 1)
)
mesh$distance(points) # should be 1 and 0

# cyclide example ####
library(cgalMeshes)
a <- 100; c <- 30; mu <- 80
mesh <- cgalMesh$new(cyclideMesh(a, c, mu, nu = 100L, nv = 100L))
O2 <- c(c, 0, 0)
# should be a - mu = 20 (see ?cyclideMesh):
mesh$distance(O2)    
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-edges"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-edges}{}}}
\subsection{Method \code{edges()}}{
Get the edges of the mesh.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$edges()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A dataframe with four columns; the first two ones give the 
  vertex indices of each edge (one edge per row), the third one gives 
  the lengths of each edge, and the fourth one gives the dihedral angles 
  in degrees between the two faces adjacent to each edge
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(rgl)
mesh <- cgalMesh$new(dodecahedron3d())
head(mesh$edges())
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-fair"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-fair}{}}}
\subsection{Method \code{fair()}}{
Fair a region of the mesh, i.e. make it smooth. The mesh 
  must be triangle. This modifies the reference mesh.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$fair(indices)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{indices}}{the indices of the vertices in the region to be faired}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The modified \code{cgalMesh} object.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(cgalMeshes)
rglHopf <- HopfTorusMesh(nu = 100, nv = 100)
hopf <- cgalMesh$new(rglHopf)
# squared norms of the vertices
normsq <- apply(hopf$vertices(), 1L, crossprod)
# fair the region where the squared norm is > 19
indices <- which(normsq > 19)
hopf$fair(indices)
rglHopf_faired <- hopf$getMesh()
# plot
library(rgl)
open3d(windowRect = 50 + c(0, 0, 900, 450))
mfrow3d(1L, 2L)
view3d(0, 0, zoom = 0.8)
shade3d(rglHopf, color = "orangered")
next3d()
view3d(0, 0, zoom = 0.8)
shade3d(rglHopf_faired, color = "orangered")
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-geoDists"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-geoDists}{}}}
\subsection{Method \code{geoDists()}}{
Estimated geodesic distances between vertices. The mesh 
  must be triangle.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$geoDists(index)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index}}{index of the source vertex}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The estimated geodesic distances from the source vertex to each
  vertex.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# torus ####
library(cgalMeshes)
library(rgl)
rglmesh <- torusMesh(R = 3, r = 2, nu = 90, nv = 60)
mesh <- cgalMesh$new(rglmesh)
# estimated geodesic distances
geodists <- mesh$geoDists(1L)
# normalization to (0, 1)
geodists <- geodists / max(geodists)
# color each vertex according to its geodesic distance from the source
fcolor <- colorRamp(viridisLite::turbo(200L))
colors <- fcolor(geodists)
colors <- rgb(colors[, 1L], colors[, 2L], colors[, 3L], maxColorValue = 255)
rglmesh[["material"]] <- list("color" = colors)
# plot
open3d(windowRect = 50 + c(0, 0, 512, 512), zoom = 0.8)
shade3d(rglmesh)
wire3d(rglmesh, color = "black")
if(!rgl.useNULL()) {
  play3d(spin3d(axis = c(1, 1, 1), rpm = 5), duration = 20)  
}

# a trefoil knot (taken from `?rgl::cylinder3d`) ####
library(cgalMeshes)
library(rgl)
theta <- seq(0, 2*pi, length.out = 50L)
knot <- cylinder3d(
  center = cbind(
    sin(theta) + 2*sin(2*theta), 
    2*sin(3*theta), 
    cos(theta) - 2*cos(2*theta)),
  e1 = cbind(
    cos(theta) + 4*cos(2*theta), 
    6*cos(3*theta), 
    sin(theta) + 4*sin(2*theta)),
  radius = 0.8, 
  closed = TRUE)
knot <- subdivision3d(knot, depth = 2)
mesh <- cgalMesh$new(knot)$triangulate()
rglmesh <- mesh$getMesh()
# estimated geodesic distances
geodists <- mesh$geoDists(1L)
# normalization to (0, 1)
geodists <- geodists / max(geodists)
# color each vertex according to its geodesic distance from the source
fcolor <- colorRamp(viridisLite::inferno(200L))
colors <- fcolor(geodists)
colors <- rgb(colors[, 1L], colors[, 2L], colors[, 3L], maxColorValue = 255)
rglmesh[["material"]] <- list("color" = colors)
# plot
open3d(windowRect = 50 + c(0, 0, 512, 512), zoom = 0.8)
shade3d(rglmesh)
if(!rgl.useNULL()) {
  play3d(spin3d(axis = c(1, 1, 0), rpm = 5), duration = 20)  
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-getMesh"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-getMesh}{}}}
\subsection{Method \code{getMesh()}}{
Get the mesh.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$getMesh(normals = TRUE, rgl = TRUE, ...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{normals}}{Boolean, whether to return the per-vertex normals}

\item{\code{rgl}}{Boolean, whether to return a \strong{rgl} mesh if possible, 
i.e. if the mesh only has triangular or quadrilateral faces}

\item{\code{...}}{arguments passed to \code{\link[rgl:mesh3d]{mesh3d}} (if 
a \strong{rgl} mesh is returned)}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \strong{rgl} mesh or a list with two or three fields: 
  \code{vertices}, \code{faces}, and \code{normals} if the argument 
  \code{normals} is set to \code{TRUE}
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(rgl)
mesh <- cgalMesh$new(cube3d())$triangulate()
mesh$getMesh(normals = FALSE)
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-intersection"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-intersection}{}}}
\subsection{Method \code{intersection()}}{
Intersection with another mesh.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$intersection(mesh2)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{mesh2}}{a \code{cgalMesh} object}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{cgalMesh} object.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(cgalMeshes)
library(rgl)
# take two cubes
rglmesh1 <- cube3d()
rglmesh2 <- translate3d(cube3d(), 1, 1, 1)
mesh1 <- cgalMesh$new(rglmesh1)
mesh2 <- cgalMesh$new(rglmesh2)
# the two meshes must be triangle
mesh1$triangulate()
mesh2$triangulate()
# intersection
imesh <- mesh1$intersection(mesh2)
rglimesh <- imesh$getMesh(normals = FALSE)
# extract edges for plotting
extEdges <- exteriorEdges(imesh$edges())
# plot
open3d(windowRect = 50 + c(0, 0, 512, 512), zoom = 0.9)
shade3d(rglimesh, color = "red")
plotEdges(imesh$vertices(), extEdges)
shade3d(rglmesh1, color = "yellow", alpha = 0.2)
shade3d(rglmesh2, color = "cyan", alpha = 0.2)
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-isClosed"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-isClosed}{}}}
\subsection{Method \code{isClosed()}}{
Check whether the mesh is closed.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$isClosed()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A Boolean value, whether the mesh is closed.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-isOutwardOriented"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-isOutwardOriented}{}}}
\subsection{Method \code{isOutwardOriented()}}{
Check whether the mesh is outward oriented. The mesh must 
  be triangle.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$isOutwardOriented()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A Boolean value, whether the mesh is outward oriented.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(rgl)
mesh <- cgalMesh$new(tetrahedron3d())
mesh$isOutwardOriented() # TRUE
mesh$reverseOrientation()
mesh$isOutwardOriented() # FALSE
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-isTriangle"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-isTriangle}{}}}
\subsection{Method \code{isTriangle()}}{
Check whether the mesh is triangle.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$isTriangle()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A Boolean value, whether the mesh is triangle.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(rgl)
mesh <- cgalMesh$new(cube3d())
mesh$isTriangle()
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-isValid"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-isValid}{}}}
\subsection{Method \code{isValid()}}{
Check whether the mesh is valid.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$isValid()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A Boolean value, whether the mesh is valid.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-orientToBoundVolume"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-orientToBoundVolume}{}}}
\subsection{Method \code{orientToBoundVolume()}}{
Reorient the connected components of the mesh in order that 
it bounds a volume. The mesh must be triangle.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$orientToBoundVolume()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
The modified \code{cgalMesh} object, invisibly. \strong{WARNING}: 
  even if you store the result in a new variable, the original mesh is 
  modified.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# two disjoint tetrahedra ####
vertices <- rbind(
  c(0, 0, 0),
  c(2, 2, 0),
  c(2, 0, 2),
  c(0, 2, 2),
  c(3, 3, 3),
  c(5, 5, 3),
  c(5, 3, 5),
  c(3, 5, 5)
)
faces <- rbind(
  c(3, 2, 1),
  c(3, 4, 2),
  c(1, 2, 4),
  c(4, 3, 1),
  c(5, 6, 7),
  c(6, 8, 7),
  c(8, 6, 5),
  c(5, 7, 8)
)
mesh <- cgalMesh$new(vertices = vertices, faces = faces)
mesh$boundsVolume() # FALSE
mesh$orientToBoundVolume()
mesh$boundsVolume() # TRUE
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-removeSelfIntersections"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-removeSelfIntersections}{}}}
\subsection{Method \code{removeSelfIntersections()}}{
Remove self-intersections (experimental). The mesh must 
  be triangle.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$removeSelfIntersections()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
The modified \code{cgalMesh} object, invisibly.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-reverseOrientation"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-reverseOrientation}{}}}
\subsection{Method \code{reverseOrientation()}}{
Reverse the orientation of the faces of the mesh.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$reverseOrientation()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
The modified \code{cgalMesh} object, invisibly. \strong{WARNING}: 
  even if you store the result in a new variable, the original mesh is 
  modified.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(rgl)
mesh <- cgalMesh$new(tetrahedron3d())
mesh$isOutwardOriented() # TRUE
mesh$reverseOrientation()
mesh$isOutwardOriented() # FALSE
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-selfIntersects"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-selfIntersects}{}}}
\subsection{Method \code{selfIntersects()}}{
Check whether the mesh self-intersects. The mesh must be 
  triangle.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$selfIntersects()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A Boolean value, whether the mesh self-intersects.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(rgl)
mesh <- cgalMesh$new(dodecahedron3d())
mesh$selfIntersects()
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-subtract"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-subtract}{}}}
\subsection{Method \code{subtract()}}{
Subtract another mesh. Both meshes must be triangle.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$subtract(mesh2)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{mesh2}}{a \code{cgalMesh} object}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{cgalMesh} object.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(cgalMeshes)
library(rgl)
# take two cubes
rglmesh1 <- cube3d()
rglmesh2 <- translate3d(cube3d(), 1, 1, 1)
mesh1 <- cgalMesh$new(rglmesh1)
mesh2 <- cgalMesh$new(rglmesh2)
# the two meshes must be triangle
mesh1$triangulate()
mesh2$triangulate()
# difference
mesh <- mesh1$subtract(mesh2)
rglmesh <- mesh$getMesh(normals = FALSE)
# extract edges for plotting
extEdges <- exteriorEdges(mesh$edges())
# plot
open3d(windowRect = 50 + c(0, 0, 512, 512), zoom = 0.9)
shade3d(rglmesh, color = "red")
plotEdges(mesh$vertices(), extEdges)
shade3d(rglmesh2, color = "cyan", alpha = 0.2)
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-triangulate"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-triangulate}{}}}
\subsection{Method \code{triangulate()}}{
Triangulate mesh.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$triangulate()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
The modified \code{cgalMesh} object, invisibly. \strong{WARNING}: 
  even if you store the result in a new variable, the original mesh is 
  modified (see the example). You may want to triangulate a copy of the 
  mesh; see the \code{copy} method.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(rgl)
mesh <- cgalMesh$new(cube3d())
mesh$isTriangle() # FALSE
# warning: triangulating the mesh modifies it
mesh$triangulate()
mesh$isTriangle() # TRUE
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-union"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-union}{}}}
\subsection{Method \code{union()}}{
Union with another mesh. Both meshes must be triangle.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$union(mesh2)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{mesh2}}{a \code{cgalMesh} object}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{cgalMesh} object.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(cgalMeshes)
library(rgl)
# take two cubes
rglmesh1 <- cube3d()
rglmesh2 <- translate3d(cube3d(), 1, 1, 1)
mesh1 <- cgalMesh$new(rglmesh1)
mesh2 <- cgalMesh$new(rglmesh2)
# the two meshes must be triangle
mesh1$triangulate()
mesh2$triangulate()
# union
umesh <- mesh1$union(mesh2)
rglumesh <- umesh$getMesh(normals = FALSE)
# extract edges for plotting
extEdges <- exteriorEdges(umesh$edges())
# plot
open3d(windowRect = 50 + c(0, 0, 512, 512), zoom = 0.9)
shade3d(rglumesh, color = "red")
plotEdges(umesh$vertices(), extEdges)
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-vertices"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-vertices}{}}}
\subsection{Method \code{vertices()}}{
Get the vertices of the mesh.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$vertices()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
The vertices in a matrix.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-volume"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-volume}{}}}
\subsection{Method \code{volume()}}{
Compute the volume of the mesh. The mesh must be closed,
  triangle, and must not self-intersect.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$volume()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A number, the mesh volume.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(rgl)
mesh <- cgalMesh$new(cube3d())$triangulate()
mesh$volume()
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-writeMeshFile"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-writeMeshFile}{}}}
\subsection{Method \code{writeMeshFile()}}{
Write mesh to a file.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$writeMeshFile(filename, precision = 17, binary = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{filename}}{path to the file to be written, with extension 
\code{off} or \code{ply}}

\item{\code{precision}}{a positive integer, the desired number of decimal 
places}

\item{\code{binary}}{Boolean, whether to write a binary PLY file if 
\code{filename} has the \code{ply} extension}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Nothing, just writes a file.
}
}
}
