% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/densities.R
\name{increment}
\alias{increment}
\alias{increment.cauphylm}
\alias{increment.cauphyfit}
\title{Posterior density of an increment}
\usage{
increment(x, ...)

\method{increment}{cauphylm}(x, node, values, n_values = 100, n_cores = 1, ...)

\method{increment}{cauphyfit}(x, node, values, n_values = 100, n_cores = 1, ...)
}
\arguments{
\item{x}{an object of class \code{\link{fitCauchy}} or \code{\link{cauphylm}}.}

\item{...}{other arguments to be passed to the method.}

\item{node}{vector of nodes ending the branches for which to compute the posterior density of the increment. If not specified, the reconstruction is done on all the possible edges.}

\item{values}{the vector of values where the density should be computed. If not specified, the reconstruction is done for a grid of \code{n_values} values between \code{-1.5 * maxdiff} and \code{1.5 * maxdiff}, where \code{maxdiff} is the difference between the larger and smaller tip value.}

\item{n_values}{the number of point for the grid of values. Default to \code{100}. Ignored if \code{values} is provided.}

\item{n_cores}{number of cores for the parallelization. Default to 1.}
}
\value{
an object of S3 class \code{ancestralCauchy},
 which is a matrix of posterior increment values, with nodes in rows and values in columns.
}
\description{
Compute the posterior density of a branch increment under a fitted Cauchy process on a phylogenetic tree.
}
\details{
This function assumes a Cauchy Process on the tree with fitted parameters 
(see \code{\link{fitCauchy}}),
and computes the posterior ancestral density of trait increments at branches
(ie, the difference between the traits value at the end and beginning of the branch),
conditionally on the vector of tip values.

It computes the posterior density on all the points in \code{values},
that should be refined enough to get a good idea of the density curve.
}
\section{Methods (by class)}{
\itemize{
\item \code{increment(cauphylm)}: \code{\link{cauphylm}} object

\item \code{increment(cauphyfit)}: \code{\link{fitCauchy}} object

}}
\examples{
set.seed(1289)
# Simulate tree and data
phy <- ape::rphylo(10, 0.1, 0)
dat <- rTraitCauchy(n = 1, phy = phy, model = "cauchy",
                    parameters = list(root.value = 10, disp = 0.1))
# Fit the data
fit <- fitCauchy(phy, dat, model = "cauchy", method = "reml")
# Reconstruct the ancestral increments
inc <- increment(fit)
plot_asr(fit, inc = inc, offset = 3)
plot(inc, node = c(3, 8), type = "l")
# Refine grid for edges ending at tips 3 and 8
inc2 <- increment(fit, node = c(3, 8), values = seq(-3, 3, 0.01))
plot(inc2, type = "l")
# Find HDI
library(HDInterval)
hdi_inc <- hdi(inc2)
hdi_inc
plot(inc2, interval = hdi_inc, type = "l")

}
\references{
Bastide, P. and Didier, G. 2023. The Cauchy Process on Phylogenies: a Tractable Model for Pulsed Evolution. Systematic Biology. doi:10.1093/sysbio/syad053.
}
\seealso{
\code{\link{fitCauchy}}, \code{\link{cauphylm}},
\code{\link{plot.ancestralCauchy}}, \code{\link{plot_asr}}, \code{\link{ancestral}},
\code{\link{hdi.ancestralCauchy}}
}
