% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bvar.R, R/predict.bvar.R
\name{bvar}
\alias{bvar}
\alias{predict.bvar}
\title{Bayesian Vector Autoregression Objects}
\usage{
bvar(data = NULL, exogen = NULL, y = NULL, x = NULL, A0 = NULL,
  A = NULL, B = NULL, C = NULL, Sigma = NULL)

\method{predict}{bvar}(object, ..., n.ahead = 10, new_x = NULL,
  new_D = NULL, ci = 0.95)
}
\arguments{
\item{data}{the original time-series object of endogenous variables.}

\item{exogen}{the original time-series object of unmodelled variables.}

\item{y}{a \eqn{K \times T} matrix of endogenous variables,
usually, a result of a call to \code{\link{gen_var}}.}

\item{x}{a \eqn{(pK + (1+s)M + N) \times T} matrix of regressor variables, usually, a result of a
call to \code{\link{gen_var}}.}

\item{A0}{a \eqn{K^2 \times S} matrix of MCMC coefficient draws of structural parameters.}

\item{A}{a \eqn{pK^2 \times S} matrix of MCMC coefficient draws of lagged endogenous variables.}

\item{B}{a \eqn{((1 + s)MK) \times S} matrix of MCMC coefficient draws of unmodelled, non-deterministic variables.}

\item{C}{an \eqn{KN \times S} matrix of MCMC coefficient draws of deterministic terms.}

\item{Sigma}{a \eqn{K^2 \times S} matrix of variance-covariance MCMC draws.}

\item{object}{an object of class \code{"bvar"}, usually, a result of a call to
\code{\link{bvar}} or \code{\link{bvec_to_bvar}}.}

\item{...}{additional arguments.}

\item{n.ahead}{number of steps ahead at which to predict.}

\item{new_x}{a matrix of new non-deterministic, exogenous variables. Must have \code{n.ahead} rows.}

\item{new_D}{a matrix of new deterministic variables. Must have \code{n.ahead} rows.}

\item{ci}{a numeric between 0 and 1 specifying the probability mass covered by the
credible intervals. Defaults to 0.95.}
}
\value{
An object of class "bvar" containing the following components, if specified:
\item{data}{the original time-series object of endogenous variables.}
\item{exogen}{the original time-series object of unmodelled variables.}
\item{y}{a \eqn{K \times T} matrix of endogenous variables.}
\item{x}{a \eqn{(pK + (1+s)M + N) \times T} matrix of regressor variables.}
\item{A0}{an \eqn{S \times K^2} "mcmc" object of coefficient draws of structural parameters.}
\item{A}{an \eqn{S \times pK^2} "mcmc" object of coefficient draws of lagged endogenous variables.}
\item{B}{an \eqn{S \times ((1 + s)MK)} "mcmc" object of coefficient draws of unmodelled, non-deterministic variables.}
\item{C}{an \eqn{S \times NK} "mcmc" object of coefficient draws of deterministic terms.}
\item{Sigma}{an \eqn{S \times K^2} "mcmc" object of variance-covariance draws.}
\item{specifications}{a list containing information on the model specification.}

A time-series object of class \code{"bvarprd"}.
}
\description{
\code{bvar} is used to create objects of class "bvar".

Forecasting a Bayesian VAR object of class \code{"bvar"} with credible bands.
}
\details{
For the VARX model
\deqn{A_0 y_t = \sum_{i = 1}^{p} A_i y_{t-i} + \sum_{i = 0}^{s} B_i x_{t - i} + C d_t + u_t}
the function collects the S draws of a Gibbs sampler (after the burn-in phase) in a standardised object,
where \eqn{y_t} is a K-dimensional vector of endogenous variables,
\eqn{A_0} is a \eqn{K \times K} matrix of structural coefficients.
\eqn{A_i} is a \eqn{K \times K} coefficient matrix of lagged endogenous variabels.
\eqn{x_t} is an M-dimensional vector of unmodelled, non-deterministic variables
and \eqn{B_i} its corresponding coefficient matrix.
\eqn{d_t} is an N-dimensional vector of deterministic terms
and \eqn{C} its corresponding coefficient matrix.
\eqn{u_t} is an error term with \eqn{u_t \sim N(0, \Sigma_u)}.

The draws of the different coefficient matrices provided in \code{A0}, \code{A},
\code{B}, \code{C} and \code{Sigma} have to correspond to the same MCMC iteration.

For the VAR model
\deqn{y_t = \sum_{i = 1}^{p} A_{i} y_{t-i} + \sum_{i = 0}^{s} B_{i} x_{t-i} + C D_t + A_0^{-1} u_t,}
with \eqn{u_t \sim N(0, \Sigma)} the function produces \code{n.ahead} forecasts.
}
\examples{
data("e1")
e1 <- diff(log(e1))

data <- gen_var(e1, p = 2, deterministic = "const")

y <- data$Y[, 1:73]
x <- data$Z[, 1:73]

set.seed(1234567)

iter <- 500 # Number of iterations of the Gibbs sampler
# Chosen number of iterations should be much higher, e.g. 30000.

burnin <- 100 # Number of burn-in draws
store <- iter - burnin

t <- ncol(y) # Number of observations
k <- nrow(y) # Number of endogenous variables
m <- k * nrow(x) # Number of estimated coefficients

# Set (uninformative) priors
a_mu_prior <- matrix(0, m) # Vector of prior parameter means
a_v_i_prior <- diag(0, m) # Inverse of the prior covariance matrix

u_sigma_df_prior <- 0 # Prior degrees of freedom
u_sigma_scale_prior <- diag(0, k) # Prior covariance matrix
u_sigma_df_post <- t + u_sigma_df_prior # Posterior degrees of freedom

# Initial values
u_sigma_i <- diag(.00001, k)
u_sigma <- solve(u_sigma_i)

# Data containers for posterior draws
draws_a <- matrix(NA, m, store)
draws_sigma <- matrix(NA, k^2, store)

# Start Gibbs sampler
for (draw in 1:iter) {
  # Draw conditional mean parameters
  a <- post_normal(y, x, u_sigma_i, a_mu_prior, a_v_i_prior)

# Draw variance-covariance matrix
u <- y - matrix(a, k) \%*\% x # Obtain residuals
u_sigma_scale_post <- solve(u_sigma_scale_prior + tcrossprod(u))
u_sigma_i <- matrix(rWishart(1, u_sigma_df_post, u_sigma_scale_post)[,, 1], k)
u_sigma <- solve(u_sigma_i) # Invert Sigma_i to obtain Sigma

# Store draws
if (draw > burnin) {
  draws_a[, draw - burnin] <- a
  draws_sigma[, draw - burnin] <- u_sigma
  }
}

# Generate bvar object
bvar_est <- bvar(y = y, x = x, A = draws_a[1:18,],
                 C = draws_a[19:21, ], Sigma = draws_sigma)
data("e1")
e1 <- diff(log(e1))
data <- gen_var(e1, p = 2, deterministic = "const")

y <- data$Y[, 1:73]
x <- data$Z[, 1:73]

set.seed(1234567)

iter <- 500 # Number of iterations of the Gibbs sampler
# Chosen number of iterations should be much higher, e.g. 30000.

burnin <- 100 # Number of burn-in draws
store <- iter - burnin

t <- ncol(y) # Number of observations
k <- nrow(y) # Number of endogenous variables
m <- k * nrow(x) # Number of estimated coefficients

# Set (uninformative) priors
a_mu_prior <- matrix(0, m) # Vector of prior parameter means
a_v_i_prior <- diag(0, m) # Inverse of the prior covariance matrix

u_sigma_df_prior <- 0 # Prior degrees of freedom
u_sigma_scale_prior <- diag(0, k) # Prior covariance matrix
u_sigma_df_post <- t + u_sigma_df_prior # Posterior degrees of freedom

# Initial values
u_sigma_i <- diag(.00001, k)
u_sigma <- solve(u_sigma_i)

# Data containers for posterior draws
draws_a <- matrix(NA, m, store)
draws_sigma <- matrix(NA, k^2, store)

# Start Gibbs sampler
for (draw in 1:iter) {
  # Draw conditional mean parameters
  a <- post_normal(y, x, u_sigma_i, a_mu_prior, a_v_i_prior)

# Draw variance-covariance matrix
u <- y - matrix(a, k) \%*\% x # Obtain residuals
u_sigma_scale_post <- solve(u_sigma_scale_prior + tcrossprod(u))
u_sigma_i <- matrix(rWishart(1, u_sigma_df_post, u_sigma_scale_post)[,, 1], k)
u_sigma <- solve(u_sigma_i) # Invert Sigma_i to obtain Sigma

# Store draws
if (draw > burnin) {
  draws_a[, draw - burnin] <- a
  draws_sigma[, draw - burnin] <- u_sigma
  }
}

# Generate bvar object
bvar_est <- bvar(y = y, x = x, A = draws_a[1:18,],
                 C = draws_a[19:21, ], Sigma = draws_sigma)

# Generate forecasts
bvar_pred <- predict(bvar_est, n.ahead = 10, new_D = rep(1, 10))

# Plot forecasts
plot(bvar_pred)

}
\references{
Lütkepohl, H. (2007). \emph{New introduction to multiple time series analysis} (2nd ed.). Berlin: Springer.
}
