% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/stats-lm.R
\name{augment.lm}
\alias{augment.lm}
\title{Augment data with information from a(n) lm object}
\usage{
\method{augment}{lm}(
  x,
  data = model.frame(x),
  newdata = NULL,
  se_fit = FALSE,
  interval = c("none", "confidence", "prediction"),
  conf.level = 0.95,
  ...
)
}
\arguments{
\item{x}{An \code{lm} object created by \code{\link[stats:lm]{stats::lm()}}.}

\item{data}{A \link[base:data.frame]{base::data.frame} or \code{\link[tibble:tibble]{tibble::tibble()}} containing the original
data that was used to produce the object \code{x}. Defaults to
\code{stats::model.frame(x)} so that \code{augment(my_fit)} returns the augmented
original data. \strong{Do not} pass new data to the \code{data} argument.
Augment will report information such as influence and cooks distance for
data passed to the \code{data} argument. These measures are only defined for
the original training data.}

\item{newdata}{A \code{\link[base:data.frame]{base::data.frame()}} or \code{\link[tibble:tibble]{tibble::tibble()}} containing all
the original predictors used to create \code{x}. Defaults to \code{NULL}, indicating
that nothing has been passed to \code{newdata}. If \code{newdata} is specified,
the \code{data} argument will be ignored.}

\item{se_fit}{Logical indicating whether or not a \code{.se.fit} column should be
added to the augmented output. For some models, this calculation can be
somewhat time-consuming. Defaults to \code{FALSE}.}

\item{interval}{Character indicating the type of confidence interval columns
to be added to the augmented output. Passed on to \code{predict()} and defaults
to "none".}

\item{conf.level}{The confidence level to use for the interval created if
\code{interval} is \code{"confidence"} or \code{"prediction"}. Must be strictly greater
than 0 and less than 1. Defaults to 0.95, which corresponds to a 95
percent confidence/prediction interval.}

\item{...}{Additional arguments. Not used. Needed to match generic
signature only. \strong{Cautionary note:} Misspelled arguments will be
absorbed in \code{...}, where they will be ignored. If the misspelled
argument has a default value, the default value will be used.
For example, if you pass \code{conf.lvel = 0.9}, all computation will
proceed using \code{conf.level = 0.95}. Two exceptions here are:
\itemize{
\item \code{tidy()} methods will warn when supplied an \code{exponentiate} argument if
it will be ignored.
\item \code{augment()} methods will warn when supplied a \code{newdata} argument if it
will be ignored.
}}
}
\description{
Augment accepts a model object and a dataset and adds
information about each observation in the dataset. Most commonly, this
includes predicted values in the \code{.fitted} column, residuals in the
\code{.resid} column, and standard errors for the fitted values in a \code{.se.fit}
column. New columns always begin with a \code{.} prefix to avoid overwriting
columns in the original dataset.

Users may pass data to augment via either the \code{data} argument or the
\code{newdata} argument. If the user passes data to the \code{data} argument,
it \strong{must} be exactly the data that was used to fit the model
object. Pass datasets to \code{newdata} to augment data that was not used
during model fitting. This still requires that at least all predictor
variable columns used to fit the model are present. If the original outcome
variable used to fit the model is not included in \code{newdata}, then no
\code{.resid} column will be included in the output.

Augment will often behave differently depending on whether \code{data} or
\code{newdata} is given. This is because there is often information
associated with training observations (such as influences or related)
measures that is not meaningfully defined for new observations.

For convenience, many augment methods provide default \code{data} arguments,
so that \code{augment(fit)} will return the augmented training data. In these
cases, augment tries to reconstruct the original data based on the model
object with varying degrees of success.

The augmented dataset is always returned as a \link[tibble:tibble]{tibble::tibble} with the
\strong{same number of rows} as the passed dataset. This means that the passed
data must be coercible to a tibble. If a predictor enters the model as part
of a matrix of covariates, such as when the model formula uses
\code{\link[splines:ns]{splines::ns()}}, \code{\link[stats:poly]{stats::poly()}}, or \code{\link[survival:Surv]{survival::Surv()}}, it is represented
as a matrix column.

We are in the process of defining behaviors for models fit with various
\code{na.action} arguments, but make no guarantees about behavior when data is
missing at this time.
}
\details{
When the modeling was performed with \code{na.action = "na.omit"}
(as is the typical default), rows with NA in the initial data are omitted
entirely from the augmented data frame. When the modeling was performed
with \code{na.action = "na.exclude"}, one should provide the original data
as a second argument, at which point the augmented data will contain those
rows (typically with NAs in place of the new columns). If the original data
is not provided to \code{\link[=augment]{augment()}} and \code{na.action = "na.exclude"}, a
warning is raised and the incomplete rows are dropped.

Some unusual \code{lm} objects, such as \code{rlm} from MASS, may omit
\code{.cooksd} and \code{.std.resid}. \code{gam} from mgcv omits \code{.sigma}.

When \code{newdata} is supplied, only returns \code{.fitted}, \code{.resid} and
\code{.se.fit} columns.
}
\examples{
\dontshow{if (rlang::is_installed("ggplot2")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}

library(ggplot2)
library(dplyr)

mod <- lm(mpg ~ wt + qsec, data = mtcars)

tidy(mod)
glance(mod)

# coefficient plot
d <- tidy(mod, conf.int = TRUE)

ggplot(d, aes(estimate, term, xmin = conf.low, xmax = conf.high, height = 0)) +
  geom_point() +
  geom_vline(xintercept = 0, lty = 4) +
  geom_errorbarh()

# aside: There are tidy() and glance() methods for lm.summary objects too.
# this can be useful when you want to conserve memory by converting large lm
# objects into their leaner summary.lm equivalents.
s <- summary(mod)
tidy(s, conf.int = TRUE)
glance(s)

augment(mod)
augment(mod, mtcars, interval = "confidence")

# predict on new data
newdata <- mtcars \%>\%
  head(6) \%>\%
  mutate(wt = wt + 1)
augment(mod, newdata = newdata)

# ggplot2 example where we also construct 95\% prediction interval

# simpler bivariate model since we're plotting in 2D
mod2 <- lm(mpg ~ wt, data = mtcars)

au <- augment(mod2, newdata = newdata, interval = "prediction")

ggplot(au, aes(wt, mpg)) +
  geom_point() +
  geom_line(aes(y = .fitted)) +
  geom_ribbon(aes(ymin = .lower, ymax = .upper), col = NA, alpha = 0.3)

# predict on new data without outcome variable. Output does not include .resid
newdata <- newdata \%>\%
  select(-mpg)

augment(mod, newdata = newdata)

au <- augment(mod, data = mtcars)

ggplot(au, aes(.hat, .std.resid)) +
  geom_vline(size = 2, colour = "white", xintercept = 0) +
  geom_hline(size = 2, colour = "white", yintercept = 0) +
  geom_point() +
  geom_smooth(se = FALSE)

plot(mod, which = 6)

ggplot(au, aes(.hat, .cooksd)) +
  geom_vline(xintercept = 0, colour = NA) +
  geom_abline(slope = seq(0, 3, by = 0.5), colour = "white") +
  geom_smooth(se = FALSE) +
  geom_point()

# column-wise models
a <- matrix(rnorm(20), nrow = 10)
b <- a + rnorm(length(a))
result <- lm(b ~ a)

tidy(result)
\dontshow{\}) # examplesIf}
}
\seealso{
\link[stats:na.action]{stats::na.action}

\code{\link[=augment]{augment()}}, \code{\link[stats:predict.lm]{stats::predict.lm()}}

Other lm tidiers: 
\code{\link{augment.glm}()},
\code{\link{glance.glm}()},
\code{\link{glance.lm}()},
\code{\link{glance.summary.lm}()},
\code{\link{glance.svyglm}()},
\code{\link{tidy.glm}()},
\code{\link{tidy.lm}()},
\code{\link{tidy.lm.beta}()},
\code{\link{tidy.mlm}()},
\code{\link{tidy.summary.lm}()}
}
\concept{lm tidiers}
\value{
A \code{\link[tibble:tibble]{tibble::tibble()}} with columns:
  \item{.cooksd}{Cooks distance.}
  \item{.fitted}{Fitted or predicted value.}
  \item{.hat}{Diagonal of the hat matrix.}
  \item{.lower}{Lower bound on interval for fitted values.}
  \item{.resid}{The difference between observed and fitted values.}
  \item{.se.fit}{Standard errors of fitted values.}
  \item{.sigma}{Estimated residual standard deviation when corresponding observation is dropped from model.}
  \item{.std.resid}{Standardised residuals.}
  \item{.upper}{Upper bound on interval for fitted values.}

}
