\name{CE.Normal.MeanVar}
\alias{CE.Normal.MeanVar}
\title{
Multiple break-point detection via the CE method for continuous data (both mean and variance changes)
}
\description{
This function performs calculations to estimate both the number of break-points and their corresponding locations of continuous measurements with the CE method. The normal distribution is used to model the observed continous data. This function supports for the simulation of break-point locations based on the four parameter beta distribution or truncated normal distribution. User can select either from the genral BIC or AIC to obtain the optimal number of break-points. 
}
\usage{
CE.Normal.MeanVar(data, Nmax = 10, eps = 0.01, rho = 0.05, M = 200, h = 5, a = 0.8,
b = 0.8, distyp = 1, penalty = "BIC", parallel = FALSE)
}
\arguments{
  \item{data}{
data to be analysed. A single column array or a dataframe.
}
  \item{Nmax}{
maximum number of break-points. Default value is 10. 
}
  \item{eps}{
the cut-off value for the stopping criterion in the CE method. Default value is 0.01.
}
  \item{rho}{
the fraction which is used to obtain the best performing set of sample solutions (i.e., elite sample). Default value is 0.05.
}
  \item{M}{
sample size to be used in simulating the locations of break-points. Default value is 200.
}
  \item{h}{
minimum aberration width. Default is 5.
}
  \item{a}{
a smoothing parameter value. It is used in the four parameter beta distribution to smooth both shape parameters. When simulating from the truncated normal distribution, this value is used to smooth the estimates of the mean values. Default is 0.8.
}
  \item{b}{
a smoothing parameter value. It is used in the truncated normal distribution to smooth the estimates of the standard deviation. Default is 0.8.
}
  \item{distyp}{
distributions to simulate break-point locations. Options: 1 = four parameter beta distribution, 2 = truncated normal distribution. Default is 1. 
}

  \item{penalty}{
User can select from BIC or AIC to obtain the optimal number of break-points. Options: "BIC" and "AIC". Default is "BIC".
}
  \item{parallel}{
A logical argument specifying if parallel computation should be carried-out (TRUE) or not (FALSE). By default it is set as `FALSE'. In WINDOWS OS systems "snow" functionalities are used, whereas in Unix/Linux/MAC OSX "multicore" functionalities are used to carryout parallel computations with the maximum number of cores available.
}
}
\details{
The normal distribution is used to model the continuous data. A performance function score (BIC/AIC) is calculated for each of the solutions generated by the statistical distribution (four parameter beta distribution or truncated normal distribution), which is used to simulate break-points from no break-point to the user provided maximum number of break-points. Changes in both mean and variance are estimated. The solution that maximizes the selection criteria with respect to the number of break-points is reported as the optimal solution. Finally, a list containing a vector of break-point locations, number of break-points,  BIC/AIC values and log-likelihood value is returned in the console.
}
\value{
A list is returned with following items:
\item{No.BPs}{The number of break-points}
\item{BP.Loc}{A vector of break-point locations}
\item{BIC/AIC}{BIC/AIC value}
\item{ll}{Loglikelihood of the optimal solution}
}
\references{
 Priyadarshana, W. J. R. M., Sofronov G. (2015). Multiple Break-Points Detection in Array CGH Data via the Cross-Entropy Method, IEEE/ACM Transactions on Computational Biology and Bioinformatics, 12 (2), pp.487-498.

  Priyadarshana, W. J. R. M. and Sofronov, G. (2012) A Modified Cross- Entropy Method for Detecting Multiple Change-Points in DNA Count Data, In Proc. of the IEEE Conference on Evolutionary Computation (CEC), 1020-1027, DOI: 10.1109/CEC.2012.6256470.
    
  Rubinstein, R., and Kroese, D. (2004) The Cross-Entropy Method: A Unified Approach to Combinatorial Optimization, Monte-Carlo Simulation and Machine Learning. Springer-Verlag, New York.
  
  Zhang, N.R., and Siegmund, D.O. (2007) A modified Bayes information criterion with applications to the analysis of comparative genomic hybridization data. Biometrics, 63, 22-32.
}
\author{
Priyadarshana, W.J.R.M. <mjayawardana@swin.edu.au>
}

\seealso{
\code{\link{CE.Normal.Init.Mean}} for CE with normal with initial locations,

\code{\link{CE.Normal.Mean}} for CE with normal to detect break-points in mean levels,

\code{\link{CE.Normal.Init.MeanVar}} for CE with normal to detect break-points in both mean and variance with initial locations,

\code{\link{profilePlot}} to obtain mean profile plot.
}
\examples{
\dontrun{

simdata <- as.data.frame(c(rnorm(200,100,5),rnorm(1000,160,8),rnorm(300,120,10)))

## CE with four parameter beta distribution with BIC as the selection criterion  ##
obj1 <- CE.Normal.MeanVar(simdata, distyp = 1, penalty = "BIC", parallel =TRUE)
profilePlot(obj1, simdata)

## CE with truncated normal distribution with BIC as the selection criterion  ##
obj2 <- CE.Normal.MeanVar(simdata, distyp = 2, penalty = "BIC", parallel =TRUE)
profilePlot(obj2, simdata)
}
}
