#pragma once

#include "../Alphabet.h"

struct Protein {
  typedef char CharType;
};

// Based on BLOSUM62
// Collapse AAs into 4 bits
template <>
struct BitMapPolicy< Protein > {
  static const size_t NumBits = 4;

  inline static int8_t BitMap( const char aa ) {
    static const uint8_t BitMapping[] = {
      0b00000, // 'A'
      0b10000, // 'B' ambiguous/invalid
      0b00011, // 'C'
      0b00100, // 'D'
      0b00100, // 'E'
      0b01111, // 'F'
      0b00101, // 'G'
      0b00110, // 'H'
      0b00111, // 'I'
      0b10000, // 'J' ambiguous/invalid
      0b01001, // 'K'
      0b01000, // 'L'
      0b01010, // 'M'
      0b00010, // 'N'
      0b10000, // 'O' ambiguous/invalid
      0b01011, // 'P'
      0b00100, // 'Q'
      0b00001, // 'R'
      0b01100, // 'S'
      0b01101, // 'T'
      0b10000, // 'U' ambiguous/invalid
      0b00111, // 'V'
      0b01110, // 'W'
      0b10010, // 'X' ambiguous/invalid
      0b01111, // 'Y'
      0b10001, // 'Z' ambiguous/invalid
    };

    auto val = BitMapping[ aa - 'A' ];
    if( ( val & 0b10000 ) > 0 )
      return -1; // ambiguity

    return ( val & 0b1111 );
  }
};

// BLOSUM62
template <>
struct ScorePolicy< Protein > {
  inline static int8_t Score( const char aaA, const char aaB ) {
    static const int ScoreMatrixSize = 26; // 'A'...'Z', waste some space for faster lookup
    static const int8_t ScoreMatrix[ ScoreMatrixSize ][ ScoreMatrixSize ] = {
      /* A,   B,   C,   D,   E,   F,   G,   H,   I,   J,   K,   L,   M,   N,   O,   P,   Q,   R,   S,   T,   U,   V,   W,   X,   Y,   Z */
      {  4,  -2,   0,  -2,  -1,  -2,   0,  -2,  -1,   0,  -1,  -1,  -1,  -2,   0,  -1,  -1,  -1,   1,   0,   0,   0,  -3,   0,  -2,  -1}, // A
      { -2,   4,  -3,   4,   1,  -3,  -1,   0,  -3,   0,   0,  -4,  -3,   3,   0,  -2,   0,  -1,   0,  -1,   0,  -3,  -4,  -1,  -3,   1}, // B
      {  0,  -3,   9,  -3,  -4,  -2,  -3,  -3,  -1,   0,  -3,  -1,  -1,  -3,   0,  -3,  -3,  -3,  -1,  -1,   0,  -1,  -2,  -2,  -2,  -3}, // C
      { -2,   4,  -3,   6,   2,  -3,  -1,  -1,  -3,   0,  -1,  -4,  -3,   1,   0,  -1,   0,  -2,   0,  -1,   0,  -3,  -4,  -1,  -3,   1}, // D
      { -1,   1,  -4,   2,   5,  -3,  -2,   0,  -3,   0,   1,  -3,  -2,   0,   0,  -1,   2,   0,   0,  -1,   0,  -2,  -3,  -1,  -2,   4}, // E
      { -2,  -3,  -2,  -3,  -3,   6,  -3,  -1,   0,   0,  -3,   0,   0,  -3,   0,  -4,  -3,  -3,  -2,  -2,   0,  -1,   1,  -1,   3,  -3}, // F
      {  0,  -1,  -3,  -1,  -2,  -3,   6,  -2,  -4,   0,  -2,  -4,  -3,   0,   0,  -2,  -2,  -2,   0,  -2,   0,  -3,  -2,  -1,  -3,  -2}, // G
      { -2,   0,  -3,  -1,   0,  -1,  -2,   8,  -3,   0,  -1,  -3,  -2,   1,   0,  -2,   0,   0,  -1,  -2,   0,  -3,  -2,  -1,   2,   0}, // H
      { -1,  -3,  -1,  -3,  -3,   0,  -4,  -3,   4,   0,  -3,   2,   1,  -3,   0,  -3,  -3,  -3,  -2,  -1,   0,   3,  -3,  -1,  -1,  -3}, // I
      {  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0}, // J
      { -1,   0,  -3,  -1,   1,  -3,  -2,  -1,  -3,   0,   5,  -2,  -1,   0,   0,  -1,   1,   2,   0,  -1,   0,  -2,  -3,  -1,  -2,   1}, // K
      { -1,  -4,  -1,  -4,  -3,   0,  -4,  -3,   2,   0,  -2,   4,   2,  -3,   0,  -3,  -2,  -2,  -2,  -1,   0,   1,  -2,  -1,  -1,  -3}, // L
      { -1,  -3,  -1,  -3,  -2,   0,  -3,  -2,   1,   0,  -1,   2,   5,  -2,   0,  -2,   0,  -1,  -1,  -1,   0,   1,  -1,  -1,  -1,  -1}, // M
      { -2,   3,  -3,   1,   0,  -3,   0,   1,  -3,   0,   0,  -3,  -2,   6,   0,  -2,   0,   0,   1,   0,   0,  -3,  -4,  -1,  -2,   0}, // N
      {  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0}, // O
      { -1,  -2,  -3,  -1,  -1,  -4,  -2,  -2,  -3,   0,  -1,  -3,  -2,  -2,   0,   7,  -1,  -2,  -1,  -1,   0,  -2,  -4,  -2,  -3,  -1}, // P
      { -1,   0,  -3,   0,   2,  -3,  -2,   0,  -3,   0,   1,  -2,   0,   0,   0,  -1,   5,   1,   0,  -1,   0,  -2,  -2,  -1,  -1,   3}, // Q
      { -1,  -1,  -3,  -2,   0,  -3,  -2,   0,  -3,   0,   2,  -2,  -1,   0,   0,  -2,   1,   5,  -1,  -1,   0,  -3,  -3,  -1,  -2,   0}, // R
      {  1,   0,  -1,   0,   0,  -2,   0,  -1,  -2,   0,   0,  -2,  -1,   1,   0,  -1,   0,  -1,   4,   1,   0,  -2,  -3,   0,  -2,   0}, // S
      {  0,  -1,  -1,  -1,  -1,  -2,  -2,  -2,  -1,   0,  -1,  -1,  -1,   0,   0,  -1,  -1,  -1,   1,   5,   0,   0,  -2,   0,  -2,  -1}, // T
      {  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0}, // U
      {  0,  -3,  -1,  -3,  -2,  -1,  -3,  -3,   3,   0,  -2,   1,   1,  -3,   0,  -2,  -2,  -3,  -2,   0,   0,   4,  -3,  -1,  -1,  -2}, // V
      { -3,  -4,  -2,  -4,  -3,   1,  -2,  -2,  -3,   0,  -3,  -2,  -1,  -4,   0,  -4,  -2,  -3,  -3,  -2,   0,  -3,  11,  -2,   2,  -3}, // W
      {  0,  -1,  -2,  -1,  -1,  -1,  -1,  -1,  -1,   0,  -1,  -1,  -1,  -1,   0,  -2,  -1,  -1,   0,   0,   0,  -1,  -2,  -1,  -1,  -1}, // X
      { -2,  -3,  -2,  -3,  -2,   3,  -3,   2,  -1,   0,  -2,  -1,  -1,  -2,   0,  -3,  -1,  -2,  -2,  -2,   0,  -1,   2,  -1,   7,  -2}, // Y
      { -1,   1,  -3,   1,   4,  -3,  -2,   0,  -3,   0,   1,  -3,  -1,   0,   0,  -1,   3,   0,   0,  -1,   0,  -2,  -3,  -1,  -2,   4}, // Z
    };

    return ScoreMatrix[ aaA - 'A' ][ aaB - 'A' ];
  }
};

template <>
struct MatchPolicy< Protein > {
  inline static bool Match( const char aaA, const char aaB ) {
    return ScorePolicy< Protein >::Score( aaA, aaB ) >= 4;
  }
};
