library(bioset)
context("Calculations")

test_that("canonicalise_units works as expected", {
  expect_equal(bioset:::canonicalise_units("   \u00B5   l   "), "\u00B5l")
  expect_equal(bioset:::canonicalise_units(" \u00B5 l / \tf g\t"), "\u00B5l/fg")
  expect_error(bioset:::canonicalise_units("%"))
  expect_error(bioset:::canonicalise_units(""))
  expect_error(bioset:::canonicalise_units(" \t "))
  expect_error(bioset:::canonicalise_units(" \t / \t"))
  expect_error(bioset:::canonicalise_units(" %\t / \t"))
  expect_error(bioset:::canonicalise_units(" %"))
  expect_warning(bioset:::canonicalise_units("% w / v"))
  expect_warning(bioset:::canonicalise_units("%w/v"))
  expect_equal(bioset:::canonicalise_units("v/v"), "l/l")
  expect_equal(bioset:::canonicalise_units("w/w"), "g/g")
  expect_equal(bioset:::canonicalise_units("%v/v"), "cl/l")
  expect_equal(bioset:::canonicalise_units("%w/w"), "cg/g")
  expect_equal(bioset:::canonicalise_units("\u00B5M"), "\u00B5mol/l")
  expect_equal(bioset:::canonicalise_units("M"), "mol/l")
})

test_that("concentration types are detected correctly", {
  expect_error(bioset:::get_conc_type("\u00B5l"))
  # ".g / .l", ".g / .m^3", "% w / v" (= 0.1 g / l)
  expect_equal(bioset:::get_conc_type("\u00B5g/ml"), "mass_vol")
  expect_equal(bioset:::get_conc_type("\u00B5g/dm^3"), "mass_vol")
  # ".M", ".mol / .l",  ".mol / .m^3"
  expect_equal(bioset:::get_conc_type("nmol/\u00B5l"), "molar_vol")
  expect_equal(bioset:::get_conc_type("pmol/cm^3"), "molar_vol")
  # ".l/.l", ".l / m^3", ".m^3/.m^3", ".m^3 / .l", "% v / v", "v / v"
  expect_equal(bioset:::get_conc_type("nl/ml"), "vol_vol")
  expect_equal(bioset:::get_conc_type("nl/mm^3"), "vol_vol")
  expect_equal(bioset:::get_conc_type("nm^3/mm^3"), "vol_vol")
  expect_equal(bioset:::get_conc_type("nm^3/ml"), "vol_vol")
  # ".g / .g", "w / w", "% w / w"
  expect_equal(bioset:::get_conc_type("\u00B5g/kg"), "mass_mass")
})

test_that("prefixes are converted correctly", {
  # same concentration type
  expect_error(calc_factor_prefix("M", "mM"))
  expect_error(calc_factor_prefix("", "mg"))
  expect_error(calc_factor_prefix("mg", ""))
  expect_error(calc_factor_prefix("mg", 0))
  expect_error(calc_factor_prefix(0, "mg"))
  expect_error(calc_factor_prefix(NA, "mg"))
  expect_error(calc_factor_prefix(NULL, "mg"))
  expect_equal(calc_factor_prefix("l", "dl"), 10)
  expect_equal(calc_factor_prefix("l", "cl"), 100)
  expect_equal(calc_factor_prefix("l", "ml"), 1000)
  expect_error(calc_factor_prefix("\u00B5m^3", "m^3"))
  expect_equal(calc_factor_prefix("dm^3", "m^3"), 0.001)
  expect_equal(calc_factor_prefix("cm^3", "m^3"), 0.000001)
  expect_error(calc_factor_prefix("\u00B5g", "ng / l"))
})

test_that("concentration are converted correctly", {
  expect_error(calc_factor_conc("M", ""))
  expect_error(calc_factor_conc("M"))
  expect_error(calc_factor_conc("M", 0))
  expect_error(calc_factor_conc("M", NULL))
  expect_error(calc_factor_conc("M", NA))
  # same concentration type
  expect_equal(calc_factor_conc("M", "mM"), 1000)
  expect_equal(calc_factor_conc("\u00B5g / ml", "ng / l"), 1000000)
  expect_warning(calc_factor_conc("% w / v", "\u00B5g / \u00B5l"))
  expect_equal(calc_factor_conc("g / l", "\u00B5g / \u00B5l"), 1)
  expect_equal(calc_factor_conc("g / dm^3", "\u00B5g / \u00B5l"), 1)
})

test_that("concentration requirements are stated", {
  expect_error(calc_factor_conc("g/l", "M"))
  expect_error(calc_factor_conc("M", "g/l"))
  expect_error(calc_factor_conc("M", "g/g"))
  expect_error(calc_factor_conc("M", "l/l"))
  expect_error(calc_factor_conc("g/l", "g/g"))
  expect_error(calc_factor_conc("l/l", "g/g"))
  expect_error(calc_factor_conc("g/l", "l/l"))
})
