% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cut_tree.R
\name{cut_tree}
\alias{cut_tree}
\title{Cut a hierarchical tree}
\usage{
cut_tree(
  tree,
  n_clust = NULL,
  cut_height = NULL,
  find_h = TRUE,
  h_max = 1,
  h_min = 0,
  dynamic_tree_cut = FALSE,
  dynamic_method = "tree",
  dynamic_minClusterSize = 5,
  dissimilarity = NULL,
  ...
)
}
\arguments{
\item{tree}{a \code{bioregion.hierar.tree} or a \code{hclust} object}

\item{n_clust}{an integer or a vector of integers indicating the number of
clusters to be obtained from the hierarchical tree, or the output from
\code{\link[=partition_metrics]{partition_metrics()}}. Should not be used at the same time as \code{cut_height}}

\item{cut_height}{a numeric vector indicating the height(s) at which the
tree should be cut. Should not be used at the same time as \code{n_clust} or
\code{optim_method}}

\item{find_h}{a boolean indicating if the height of cut should be found for
the requested \code{n_clust}}

\item{h_max}{a numeric indicating the maximum possible tree height for
finding the height of cut when \code{find_h = TRUE}}

\item{h_min}{a numeric indicating the minimum possible height in the tree
for finding the height of cut when \code{find_h = TRUE}}

\item{dynamic_tree_cut}{a boolean indicating if the dynamic tree cut method
should be used, in which case \code{n_clust} & \code{cut_height} are ignored}

\item{dynamic_method}{a character vector indicating the method to be used
to dynamically cut the tree: either \code{"tree"} (clusters searched only
in the tree) or \code{"hybrid"} (clusters searched on both tree and dissimilarity
matrix)}

\item{dynamic_minClusterSize}{an integer indicating the minimum cluster size
to use in the dynamic tree cut method (see
\link[dynamicTreeCut:cutreeDynamic]{dynamicTreeCut::cutreeDynamic()})}

\item{dissimilarity}{only useful if \code{dynamic_method = "hybrid"}.
Provide here the dissimilarity \code{data.frame} used to build the \code{tree}}

\item{...}{further arguments to be passed to
\link[dynamicTreeCut:cutreeDynamic]{dynamicTreeCut::cutreeDynamic()} to
customize the dynamic tree cut method.}
}
\value{
If \code{tree} is an output from \code{\link[=hclu_hierarclust]{hclu_hierarclust()}}, then the same
object is returned with content updated (i.e., \code{args} and \code{clusters}). If
\code{tree} is a \code{hclust} object, then a \code{data.frame} containing the clusters is
returned.
}
\description{
This functions is designed to work on a hierarchical tree and cut it
at user-selected heights. It works on either outputs from
\code{hclu_hierarclust} or \code{hclust} objects. It cuts the tree for the chosen
number(s) of clusters or selected height(s). It also includes a procedure to
automatically return the height of cut for the chosen number(s) of clusters.
}
\details{
The function can cut the tree with two main methods. First, it can cut
the entire tree at the same height (either specified by \code{cut_height} or
automatically defined for the chosen \code{n_clust}). Second, it can use
the dynamic tree cut method \insertCite{Langfelder2008}{bioregion}, in which
case clusters are detected with an adaptive method based on the shape of
branches in the tree (thus cuts happen at multiple heights depending on
cluster positions in the tree).

The dynamic tree cut method has two variants.
\itemize{
\item{The tree-based only variant
(\code{dynamic_method = "tree"}) is a top-down approach which relies only
on the tree and follows the order of clustered objects on it}
\item{The hybrid variant
(\code{dynamic_method = "hybrid"}) is a bottom-up approach which relies on
both the tree and the dissimilarity matrix to build clusters on the basis of
dissimilarity information among sites. This method is useful to detect
outlying members in each cluster.}
}
}
\note{
The argument \code{find_h} is ignored if \code{dynamic_tree_cut = TRUE},
because heights of cut cannot be estimated in this case.
}
\examples{
comat <- matrix(sample(0:1000, size = 500, replace = TRUE, prob = 1/1:1001),
20, 25)
rownames(comat) <- paste0("Site", 1:20)
colnames(comat) <- paste0("Species", 1:25)

simil <- similarity(comat, metric = "all")
dissimilarity <- similarity_to_dissimilarity(simil)

# User-defined number of clusters
tree1 <- hclu_hierarclust(dissimilarity, n_clust = 5)
tree2 <- cut_tree(tree1, cut_height = .05)
tree3 <- cut_tree(tree1, n_clust = c(3, 5, 10))
tree4 <- cut_tree(tree1, cut_height = c(.05, .1, .15, .2, .25))
tree5 <- cut_tree(tree1, n_clust = c(3, 5, 10), find_h = FALSE)

hclust_tree <- tree2$algorithm$final.tree
clusters_2 <- cut_tree(hclust_tree, n_clust = 10)

cluster_dynamic <- cut_tree(tree1, dynamic_tree_cut = TRUE,
                            dissimilarity = dissimilarity)

}
\references{
\insertRef{Langfelder2008}{bioregion}
}
\seealso{
\link{hclu_hierarclust}
}
\author{
Pierre Denelle (\email{pierre.denelle@gmail.com}),
Maxime Lenormand (\email{maxime.lenormand@inrae.fr}) and
Boris Leroy (\email{leroy.boris@gmail.com})
}
