% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/summary.R
\name{summary.ggum}
\alias{summary.ggum}
\alias{summary.list}
\title{Summarize Posterior Draws for GGUM Parameters}
\usage{
\method{summary}{ggum}(object, ...)

\method{summary}{list}(object, ..., combine = TRUE)
}
\arguments{
\item{object}{A numeric matrix of posterior draws as returned by
\code{\link{ggumMCMC}} or \code{\link{ggumMC3}}, or a list of
such matrices.}

\item{...}{Arguments to be passed to or from other methods}

\item{combine}{A logical vector of length one; if \code{TRUE} and
\code{object} is a list of \code{ggum} result objects, the matrices are
combined and a summary of the combined sample is given; if \code{FALSE}
and \code{object} is a list of \code{ggum} result objects, each matrix
will be summarized individually; and if \code{object} is not a list, it
has no effect. The default is \code{TRUE}.}
}
\value{
A list with three elements: estimates (a list of length four;
  a numeric vector giving the means of the theta draws, a numeric vector
  giving the means of the alpha draws, a numeric vector giving the means
  of the delta draws, and a list where the means of the tau draws are
  collated into a tau estimate vector for each item), sds (a list of length
  four giving the posterior standard deviations for the theta, alpha, delta,
  and tau draws), and statistics (a matrix with five columns and one row
  for each parameter giving the 0.025 quantile, the 0.5 quantile, the mean,
  the 0.975 quantile, and the standard deviation of the posterior draws
  for each parameter; please note the quantiles are calculated using the
  type 8 algorithm from Hyndman and Fan 1996, as suggested by Hyndman and
  Fan 1996, rather than the type 7 algorithm that would be the default
  from R's \code{quantile()}).

  If \code{object} is a list and \code{combine} is \code{FALSE},
  a list of such lists will be returned.
}
\description{
Summarize the results of \code{\link{ggumMCMC}} or \code{\link{ggumMC3}}.
}
\details{
This function provides the posterior mean, median, standard deviation,
and 0.025 and 0.975 quantiles for GGUM parameters from posterior samples
drawn using \code{\link{ggumMCMC}} or \code{\link{ggumMC3}}.
Please note that the quantiles are calculated using the type 8 algorithm
from Hyndman and Fan (1996), as suggested by Hyndman and Fan (1996), rather
than the type 7 algorithm that would be the default from R's
\code{quantile()}).
Before calling this function, care should be taken to ensure that
post-processing has been done if necessary to identify the correct
reflective posterior mode, as discussed in the vignette and Duck-Mayr and
Montgomery (2019).

Please see the vignette (via \code{vignette("bggum")}) for a full in-depth
practical guide to Bayesian estimation of GGUM parameters.
}
\examples{
## NOTE: This is a toy example just to demonstrate the function, which uses
## a small dataset and an unreasonably low number of sampling interations.
## For a longer practical guide on Bayesian estimation of GGUM parameters,
## please see the vignette ( via vignette("bggum") ).
## We'll simulate data to use for this example:
set.seed(123)
sim_data <- ggum_simulation(100, 10, 2)
## Now we can generate posterior draws
## (for the purposes of example, we use 100 iterations,
## though in practice you would use much more)
draws <- ggumMC3(data = sim_data$response_matrix, n_temps = 2,
                 sd_tune_iterations = 100, temp_tune_iterations = 100,
                 temp_n_draws = 50,
                 burn_iterations = 100, sample_iterations = 100)
## Then post-process the output
processed_draws <- post_process(sample = draws,
                                constraint = which.min(sim_data$theta),
                                expected_sign = "-")
## And now we can obtain a summary of the posterior
posterior_summary <- summary(processed_draws)
## It contains all the parameter estimates
str(posterior_summary$estimates)
## As well as the posterior standard deviations
str(posterior_summary$sds)
## And a matrix of the mean (estimates), median, standard deviations,
## and 0.025 and 0.975 quantiles
head(posterior_summary$statistics)

}
\references{
Duck-Mayr, JBrandon, and Jacob Montgomery. 2019.
  \dQuote{Ends Against the Middle: Scaling Votes When Ideological Opposites
  Behave the Same for Antithetical Reasons.}
  \url{http://jbduckmayr.com/papers/ggum.pdf}.

Hyndman, R. J. and Fan, Y. 1996. \dQuote{Sample Quantiles in
  Packages.} \emph{American Statistician} 50, 361--365.
}
\seealso{
\code{\link{ggumMCMC}}, \code{\link{ggumMC3}}
}
