#' Estimate Improvement in R-Squared
#' and Generate the Corresponding Sampling Distribution
#' Using the Monte Carlo Method
#'
#' @author Ivan Jacob Agaloos Pesigan
#'
#' @details The vector of improvement in R-squared
#'   (\eqn{\Delta R^{2}})
#'   is derived from each randomly generated vector of parameter estimates.
#'   Confidence intervals are generated by obtaining
#'   percentiles corresponding to \eqn{100(1 - \alpha)\%}
#'   from the generated sampling
#'   distribution of \eqn{\Delta R^{2}},
#'   where \eqn{\alpha} is the significance level.
#'
#' @return Returns an object
#'   of class `betamc` which is a list with the following elements:
#'   \describe{
#'     \item{call}{Function call.}
#'     \item{object}{The function argument `object`.}
#'     \item{thetahatstar}{Sampling distribution of
#'       \eqn{\Delta R^{2}}.}
#'     \item{vcov}{Sampling variance-covariance matrix of
#'       \eqn{\Delta R^{2}}.}
#'     \item{est}{Vector of estimated
#'       \eqn{\Delta R^{2}}.}
#'     \item{fun}{Function used ("DeltaRSqMC").}
#'   }
#'
#' @inheritParams BetaMC
#'
#' @examples
#' # Fit the regression model
#' object <- lm(QUALITY ~ NARTIC + PCTGRT + PCTSUPP, data = nas1982)
#' # Generate the sampling distribution of parameter estimates
#' # (use a large R, for example, R = 20000 for actual research)
#' mc <- MC(object, R = 100)
#' # Generate confidence intervals for improvement in R-squared
#' deltarsq <- DeltaRSqMC(mc)
#' # Methods --------------------------------------------------------
#' print(deltarsq)
#' summary(deltarsq)
#' coef(deltarsq)
#' vcov(deltarsq)
#' confint(deltarsq, level = 0.95)
#' @export
#' @family Beta Monte Carlo Functions
#' @keywords betaMC deltarsq
DeltaRSqMC <- function(object) {
  sr <- SCorMC(object)
  out <- list(
    call = match.call(),
    object = object,
    thetahatstar = lapply(
      X = sr$thetahatstar,
      FUN = function(x) {
        x^2
      }
    ),
    est = (sr$est)^2,
    fun = "DeltaRSqMC"
  )
  class(out) <- c(
    "betamc",
    class(out)
  )
  return(
    out
  )
}
